﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_GRAPHICSRESOURCE_H_
#define NW_LYT_GRAPHICSRESOURCE_H_

#include <nw/lyt/lyt_Types.h>
#include <nw/gfnd/gfnd_ShaderHelper.h>
#include <nw/gfnd/gfnd_DisplayList.h>
#include <nw/gfnd/gfnd_Memory.h>

#include <nw/font/font_WideTextWriter.h>
#include <nw/font/font_RectDrawer.h>
#include <nw/font/font_DispStringBuffer.h>
#include <nw/ut/ut_Rect.h>
#include <nw/ut/ut_Range.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <gl/glew.h>
#endif

namespace nw
{
namespace lyt
{

class Layout;

// キャッシュされた Uniform Location の ID です。

//! @brief 内部用機能のため使用禁止です。
enum Uniform
{
    UNIFORM_uProjection,
    UNIFORM_uModelView,
    UNIFORM_uTexMtx0_xz,
    UNIFORM_uTexMtx1_xz,
    UNIFORM_uTexMtx2_xz,
    UNIFORM_uTexMtx0_yw,
    UNIFORM_uTexMtx1_yw,
    UNIFORM_uTexMtx2_yw,
    UNIFORM_uColor,
    UNIFORM_uTransform,
    UNIFORM_uFrameSpec,
    UNIFORM_uFrameSize,
    UNIFORM_uPaneSize,
    UNIFORM_uVertexColor,
    UNIFORM_uVertexTexCoord0,
    UNIFORM_uVertexTexCoord1,
    UNIFORM_uVertexTexCoord2,
    UNIFORM_uGeneratingTexCoord,
    UNIFORM_uRcpTexSize0,
    UNIFORM_uTextureMode,
    UNIFORM_uInterpolateWidth,
    UNIFORM_uInterpolateOffset,
    UNIFORM_uTexture0,
    UNIFORM_uTexture1,
    UNIFORM_uTexture2,
    UNIFORM_uIndirectMtx0,
    UNIFORM_uIndirectMtx1,
    UNIFORM_MAX
};

//! @brief シェーダの種類を識別するための ID です。
enum ShaderId
{
    DEFAULT_SHADER,
    NULL_TEXTURE_SHADER,
    SINGLE_TEXTURE_SHADER,
    DOUBLE_TEXTURE1_SHADER,
    DOUBLE_TEXTURE2_SHADER,
    DOUBLE_TEXTURE3_SHADER,
    DOUBLE_TEXTURE4_SHADER,
    DOUBLE_TEXTURE5_SHADER,
    DOUBLE_TEXTURE6_SHADER,
    DOUBLE_TEXTURE7_SHADER,
    DOUBLE_TEXTURE8_SHADER,
    DOUBLE_INDIRECT_TEXTURE_SHADER,
    SHADER_ID_QUANTITY,
};

//! @brief シェーダの種類とバリエーションからシェーダのインデックスを取得します。
//!
//! @param[in] id           シェーダの種類です。
//! @param[in] variation    バリエーションです。
//!
//! @return シェーダのインデックスです。
//!
NW_INLINE int GetShaderIndex(ShaderId id, ShaderVariation variation)
{
    return static_cast<int>(id) + (static_cast<int>(variation) * static_cast<int>(SHADER_ID_QUANTITY));
}

#if defined(NW_PLATFORM_CAFE)
//---------------------------------------------------------------------------
//! @brief シェーダスロットを生成するための設定です。
//!
//---------------------------------------------------------------------------
struct ShaderSlotDescription
{
    enum {
        DISPLAY_LIST_CAPACITY_DEFAULT = 512
    };

    ShaderSlotDescription()
    : requiredBufferSize(0)
    , maxAttributeStreamQuantity(1)
    , maxUniformQuantity(1)
    , displayListCapacity(DISPLAY_LIST_CAPACITY_DEFAULT)
    , shaderQuantity(1)
    {
        std::fill_n(shaderEnabled, ut::GetArrayLength(shaderEnabled), false);
        shaderEnabled[DEFAULT_SHADER] = true;
    }

    u32 EstimateBufferSize();

    u32 requiredBufferSize;
    u32 maxAttributeStreamQuantity;
    u32 maxUniformQuantity;
    u32 displayListCapacity;
    u32 shaderQuantity;
    bool shaderEnabled[SHADER_ID_QUANTITY * SHADER_VARIATION_QUANTITY];
};

//! @brief シェーダにアクセスするための構造体です。
struct ShaderConnection;

//---------------------------------------------------------------------------
//! @brief シェーダを管理するためのクラスです。
//!
//---------------------------------------------------------------------------
class ShaderSlot
{
    NW_DISALLOW_COPY_AND_ASSIGN(ShaderSlot);

public:
    ShaderSlot() : m_Buffer(NULL)
    {
        std::fill_n(
            m_Connections,
            ut::GetArrayLength(m_Connections),
            static_cast<ShaderConnection*>(NULL));
    }

    //! @brief 初期化します。
    void Initialize(const ShaderSlotDescription& description, void* workBuffer);

    //! @brief シェーダを関連付けます。
    ShaderConnection* Connect(
        ShaderId id, GX2VertexShader* vertexShader, GX2PixelShader* pixelShader);

    //! @brief ディスプレイリストを作成します。
    //!
    //! このメソッドをシェーダーの初期設定直後に呼んでおくことで
    //! レイアウトの Draw 時にシェーダーのディスプレイリストを作成しなくなります。
    //!
    void MakeDisplayList(ShaderConnection* connection);

    //! @brief 終了処理を行います。
    void* Shutdown();

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 設定内容を取得する。
    const ShaderSlotDescription& GetDescription() const { return m_Description; }

    //! @brief シェーダにアクセスするためのオブジェクトを取得します。
    ShaderConnection* GetConnection(int id);

    //! @brief 頂点シェーダを取得します。
    GX2VertexShader* GetVertexShader(ShaderConnection* connection);

    //! @brief ピクセルシェーダを取得します。
    GX2PixelShader* GetPixelShader(ShaderConnection* connection);

    //! @brief フェッチシェーダを取得します。
    GX2FetchShader* GetFetchShader(ShaderConnection* connection);

    //! @brief アトリビュートストリームの配列を取得します。
    GX2AttribStream* GetAttribStreams(ShaderConnection* connection);

    //! @brief アトリビュートストリームを取得します。
    GX2AttribStream* GetAttribStream(ShaderConnection* connection, u32 index);

    //! @brief ユニフォームIDの配列を取得します。
    gfnd::ShaderUniformId* GetUniformIds(ShaderConnection* connection);

    //! @brief ユニフォームIDの配列を取得します。
    const gfnd::ShaderUniformId* GetUniformIds(const ShaderConnection* connection) const;

    //! @brief ユニフォームIDを取得します。
    gfnd::ShaderUniformId GetUniformId(const ShaderConnection* connection, Uniform uniformId) const;

    //! @brief シェーダをデバイスに設定します。
    void SetToDevice(ShaderConnection* connection);

    //! @brief マルチテクスチャに対応しているかを取得します。
    bool IsMultiTextureEnabled(ShaderConnection* connection);

    //@}

private:
    void* m_Buffer;
    const ShaderSlotDescription m_Description;
    ShaderConnection* m_Connections[SHADER_ID_QUANTITY * SHADER_VARIATION_QUANTITY];
};

class GraphicsResource;

//---------------------------------------------------------------------------
//! @brief シェーダを簡易にGraphicsResourceに設定するためのクラスです。
//!
//---------------------------------------------------------------------------
class ShaderSetupHelper
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! メンバ変数の初期化のみ行っています。
    //!
    ShaderSetupHelper();

    //! @brief 引数で指定したGraphicsResourceにシェーダを設定します。
    //!
    //! @details
    //! 内部でlyt::Initializeで与えたアロケータが使用されますのでご注意ください。
    //!
    //! @param[in,out] graphicsResource 設定する GraphicsResource クラスです。
    //! @param[in] shaderBinary シェーダバイナリです。
    //!
    //! @sa GraphicsResource
    //!
    void Setup(GraphicsResource* graphicsResource, const ut::MemoryRange& shaderBinary);

    //! @brief Setupで設定したシェーダを解放します。
    //!
    //! @details
    //! このメソッドを呼び出すとシェーダが解放されますので、このメソッドを呼び出すのはSetupで指定したGraphicsResourceをFinalizeした
    //! 後にしてください。
    //!
    void Finalize();

protected:
    //! @brief データを解析して構築した頂点シェーダー情報を返します。
    //!
    //! @param[out] ppShader 構築したシェーダを返します。
    //! @param[in] index シェーダのインデックス番号です。
    //! @param[in] data 解析するデータのアドレスです。
    //! @return 構築に成功したら true を返します。
    //!
    //! @sa FreeVertexShader
    //!
    bool ReadVertexShader(GX2VertexShader** ppShader, u32 index, const void *data);

    //! @brief データを解析して構築したピクセルシェーダー情報を返します。
    //!
    //! @param[out] ppShader 構築したシェーダを返します。
    //! @param[in] index シェーダのインデックス番号です。
    //! @param[in] data 解析するデータのアドレスです。
    //! @return 構築に成功したら true を返します。
    //!
    //! @sa FreePixelShader
    //!
    bool ReadPixelShader(GX2PixelShader** ppShader, u32 index, const void *data);

    //! @brief 頂点シェーダを解放します。
    //!
    //! @param[in] shader 解放するシェーダです。
    //!
    //! @sa ReadVertexShader
    //!
    void FreeVertexShader(GX2VertexShader* shader);

    //! @brief ピクセルシェーダを解放します。
    //!
    //! @param[in] shader 解放するシェーダです。
    //!
    //! @sa ReadPixelShader
    //!
    void FreePixelShader(GX2PixelShader* shader);

    struct ShaderPtr {
        GX2VertexShader* vertexShader;
        GX2PixelShader* pixelShader;
        void* fetchShaderBuffer;

        ShaderPtr() : vertexShader(NULL), pixelShader(NULL), fetchShaderBuffer(NULL) {}
    };

    ShaderPtr m_ShaderPtrs[SHADER_ID_QUANTITY * SHADER_VARIATION_QUANTITY];
};
#endif // defined(NW_PLATFORM_CAFE)

//---------------------------------------------------------------------------
//! @brief 複数の %Layout で共通に使用される %OpenGL のリソースを保持するクラスです。
//!
//! @details
//! DrawInfo::SetGraphicsResource() で DrawInfo に設定します。
//!
//---------------------------------------------------------------------------
class GraphicsResource
{
    NW_DISALLOW_COPY_AND_ASSIGN(GraphicsResource);

public:

    // バッファオブジェクトのインデックスです。

    //! @brief 内部用機能のため使用禁止です。
    enum VBO
    {
        VBO_ELEMENT,
        VBO_VERTEX_INDEX,
        VBO_MAX
    };

    //! @brief 頂点に関する定数です。
    enum
    {
        VBO_INDEX_COUNT = 4, //!< 頂点バッファオブジェクトのインデックス数です。
        VERTEX_ATTRIBUTE_NUMBER = 1 //!< 頂点属性の数です。
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    GraphicsResource();

    //! @brief デストラクタです。
    //!
    ~GraphicsResource();

    //@}

    //----------------------------------------
    //! @name 初期化／開放
    //!
    //@{

    //! @brief セットアップを行います。
    //!
    //! @param[in] charMax          レイアウト内で描画可能な文字の最大数
#if defined(NW_PLATFORM_CAFE)
    //! @param[in] fontShaderBinary シェーダバイナリ
    void Setup(u32 charMax, const ut::MemoryRange& fontShaderBinary);
#else
    void Setup(u32 charMax);
#endif

    //! @brief リソースを開放します。
    //!
    //! @details
    //! デストラクタの呼び出し以前にリソースを開放しなければならない場合に
    //! 使用します。
    //!
    void Finalize();

    //! @brief 初期化が済んでいるか調べます。
    //!
    //! @return 初期化が済んでいれば true を返します。
    //!
    bool Initialized() const
    {
        return m_Initialized;
    }

    //@}

    // TextBox ペインの描画に使用する font::RectDrawer を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return font::RectDrawer への参照を返します。
    //!
    font::RectDrawer& GetFontDrawer()
    {
        return m_FontDrawer;
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // バッファオブジェクトを取得します。

    //! @brief 内部用機能のため使用禁止です。
    GLuint GetVBO(int index) const
    {
        return m_GlVertexBufferObject[index];
    }

#elif defined(NW_PLATFORM_CAFE)
    //! @brief シェーダに必要なレジスタの一覧を取得します。
    static gfnd::shader_helper::UniformParamRange GetUniformList(ShaderId shaderId);

    //! @brief シェーダに必要なレジスタの一覧を取得します。
    static gfnd::shader_helper::UniformParamRange GetUniformList(const char* name, bool withoutVertexColor);

    //! @brief シェーダースロットを取得します。
    ShaderSlot& GetShaderSlot() { return m_ShaderSlot; }

    //! @brief シェーダースロットを取得します。
    const ShaderSlot& GetShaderSlot() const { return m_ShaderSlot; }

    // 共通で利用するバッファオブジェクトを描画します。

    //! @brief 内部用機能のため使用禁止です。
    void DrawVBO();
#endif

private:
    friend class DrawInfo;

    // 共通で利用するバッファオブジェクトを初期化します。

    //! @brief 内部用機能のため使用禁止です。
    void InitVBO();

    // 共通で利用するバッファオブジェクトを有効化します。

    //! @brief 内部用機能のため使用禁止です。
    void ActiveVBO();

    void* m_pRectShaderBinary;
    u32 m_RectShaderBinarySize;
    font::RectDrawer m_FontDrawer;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    gfnd::ShaderUniformId m_UniformIds[UNIFORM_MAX];
    GLuint m_GlProgram;
    GLuint m_GlVertexBufferObject[VBO_MAX];
    GLuint m_VertexShader;
    GLuint m_FragmentShader;

    static const GLushort s_VertexIndex[VBO_INDEX_COUNT];
    static const GLshort s_UniformVertexIndex[][VERTEXATTRSIZE_INDEX];
#elif defined(NW_PLATFORM_CAFE)
    f32* m_Vertices;
    u16* m_Indices;
    ShaderSlot m_ShaderSlot;
#endif

    bool m_Initialized;
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_GRAPHICSRESOURCE_H_

