﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_GFND_GRAPHICS_H_
#define NW_GFND_GRAPHICS_H_

#include <nw/ut/os/ut_Mutex.h>
#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_LockObject.h>

#if defined( NW_PLATFORM_CAFE )
#include <sdk_ver.h>
#endif


#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    #include <GL/glew.h>
    #define NW_GFND_ENUM( enum_name, gl, cafe ) enum_name = gl
#elif defined(NW_PLATFORM_CAFE)
    #include <cafe/gx2.h>
    #include <nw/ut/ut_BitFlag.h>
    #define NW_GFND_ENUM( enum_name, gl, cafe ) enum_name = cafe
#endif

namespace nw
{
namespace gfnd
{

//---------------------------------------------------------------------------
//! @brief グラフィックス全般に関わるエンジン依存処理をラップするクラスです。
//---------------------------------------------------------------------------
class Graphics
{
public:
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。

    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg
    {
        void* hglrc;    //!< GL ハンドルです。
        void* hdc;      //!< デバイスコンテキストのハンドルです。
    };
#endif

    //! @brief SetCullingMode に渡す定数です。
    enum CullingMode
    {
        NW_GFND_ENUM( CULLING_MODE_FRONT, GL_FRONT,   0x0 ),   //!< 表面をカリング。
        NW_GFND_ENUM( CULLING_MODE_BACK,  GL_BACK,    0x1 ),   //!< 裏面をカリング。
        NW_GFND_ENUM( CULLING_MODE_NONE,  0xfffffff0, 0x2 ),   //!< カリングなし(全て表示)。
        NW_GFND_ENUM( CULLING_MODE_ALL,   0xfffffff1, 0x3 ),   //!< 全てカリング(全て非表示)。
        CULLING_MODE_INVALID = 0xffffffff                      //!< 未定義値(エラー)。
    };

    //! @brief 深度テスト用の比較関数を表す定数です。
    enum DepthFunc
    {
        NW_GFND_ENUM( DEPTH_FUNC_LESS_EQUAL,    GL_LEQUAL,   GX2_COMPARE_LEQUAL   ),    //!< デプスバッファの値以下の場合に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_LESS,          GL_LESS,     GX2_COMPARE_LESS     ),    //!< デプスバッファの値より小さい場合に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_GREATER_EQUAL, GL_GEQUAL,   GX2_COMPARE_GEQUAL   ),    //!< デプスバッファの値以上の場合に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_GREATER,       GL_GREATER,  GX2_COMPARE_GREATER  ),    //!< デプスバッファの値より大きい場合に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_ALWAYS,        GL_ALWAYS,   GX2_COMPARE_ALWAYS   ),    //!< 常に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_NEVER,         GL_NEVER,    GX2_COMPARE_NEVER    ),    //!< 常に失敗します。
        NW_GFND_ENUM( DEPTH_FUNC_EQUAL,         GL_EQUAL,    GX2_COMPARE_EQUAL    ),    //!< デプスバッファの値と等しい場合に成功します。
        NW_GFND_ENUM( DEPTH_FUNC_NOT_EQUAL,     GL_NOTEQUAL, GX2_COMPARE_NOTEQUAL ),    //!< デプスバッファの値と等しくない場合に成功します。
        DEPTH_FUNC_INVALID = 0xffffffff                                                 //!< 未定義値(エラー)。
    };

    //! @brief ブレンド式のオペレータに掛ける係数を表す定数です。それぞれ前者は RGB に適用した場合の係数で、後者は Alpha に適用した場合の係数になります。
    enum BlendFactor
    {
        NW_GFND_ENUM( BLEND_FACTOR_ZERO,               GL_ZERO,                     GX2_BLEND_ZERO                     ),  //!< 係数に０を使用します。(0, 0, 0) / (0)
        NW_GFND_ENUM( BLEND_FACTOR_ONE,                GL_ONE,                      GX2_BLEND_ONE                      ),  //!< 係数に１を使用します。(1, 1, 1) / (1)
        NW_GFND_ENUM( BLEND_FACTOR_SRC_COLOR,          GL_SRC_COLOR,                GX2_BLEND_SRC_COLOR                ),  //!< 係数にソース側のカラー成分を係数にします。(Rs, Gs, Bs) / (As)
        NW_GFND_ENUM( BLEND_FACTOR_INV_SRC_COLOR,      GL_ONE_MINUS_SRC_COLOR,      GX2_BLEND_ONE_MINUS_SRC_COLOR      ),  //!< 係数に (1 - ソース側カラー成分) を使用します。(1 - Rs, 1 - Gs, 1 - Bs) / (1 - As)
        NW_GFND_ENUM( BLEND_FACTOR_SRC_ALPHA,          GL_SRC_ALPHA,                GX2_BLEND_SRC_ALPHA                ),  //!< 係数にソース側アルファ成分を係数にします。(As, As, As) / (As)
        NW_GFND_ENUM( BLEND_FACTOR_INV_SRC_ALPHA,      GL_ONE_MINUS_SRC_ALPHA,      GX2_BLEND_ONE_MINUS_SRC_ALPHA      ),  //!< 係数に (1 - ソース側アルファ成分) を使用します。(1 - As, 1 - As, 1 - As) / (1 - As)
        NW_GFND_ENUM( BLEND_FACTOR_DST_COLOR,          GL_DST_COLOR,                GX2_BLEND_DST_COLOR                ),  //!< 係数にフレームバッファ側カラー成分を使用します。(Rd, Gd, Bd) / (Ad)
        NW_GFND_ENUM( BLEND_FACTOR_INV_DST_COLOR,      GL_ONE_MINUS_DST_COLOR,      GX2_BLEND_ONE_MINUS_DST_COLOR      ),  //!< 係数に (1 - フレームバッファ側カラー成分) を使用します。(1 - Rd, 1 - Gd, 1 - Bd) / (1 - Ad)
        NW_GFND_ENUM( BLEND_FACTOR_DST_ALPHA,          GL_DST_ALPHA,                GX2_BLEND_DST_ALPHA                ),  //!< 係数にフレームバッファ側アルファ成分を使用します。(Ad, Ad, Ad) / (Ad)
        NW_GFND_ENUM( BLEND_FACTOR_INV_DST_ALPHA,      GL_ONE_MINUS_DST_ALPHA,      GX2_BLEND_ONE_MINUS_DST_ALPHA      ),  //!< 係数に (1 - フレームバッファ側アルファ成分) を使用します。(1 - Ad, 1 - Ad, 1 - Ad) / (1 - Ad)
        NW_GFND_ENUM( BLEND_FACTOR_CONSTANT_COLOR,     GL_CONSTANT_COLOR,           GX2_BLEND_CONSTANT_COLOR           ),  //!< 係数に (SetBlendConstantColor で指定した値) のカラー成分を使用します。(Rc, Gc, Bc) / (Ac)
        NW_GFND_ENUM( BLEND_FACTOR_INV_CONSTANT_COLOR, GL_ONE_MINUS_CONSTANT_COLOR, GX2_BLEND_ONE_MINUS_CONSTANT_COLOR ),  //!< 係数に (1 - (SetBlendConstantColor で指定した値) のカラー成分) を使用します。(1 - Rc, 1 - Gc, 1 - Bc) / (1 - Ac)
        NW_GFND_ENUM( BLEND_FACTOR_CONSTANT_ALPHA,     GL_CONSTANT_ALPHA,           GX2_BLEND_CONSTANT_ALPHA           ),  //!< 係数に (SetBlendConstantColor で指定した値) のアルファ成分を使用します。(Ac, Ac, Ac) / (Ac)
        NW_GFND_ENUM( BLEND_FACTOR_INV_CONSTANT_ALPHA, GL_ONE_MINUS_CONSTANT_ALPHA, GX2_BLEND_ONE_MINUS_CONSTANT_ALPHA ),  //!< 係数に (1 - (SetBlendConstantColor で指定した値) のアルファ成分) を使用します。(1 - Ac, 1 - Ac, 1 - Ac) / (1 - Ac)
        NW_GFND_ENUM( BLEND_FACTOR_SRC_ALPHA_SATURATE, GL_SRC_ALPHA_SATURATE,       GX2_BLEND_SRC_ALPHA_SATURATE       ),  //!< 係数にソース側アルファと (1 - フレームバッファ側アルファ) を比較した場合の小さい方を使用します。min(As, 1 - Ad) / 1
        BLEND_FACTOR_INVALID = 0xffffffff                                                                                  //!< 未定義値(エラー)。
    };

    //! @brief ブレンド式の各オペレータを最終的にどう処理するかを表す定数です。
    enum BlendEquation
    {
        NW_GFND_ENUM( BLEND_EQUATION_ADD,         GL_FUNC_ADD,              GX2_BLEND_COMBINE_ADD           ),  //!< src * srcFactor + dst * dstFactor
        NW_GFND_ENUM( BLEND_EQUATION_SUB,         GL_FUNC_SUBTRACT,         GX2_BLEND_COMBINE_SRC_MINUS_DST ),  //!< src * srcFactor - dst * dstFactor
        NW_GFND_ENUM( BLEND_EQUATION_REVERSE_SUB, GL_FUNC_REVERSE_SUBTRACT, GX2_BLEND_COMBINE_DST_MINUS_SRC ),  //!< dst * dstFactor - src * srcFactor
        NW_GFND_ENUM( BLEND_EQUATION_MIN,         GL_MIN,                   GX2_BLEND_COMBINE_MIN           ),  //!< min( src * srcFactor, dst * dstFactor )
        NW_GFND_ENUM( BLEND_EQUATION_MAX,         GL_MAX,                   GX2_BLEND_COMBINE_MAX           ),  //!< max( src * srcFactor, dst * dstFactor )
        BLEND_EQUATION_INVALID = 0xffffffff                                                                     //!< 未定義値(エラー)。
    };

    //! @brief アルファテスト用の比較関数を表す定数です。
    enum AlphaFunc
    {
        NW_GFND_ENUM( ALPHA_FUNC_LESS_EQUAL,    GL_LEQUAL,   GX2_COMPARE_LEQUAL   ),   //!< アルファテスト参照値以下の場合に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_LESS,          GL_LESS,     GX2_COMPARE_LESS     ),   //!< アルファテスト参照値より小さい場合に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_GREATER_EQUAL, GL_GEQUAL,   GX2_COMPARE_GEQUAL   ),   //!< アルファテスト参照値以上の場合に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_GREATER,       GL_GREATER,  GX2_COMPARE_GREATER  ),   //!< アルファテスト参照値より大きい場合に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_ALWAYS,        GL_ALWAYS,   GX2_COMPARE_ALWAYS   ),   //!< 常に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_NEVER,         GL_NEVER,    GX2_COMPARE_NEVER    ),   //!< 常に失敗します。
        NW_GFND_ENUM( ALPHA_FUNC_EQUAL,         GL_EQUAL,    GX2_COMPARE_EQUAL    ),   //!< アルファテスト参照値と等しい場合に成功します。
        NW_GFND_ENUM( ALPHA_FUNC_NOT_EQUAL,     GL_NOTEQUAL, GX2_COMPARE_NOTEQUAL ),   //!< アルファテスト参照値と等しくない場合に成功します。
        ALPHA_FUNC_INVALID = 0xffffffff                                                //!< 未定義値(エラー)。
    };

    //! @brief ステンシルテスト用の比較関数を表す定数です。
    enum StencilFunc
    {
        NW_GFND_ENUM( STENCIL_FUNC_LESS_EQUAL,    GL_LEQUAL,   GX2_COMPARE_LEQUAL   ),  //!< マスク値をかけたステンシルテスト参照値以下の場合に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_LESS,          GL_LESS,     GX2_COMPARE_LESS     ),  //!< マスク値をかけたステンシルテスト参照値より小さい場合に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_GREATER_EQUAL, GL_GEQUAL,   GX2_COMPARE_GEQUAL   ),  //!< マスク値をかけたステンシルテスト参照値以上の場合に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_GREATER,       GL_GREATER,  GX2_COMPARE_GREATER  ),  //!< マスク値をかけたステンシルテスト参照値より大きい場合に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_ALWAYS,        GL_ALWAYS,   GX2_COMPARE_ALWAYS   ),  //!< 常に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_NEVER,         GL_NEVER,    GX2_COMPARE_NEVER    ),  //!< 常に失敗します。
        NW_GFND_ENUM( STENCIL_FUNC_EQUAL,         GL_EQUAL,    GX2_COMPARE_EQUAL    ),  //!< マスク値をかけたステンシルテスト参照値と等しい場合に成功します。
        NW_GFND_ENUM( STENCIL_FUNC_NOT_EQUAL,     GL_NOTEQUAL, GX2_COMPARE_NOTEQUAL ),  //!< マスク値をかけたステンシルテスト参照値と等しくない場合に成功します。
        STENCIL_FUNC_INVALID = 0xffffffff                                               //!< 未定義値(エラー)。
    };

    //! @brief ステンシルテストの値の操作を表す定数です。
    enum StencilOp
    {
        NW_GFND_ENUM( STENCIL_OP_KEEP,           GL_KEEP,      GX2_STENCIL_KEEP      ), //!< 現在の値のまま書き換えない。
        NW_GFND_ENUM( STENCIL_OP_ZERO,           GL_ZERO,      GX2_STENCIL_ZERO      ), //!< 0にする。
        NW_GFND_ENUM( STENCIL_OP_REPLACE,        GL_REPLACE,   GX2_STENCIL_REPLACE   ), //!< refに置き換える。
        NW_GFND_ENUM( STENCIL_OP_INCRIMENT,      GL_INCR,      GX2_STENCIL_INCR      ), //!< 1加算する。
        NW_GFND_ENUM( STENCIL_OP_DECRIMENT,      GL_DECR,      GX2_STENCIL_DECR      ), //!< 1減算する。
        NW_GFND_ENUM( STENCIL_OP_INVERT,         GL_INVERT,    GX2_STENCIL_INVERT    ), //!< ビット反転する。
        NW_GFND_ENUM( STENCIL_OP_INCRIMENT_WRAP, GL_INCR_WRAP, GX2_STENCIL_INCR_WRAP ), //!< 1加算する(オーバーフロー時に0に戻る)。
        NW_GFND_ENUM( STENCIL_OP_DECRIMENT_WRAP, GL_DECR_WRAP, GX2_STENCIL_DECR_WRAP ), //!< 1減算する(アンダーフロー時に最大値に戻る)。
        STENCIL_OP_INVALID = 0xffffffff                                                 //!< 未定義値(エラー)。
    };

    //! @brief ポリゴンモードを表す定数です。
    enum PolygonMode
    {
        NW_GFND_ENUM( POLYGON_MODE_POINT, GL_POINT, GX2_POLYGON_MODE_POINT    ),        //!< ポイント。
        NW_GFND_ENUM( POLYGON_MODE_LINE,  GL_LINE,  GX2_POLYGON_MODE_LINE     ),        //!< ライン。
        NW_GFND_ENUM( POLYGON_MODE_FILL,  GL_FILL,  GX2_POLYGON_MODE_TRIANGLE ),        //!< フィル。デフォルトの動作。
        POLYGON_MODE_INVALID = 0xffffffff                                               //!< 未定義値(エラー)。
    };

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    arg      生成パラメータです。
    //---------------------------------------------------------------------------
    explicit Graphics( const CreateArg& arg )
      : m_LockObj( &s_DefaultLockObj )
    {
        s_DefaultLockObj.SetHandle( arg.hglrc, arg.hdc );
    }
#else
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    explicit Graphics()
      : m_LockObj( &s_DefaultLockObj ) {}
#endif

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Graphics() {}


    //---------------------------------------------------------------------------
    //! @brief        唯一のインスタンスを取得します。
    //!
    //!   あらかじめ SetInstance 関数で実装クラスが明示されている必要があります。
    //!
    //! @return       インスタンスを返します。
    //---------------------------------------------------------------------------
    static Graphics* GetInstance() { return s_Instance; }

    //---------------------------------------------------------------------------
    //! @brief        GetInstance した場合に取得できるインスタンスを指定します。
    //!
    //! @param[in]    impl     プラットフォームごとの詳細な実装クラスを指定します。
    //---------------------------------------------------------------------------
    static void SetInstance( Graphics* impl ) { s_Instance = impl; }

    //---------------------------------------------------------------------------
    //! @brief        グラフィックスエンジンの初期化を行います。
    //!
    //! @param[out]   lockObj 描画コンテキストのロック時に使用するロックオブジェクトです。
    //! @details      この関数内で、デフォルトの設定にするためのグラフィックス命令が発行されます。
    //---------------------------------------------------------------------------
    void Initialize(nw::ut::ILockObject* lockObj = NULL);


    //---------------------------------------------------------------------------
    //! @brief        ビューポートを設定します。
    //!
    //! @param[in]    x                 ビューポートの左上原点 X 座標です。
    //! @param[in]    y                 ビューポートの左上原点 Y 座標です。
    //! @param[in]    width             ビューポートの幅です。
    //! @param[in]    height            ビューポートの高さです。
    //! @param[in]    nearZ             ビューポートの near 値です。
    //! @param[in]    farZ              ビューポートの far 値です。
    //! @param[in]    frameBufferWidth  フレームバッファの幅です。
    //! @param[in]    frameBufferHeight フレームバッファの高さです。
    //---------------------------------------------------------------------------
    void SetViewport( f32 x, f32 y, f32 width, f32 height, f32 nearZ, f32 farZ, f32 frameBufferWidth, f32 frameBufferHeight )
    {
        NW_UNUSED_VARIABLE(frameBufferWidth);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。

        // GL ではビューポートの起点が左下であるため、修正する
        glViewport( static_cast<GLint>( x ), static_cast<GLint>( frameBufferHeight - y - height ), static_cast<GLsizei>( width ), static_cast<GLsizei>( height ) );
#if 0
        glDepthRangef(nearZ, farZ);
#else
        glDepthRange(nearZ, farZ);
#endif
        NW_GL_ASSERT();

#else
        GX2SetViewport( x, y, width, height, nearZ, farZ );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        シザーを設定します。
    //!
    //! @param[in]    x                 シザーの左上原点 X 座標です。
    //! @param[in]    y                 シザーの左上原点 Y 座標です。
    //! @param[in]    width             シザーの幅です。
    //! @param[in]    height            シザーの高さです。
    //! @param[in]    frameBufferWidth  フレームバッファの幅です。
    //! @param[in]    frameBufferHeight フレームバッファの高さです。
    //---------------------------------------------------------------------------
    void SetScissor( f32 x, f32 y, f32 width, f32 height, f32 frameBufferWidth, f32 frameBufferHeight )
    {
        NW_UNUSED_VARIABLE(frameBufferWidth);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。

        // GL ではビューポートの起点が左下であるため、修正する
        glScissor( static_cast<GLint>( x ), static_cast<GLint>( frameBufferHeight - y - height ), static_cast<GLsizei>( width ), static_cast<GLsizei>( height ) );
        NW_GL_ASSERT();

#else
        GX2SetScissor( u32(x), u32(y), u32(width), u32(height) );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        GL 座標でビューポートを設定します。
    //!
    //! @param[in]    x                 ビューポートの左下原点 X 座標です。
    //! @param[in]    y                 ビューポートの左下原点 Y 座標です。
    //! @param[in]    width             ビューポートの幅です。
    //! @param[in]    height            ビューポートの高さです。
    //! @param[in]    frameBufferWidth  フレームバッファの幅です。
    //! @param[in]    frameBufferHeight フレームバッファの高さです。
    //---------------------------------------------------------------------------
    void SetViewportGL( f32 x, f32 y, f32 width, f32 height, f32 frameBufferWidth, f32 frameBufferHeight )
    {
        NW_UNUSED_VARIABLE(frameBufferWidth);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_UNUSED_VARIABLE(frameBufferHeight);

        glViewport( static_cast<GLint>( x ), static_cast<GLint>( y ), static_cast<GLsizei>( width ), static_cast<GLsizei>( height ) );
        NW_GL_ASSERT();

#else
        // Cafeではビューポートの起点が左下ではなく左上であるため、修正する
        GX2SetViewport( x, frameBufferHeight - height - y, width, height, 0, 1 );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        GL 座標でシザーを設定します。
    //!
    //! @param[in]    x                 シザーの左下原点 X 座標です。
    //! @param[in]    y                 シザーの左下原点 Y 座標です。
    //! @param[in]    width             シザーの幅です。
    //! @param[in]    height            シザーの高さです。
    //! @param[in]    frameBufferWidth  フレームバッファの幅です。
    //! @param[in]    frameBufferHeight フレームバッファの高さです。
    //---------------------------------------------------------------------------
    void SetScissorGL( f32 x, f32 y, f32 width, f32 height, f32 frameBufferWidth, f32 frameBufferHeight )
    {
        NW_UNUSED_VARIABLE(frameBufferWidth);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_UNUSED_VARIABLE(frameBufferHeight);

        glScissor( static_cast<GLint>( x ), static_cast<GLint>( y ), static_cast<GLsizei>( width ), static_cast<GLsizei>( height ) );
        NW_GL_ASSERT();

#else
        // Cafeではビューポートの起点が左下ではなく左上であるため、修正する
        GX2SetScissor( static_cast<u32>(x), static_cast<u32>(frameBufferHeight - height - y), static_cast<u32>(width), static_cast<u32>(height) );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        深度テストを有効にするかどうか設定します。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthTestEnable( bool enable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if ( enable )
        {
            glEnable( GL_DEPTH_TEST );
        }
        else
        {
            glDisable( GL_DEPTH_TEST );
        }

#else
        m_Flags.ChangeMask( MASK_DEPTH_TEST_ENABLE, enable );
        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        深度書き込みを有効にするかどうか設定します。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthWriteEnable( bool enable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if ( enable )
        {
            glDepthMask( GL_TRUE );
        }
        else
        {
            glDepthMask( GL_FALSE );
        }

#else
        m_Flags.ChangeMask( MASK_DEPTH_WRITE_ENABLE, enable );
        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        深度テスト／書き込みを有効にするかどうか設定します。
    //!
    //! @param[in]    testEnable  深度テストを有効にする場合、 true を指定します。
    //! @param[in]    writeEnable 深度書き込みを有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthEnable( bool testEnable, bool writeEnable )
    {
        SetDepthTestEnable( testEnable );
        SetDepthWriteEnable( writeEnable );
    }

    //---------------------------------------------------------------------------
    //! @brief        深度テストの比較関数を設定します。
    //!
    //! @param[in]    func     比較関数です。
    //---------------------------------------------------------------------------
    void SetDepthFunc( DepthFunc func )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDepthFunc( func );

#else
        if (func == DEPTH_FUNC_INVALID)
        {
            NW_ERR( "not supported DepthFunc." );
            return;
        }
        m_DepathCompareFunc = static_cast<GX2CompareFunction>(func);
        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        カリングの向きを設定します。
    //!
    //! @param[in]    mode     設定するカリングの向きです。
    //---------------------------------------------------------------------------
    void SetCullingMode( CullingMode mode )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        switch( mode )
        {
        case Graphics::CULLING_MODE_NONE:
            glDisable( GL_CULL_FACE );
            break;
        case Graphics::CULLING_MODE_FRONT:
        case Graphics::CULLING_MODE_BACK:
            glEnable( GL_CULL_FACE );
            glCullFace( mode );
            break;
        case Graphics::CULLING_MODE_ALL:
            glEnable( GL_CULL_FACE );
            glCullFace( GL_FRONT_AND_BACK );
            break;
        default:
            NW_ERR( "undefined culling mode." );
            return;
        }

#else

        switch( mode )
        {
        case Graphics::CULLING_MODE_NONE:
            m_IsCullFront = GX2_DISABLE;
            m_IsCullBack = GX2_DISABLE;
            break;

        case Graphics::CULLING_MODE_FRONT:
            m_IsCullFront = GX2_ENABLE;
            m_IsCullBack = GX2_DISABLE;
            break;

        case Graphics::CULLING_MODE_BACK:
            m_IsCullFront = GX2_DISABLE;
            m_IsCullBack = GX2_ENABLE;
            break;

        case Graphics::CULLING_MODE_ALL:
            m_IsCullFront = GX2_ENABLE;
            m_IsCullBack = GX2_ENABLE;
            break;

        default:
            NW_ERR( "undefined culling mode." );
            return;
        }

        SetRasterizeControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理を有効にするかどうか設定します。
    //!
    //! @param[in]    enable    有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetBlendEnable( bool enable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if ( enable )
        {
            glEnable( GL_BLEND );
        }
        else
        {
            glDisable( GL_BLEND );
        }

#else
        GX2SetColorControl(GX2_LOGIC_OP_COPY, enable ? 0xff : 0x0, GX2_DISABLE, GX2_ENABLE);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理のオペレータに掛ける係数を設定します。
    //!
    //! @param[in]    srcFactor 入力側の色に掛ける係数です。
    //! @param[in]    dstFactor 書き込み先のフレームバッファの色に掛ける係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactor( Graphics::BlendFactor srcFactor, Graphics::BlendFactor dstFactor )
    {
        SetBlendFactorImpl( srcFactor, dstFactor, srcFactor, dstFactor );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理のオペレータに掛ける係数を RGB とアルファで別々に設定します。
    //!
    //! @param[in]    srcFactorRGB 入力側の色の RGB 成分に掛ける係数です。
    //! @param[in]    dstFactorRGB 書き込み先のフレームバッファの色の RGB 成分に掛ける係数です。
    //! @param[in]    srcFactorA   入力側の色のアルファ成分に掛ける係数です。
    //! @param[in]    dstFactorA   書き込み先のフレームバッファの色のアルファ成分に掛ける係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorSeparate(
        Graphics::BlendFactor srcFactorRGB,
        Graphics::BlendFactor dstFactorRGB,
        Graphics::BlendFactor srcFactorA,
        Graphics::BlendFactor dstFactorA
    )
    {
        SetBlendFactorImpl( srcFactorRGB, dstFactorRGB, srcFactorA, dstFactorA );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の式を設定します。
    //!
    //! @param[in]    equation ブレンド処理の式です。
    //---------------------------------------------------------------------------
    void SetBlendEquation( Graphics::BlendEquation equation )
    {
        SetBlendEquationImpl( equation, equation );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の式を RGB とアルファで別々に設定します。
    //!
    //! @param[in]    equationRGB RGB 成分の計算に用いる式です。
    //! @param[in]    equationA   アルファ成分の計算に用いる式です。
    //---------------------------------------------------------------------------
    void SetBlendEquationSeparate( Graphics::BlendEquation equationRGB, Graphics::BlendEquation equationA )
    {
        SetBlendEquationImpl( equationRGB, equationA );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の色定数 (ConstantColor) を設定します。
    //!
    //! @param[in]    color    設定する色定数です。
    //---------------------------------------------------------------------------
    void SetBlendConstantColor( const nw::ut::Color4u8& color )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glBlendColor( color.r, color.g, color.b, color.a );
#else
        nw::ut::FloatColor fcolor = static_cast<nw::ut::FloatColor>( color );
        GX2SetBlendConstantColor( fcolor.r, fcolor.g, fcolor.b, fcolor.a );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        VBlank を待ちます。
    //!
    //! Windows では Present 以外で VBlank 待ちを行えないので、Sleep(0) するだけです。
    //---------------------------------------------------------------------------
    void WaitForVBlank()
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
#else
        GX2WaitForVsync();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクを設定します。
    //!
    //! @param[in]    r        R 成分へのマスクです。
    //! @param[in]    g        G 成分へのマスクです。
    //! @param[in]    b        B 成分へのマスクです。
    //! @param[in]    a        アルファ成分へのマスクです。
    //---------------------------------------------------------------------------
    void SetColorMask( bool r, bool g, bool b, bool a )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glColorMask( r, g, b, a );

#else
        nw::ut::BitFlag32 flag;
        flag.ChangeMask( GX2_CHANNEL_MASK_R, r );
        flag.ChangeMask( GX2_CHANNEL_MASK_G, g );
        flag.ChangeMask( GX2_CHANNEL_MASK_B, b );
        flag.ChangeMask( GX2_CHANNEL_MASK_A, a );

        /// TODO: ひとまずTARGET0のみを設定するようにする（本来はターゲット切り替えとの同期が必要？）
        GX2SetTargetChannelMasks(static_cast<GX2ChannelMask>( *(flag.GetPtr()) ),
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA,
            GX2_CHANNEL_MASK_RGBA);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの有効／無効を切り替えます。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetAlphaTestEnable( bool enable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if ( enable )
        {
            glEnable( GL_ALPHA_TEST );
        }
        else
        {
            glDisable( GL_ALPHA_TEST );
        }

#else
        m_Flags.ChangeMask( MASK_ALPHA_TEST_ENABLE, enable );
        GX2SetAlphaTest(static_cast<GX2Boolean>(enable), m_AlphaCompareFunc, m_AlphaTestRef);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの計算式と参照値を設定します。
    //!
    //! @param[in]    func     アルファテストの計算式を指定します。
    //! @param[in]    ref      アルファテストの参照値を指定します。
    //---------------------------------------------------------------------------
    void SetAlphaTestFunc( Graphics::AlphaFunc func, f32 ref )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glAlphaFunc( func, ref );

#else
        if (func == ALPHA_FUNC_INVALID)
        {
            NW_ERR( "not supported AlphaFunc." );
            return;
        }

        m_AlphaCompareFunc = static_cast<GX2CompareFunction>(func);
        m_AlphaTestRef = ref;

        if (m_Flags.IsMaskOn( MASK_ALPHA_TEST_ENABLE ))
        {
            // 値を反映するのは有効なときのみでよい（無効なときは次に有効になったときに反映されるので）
            GX2SetAlphaTest(GX2_TRUE, m_AlphaCompareFunc, ref);
        }
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの有効／無効を切り替えます。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestEnable( bool enable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if ( enable )
        {
            glEnable( GL_STENCIL_TEST );
        }
        else
        {
            glDisable( GL_STENCIL_TEST );
        }

#else
        m_Flags.ChangeMask( MASK_STENCIL_MASK_ENABLE, enable );
        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの計算式と参照値とマスク値を設定します。
    //!
    //! @param[in]    func     ステンシルテストの計算式を指定します。
    //! @param[in]    ref      ステンシルテストの参照値を指定します。
    //! @param[in]    mask     ステンシルテストのマスク値を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestFunc( Graphics::StencilFunc func, s32 ref, u32 mask )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glStencilFunc( func, static_cast<GLint>( ref ), mask );

#else
        if (func == STENCIL_FUNC_INVALID)
        {
            NW_ERR( "not supported StencilFunc." );
            return;
        }

        m_StencilCompareFunc = static_cast<GX2CompareFunction>(func);

        // 現状では、preとwrite、フロントとバックは同じ値を設定する
        GX2SetStencilMask(
            static_cast<u8>(mask),
            static_cast<u8>(mask),
            static_cast<u8>(ref),
            static_cast<u8>(mask),
            static_cast<u8>(mask),
            static_cast<u8>(ref)
            );

        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの値の操作を設定します。
    //!
    //! @param[in]    fail     ステンシルテストに失敗したときの操作を指定します。
    //! @param[in]    zFail    ステンシルテストに成功し、デプステストに失敗したときの操作を指定します。
    //! @param[in]    zPass    デプステストに成功、またはデプステストが行われていないときの操作を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestOp( Graphics::StencilOp fail, Graphics::StencilOp zFail, Graphics::StencilOp zPass )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glStencilOp( fail, zFail, zPass );

#else
        if (fail == STENCIL_OP_INVALID || zFail == STENCIL_OP_INVALID || zPass == STENCIL_OP_INVALID)
        {
            NW_ERR( "not supported DepthFunc." );
            return;
        }

        m_StencilStencilFailFunction = static_cast<GX2StencilFunction>(fail);
        m_StencilStencilZFailFunction = static_cast<GX2StencilFunction>(zFail);
        m_StencilStencilZPassFunction = static_cast<GX2StencilFunction>(zPass);

        SetDepthStencilControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ポリゴンモードを設定します。
    //!
    //! @param[in]    front    表側のポリゴンモードです。
    //! @param[in]    back     裏側のポリゴンモードです。
    //---------------------------------------------------------------------------
    void SetPolygonMode( Graphics::PolygonMode front, Graphics::PolygonMode back )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glPolygonMode(GL_FRONT, front);
        glPolygonMode(GL_BACK, back);
#else
        m_PolygonModeFront = static_cast<GX2PolygonMode>(front);
        m_PolygonModeBack = static_cast<GX2PolygonMode>(back);
        SetRasterizeControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ポリゴンオフセットの有効／無効を設定します。
    //!
    //! @param[in]    fillFrontEnable 表側のポリゴンオフセットが有効かを指定します。
    //! @param[in]    fillBackEnable  裏側のポリゴンオフセットが有効かを指定します。 （GL では無効です。）
    //! @param[in]    pointLineEnable ポイント・ラインのポリゴンオフセットが有効かを指定します。
    //---------------------------------------------------------------------------
    void SetPolygonOffsetEnable( bool fillFrontEnable, bool fillBackEnable, bool pointLineEnable )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        if (fillFrontEnable)
        {
            glEnable(GL_POLYGON_OFFSET_FILL);
        }
        else
        {
            glDisable(GL_POLYGON_OFFSET_FILL);
        }

        NW_UNUSED_VARIABLE( fillBackEnable );

        if (pointLineEnable)
        {
            glEnable(GL_POLYGON_OFFSET_POINT);
            glEnable(GL_POLYGON_OFFSET_LINE);
        }
        else
        {
            glDisable(GL_POLYGON_OFFSET_POINT);
            glDisable(GL_POLYGON_OFFSET_LINE);
        }
#else
        m_PolygonOffsetFrontEnable = static_cast<GX2Boolean>(fillFrontEnable);
        m_PolygonOffsetBackEnable = static_cast<GX2Boolean>(fillBackEnable);
        m_PolygonOffsetPointLineEnable = static_cast<GX2Boolean>(pointLineEnable);
        SetRasterizeControl();
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        VBlank を待つ回数を設定します。
    //!
    //! @param[in]    interval VBlank を待つ回数です。
    //! @return       設定に成功すれば true が返ります。
    //---------------------------------------------------------------------------
    bool SetVBlankWaitInterval( u32 interval );

    //---------------------------------------------------------------------------
    //! @brief   描画コンテキストを取得し、ロックします。
    //!
    //!   - 描画に関する命令を呼ぶ時には、あらかじめ描画コンテキストを保持しておく必要があります。
    //!   - UnlockDrawContext を対になるように呼ばなければなりません。
    //---------------------------------------------------------------------------
    void LockDrawContext();

    //---------------------------------------------------------------------------
    //! @brief   描画コンテキストを解放します。
    //!
    //!   - LockDrawContextと対になるように呼ばなければなりません。
    //---------------------------------------------------------------------------
    void UnlockDrawContext();

    //---------------------------------------------------------------------------
    //! @brief        描画コンテキストのロック時に使用するロックオブジェクトを指定します。
    //!
    //! @param[in]    lockObj  ロックに使用するロックオブジェクトへのポインタです。
    //---------------------------------------------------------------------------
    void SetLockObj( nw::ut::ILockObject* lockObj ) { m_LockObj = lockObj; }

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief        GX2 のコンテキストを取得します。
    //!
    //! @return       GX2 のコンテキストを返します。
    //---------------------------------------------------------------------------
    GX2ContextState* GetGX2ContextState() { return m_ContextState; }

    //---------------------------------------------------------------------------
    //! @brief        GX2 のコンテキストを設定します。
    //!
    //! @param[in]    contextState 設定する GX2 のコンテキストです。
    //---------------------------------------------------------------------------
    void SetGX2ContextState(GX2ContextState* contextState) { m_ContextState = contextState; }
#endif


private:
    //! @brief ブレンド処理のオペレータに掛ける係数を設定します。
    void SetBlendFactorImpl( BlendFactor srcFactorRGB, BlendFactor dstFactorRGB, BlendFactor srcFactorA, BlendFactor dstFactorA )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glBlendFuncSeparate( srcFactorRGB, dstFactorRGB, srcFactorA, dstFactorA );

#else
        if ( srcFactorRGB == BLEND_FACTOR_INVALID || dstFactorRGB == BLEND_FACTOR_INVALID ||
            srcFactorA == BLEND_FACTOR_INVALID || dstFactorA == BLEND_FACTOR_INVALID )
        {
            NW_ERR( "not supported DepthFunc." );
            return;
        }

        m_BlendFunctionSrcColor = static_cast<GX2BlendFunction>(srcFactorRGB);
        m_BlendFunctionDstColor = static_cast<GX2BlendFunction>(dstFactorRGB);
        m_BlendFunctionSrcAlpha = static_cast<GX2BlendFunction>(srcFactorA);
        m_BlendFunctionDstAlpha = static_cast<GX2BlendFunction>(dstFactorA);

        GX2SetBlendControl(
            GX2_RENDER_TARGET_0,
            m_BlendFunctionSrcColor, m_BlendFunctionDstColor, m_BlendComblineColor,
            GX2_ENABLE,
            m_BlendFunctionSrcAlpha, m_BlendFunctionDstAlpha, m_BlendComblineAlpha
        );
#endif
    }

    //! @brief ブレンド処理の式を設定します。
    void SetBlendEquationImpl( BlendEquation equationRGB, BlendEquation equationA )
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glBlendEquationSeparate( equationRGB, equationA );

#else
        if (equationRGB == BLEND_EQUATION_INVALID || equationA == BLEND_EQUATION_INVALID)
        {
            NW_ERR( "not supported DepthFunc." );
            return;
        }

        m_BlendComblineColor = static_cast<GX2BlendCombine>(equationRGB);
        m_BlendComblineAlpha = static_cast<GX2BlendCombine>(equationA);

        GX2SetBlendControl(
            GX2_RENDER_TARGET_0,
            m_BlendFunctionSrcColor,
            m_BlendFunctionDstColor,
            m_BlendComblineColor,
            GX2_ENABLE,
            m_BlendFunctionSrcAlpha,
            m_BlendFunctionDstAlpha,
            m_BlendComblineAlpha
        );
#endif
    }

#if defined(NW_PLATFORM_CAFE)
    //! @brief 内部に保持された値を元に、デプステスト、デプスライト、ステンシルの設定を行います。
    void SetDepthStencilControl()
    {
        if (m_Flags.IsMaskOn( MASK_STENCIL_MASK_ENABLE ))
        {
            // ステンシルテストが有効な場合は、フル機能の関数を使う必要あり
            // 現状では、フロントとバックは両方有効にし、同じ値を設定する
            GX2SetDepthStencilControl(static_cast<GX2Boolean>( m_Flags.IsMaskOn( MASK_DEPTH_TEST_ENABLE ) ),  // depthTestEnable
                static_cast<GX2Boolean>( m_Flags.IsMaskOn( MASK_DEPTH_WRITE_ENABLE ) ), // depthWriteEnable
                m_DepathCompareFunc,  // depthFunc
                GX2_ENABLE,           // stencilTestEnable
                GX2_ENABLE,           // backStencilEnable

                m_StencilCompareFunc,          // frontStencilFunc
                m_StencilStencilZPassFunction, // frontStencilZPass
                m_StencilStencilZFailFunction, // frontStencilZFail
                m_StencilStencilFailFunction,  // frontStencilFail

                m_StencilCompareFunc,          // backStencilFunc
                m_StencilStencilZPassFunction, // backStencilZPass
                m_StencilStencilZFailFunction, // backStencilZFail
                m_StencilStencilFailFunction   // backStencilFail
                );
        }
        else
        {
            GX2SetDepthOnlyControl(
                static_cast<GX2Boolean>( m_Flags.IsMaskOn( MASK_DEPTH_TEST_ENABLE ) ),
                static_cast<GX2Boolean>( m_Flags.IsMaskOn( MASK_DEPTH_WRITE_ENABLE ) ),
                m_DepathCompareFunc);
        }
    }

    //! @brief 内部に保持された値を元に、ラスタライズ設定を行います。
    void SetRasterizeControl()
    {
        GX2SetPolygonControl(
            GX2_FRONT_FACE_CCW, // frontFaceMode
            m_IsCullFront,      // cullFront
            m_IsCullBack,       // cullBack

            static_cast<GX2Boolean>(m_PolygonModeFront != GX2_POLYGON_MODE_TRIANGLE || m_PolygonModeBack != GX2_POLYGON_MODE_TRIANGLE), // enablePolygonModes
            m_PolygonModeFront, // polygonModeFront
            m_PolygonModeBack,  // polygonModeBack

            m_PolygonOffsetFrontEnable,     // polyOffsetFrontEnable
            m_PolygonOffsetBackEnable,      // polyOffsetBackEnable
            m_PolygonOffsetPointLineEnable  // pointLineOffsetEnable
        );
    }
#endif


#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
#else
    enum
    {
        MASK_DEPTH_TEST_ENABLE = 0x1,
        MASK_DEPTH_WRITE_ENABLE = 0x2,
        MASK_STENCIL_MASK_ENABLE = 0x4,
        MASK_ALPHA_TEST_ENABLE = 0x8
    };

    nw::ut::BitFlag32 m_Flags;
    GX2CompareFunction m_DepathCompareFunc;
    GX2BlendCombine m_BlendComblineColor;
    GX2BlendCombine m_BlendComblineAlpha;
    GX2BlendFunction m_BlendFunctionSrcColor;
    GX2BlendFunction m_BlendFunctionDstColor;
    GX2BlendFunction m_BlendFunctionSrcAlpha;
    GX2BlendFunction m_BlendFunctionDstAlpha;
    GX2CompareFunction m_StencilCompareFunc;
    GX2StencilFunction m_StencilStencilZPassFunction;
    GX2StencilFunction m_StencilStencilZFailFunction;
    GX2StencilFunction m_StencilStencilFailFunction;
    GX2CompareFunction m_AlphaCompareFunc;
    f32 m_AlphaTestRef;
    GX2ContextState* m_ContextState;
    GX2Boolean m_IsCullFront;
    GX2Boolean m_IsCullBack;
    GX2PolygonMode m_PolygonModeFront;
    GX2PolygonMode m_PolygonModeBack;
    GX2Boolean m_PolygonOffsetFrontEnable;
    GX2Boolean m_PolygonOffsetBackEnable;
    GX2Boolean m_PolygonOffsetPointLineEnable;
#endif

    //---------------------------------------------------------------------------
    //! @brief        デフォルトのロックオブジェクトです。
    //---------------------------------------------------------------------------
    class DefaultGraphicsLockObject : public ut::ILockObject
    {
    public:

        //---------------------------------------------------------------------------
        //! @brief        コンストラクタです。
        //---------------------------------------------------------------------------
        /* ctor */ DefaultGraphicsLockObject()
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            // TODO: NintendoSdk 対応後、このコメントを削除してください。
            : m_HGLRC( NULL ),
              m_HDC( NULL )
    #endif
        {
        }

        //---------------------------------------------------------------------------
        //! @brief        デストラクタです。
        //---------------------------------------------------------------------------
        /* dtor */ virtual ~DefaultGraphicsLockObject()
        {
        }

        //---------------------------------------------------------------------------
        //! @brief        初期化処理です。
        //---------------------------------------------------------------------------
        void Initialize()
        {
            m_Mutex.Initialize();
        }

        //---------------------------------------------------------------------------
        //! @brief        ロックをおこないます。
        //---------------------------------------------------------------------------
        virtual void Lock();

        //---------------------------------------------------------------------------
        //! @brief        ロックを解除します。
        //---------------------------------------------------------------------------
        virtual void Unlock();

    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。

        //---------------------------------------------------------------------------
        //! @brief        ハンドルをセットします。
        //!
        //! @param[out]   hglc    GL ハンドルです。
        //! @param[out]   hdc     デバイスコンテキストのハンドルです。
        //---------------------------------------------------------------------------
        void SetHandle( void* hglc, void* hdc )
        {
            m_HGLRC = hglc;
            m_HDC   = hdc;
        }

    private:
        void* m_HGLRC;
        void* m_HDC;
    #endif

    private:
        nw::ut::Mutex m_Mutex; //!< 排他用のミューテックスです。
    };

    static Graphics* s_Instance;
    static DefaultGraphicsLockObject s_DefaultLockObj;
    nw::ut::ILockObject* m_LockObj;
};

} // namespace nw::gfnd
} // namespace nw

#endif // NW_GFND_GRAPHICS_H_
