﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESVISIBILITYANIM_H_
#define NW_G3D_RES_RESVISIBILITYANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_Binding.h>

namespace nw { namespace g3d { namespace res {

//! @brief ビジビリティアニメーションの構造体です。
struct ResVisibilityAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    bit16 flag;
    u16 numUserData;
    s32 numFrame;
    u16 numAnim;
    u16 numCurve;
    u32 bakedSize;

    Offset ofsBindModel;
    Offset ofsBindIndexArray;
    Offset ofsNameArray; // BinString の配列
    Offset ofsCurveArray;
    Offset ofsBaseValueArray; // 1ビット1ビジビリティの bit32 配列
    Offset ofsUserDataDic;
};

//! @brief ビジビリティアニメーションのリソースです。
class ResVisibilityAnim : private ResVisibilityAnimData
{
    NW_G3D_RES_COMMON(ResVisibilityAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'V', 'I', 'S') };

    //! @brief ビジビリティアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP,

        //! @briefprivate
        TYPE_SHIFT          = 8,

        //! @brief ボーンビジビリティを表します。
        BONE_VISIBILITY     = 0x0 << TYPE_SHIFT,

        //! @brief マテリアルビジビリティを表します。
        MAT_VISIBILITY      = 0x1 << TYPE_SHIFT,

        //! @briefprivate
        TYPE_MASK           = 0x1 << TYPE_SHIFT
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 事前に ResModel と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResModel* pModel);

    //! @brief 事前に ResModel と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResModel* target) const;

    //! @brief ビジビリティアニメーションをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief ビジビリティアニメーションをカーブに戻します。
    void* ResetCurve();

    //! @brief ビジビリティアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ビジビリティのタイプを取得します。
    int GetVisibilityType() const { return ref().flag & TYPE_MASK; }

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief アニメーション数を取得します。
    int GetAnimCount() const { return ref().numAnim; }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)
    NW_G3D_RES_FIELD_CLASS_DECL(ResModel, BindModel)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESVISIBILITYANIM_H_
