﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESFILE_H_
#define NW_G3D_RES_RESFILE_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_ResSkeleton.h>
#include <nw/g3d/res/g3d_ResShape.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/res/g3d_ResTexture.h>
#include <nw/g3d/res/g3d_ResSkeletalAnim.h>
#include <nw/g3d/res/g3d_ResTexPatternAnim.h>
#include <nw/g3d/res/g3d_ResShaderParamAnim.h>
#include <nw/g3d/res/g3d_ResVisibilityAnim.h>
#include <nw/g3d/res/g3d_ResShapeAnim.h>
#include <nw/g3d/res/g3d_ResSceneAnim.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

class BindCallback;

//--------------------------------------------------------------------------------------------------

//! @brief ファイルデータの構造体です。
struct ResFileData
{
    BinaryFileHeader fileHeader;
    u32 alignment;
    BinString ofsName; //!< バイナリ変換時に指定されたファイル名（拡張子を除く）
    u32 sizeStringPool; // !< 文字列プールのバイトサイズ
    Offset ofsStringPool;

    Offset ofsModelDic;
    Offset ofsTextureDic;
    Offset ofsSkeletalAnimDic;
    Offset ofsShaderParamAnimDic;
    Offset ofsColorAnimDic;
    Offset ofsTexSrtAnimDic;
    Offset ofsTexPatternAnimDic;
    Offset ofsBoneVisAnimDic;
    Offset ofsMatVisAnimDic;
    Offset ofsShapeAnimDic;
    Offset ofsSceneAnimDic;
    Offset ofsExternalFileDic;

    // 辞書が無い場合の判定を無くすためにここで数を持つ。
    // 辞書が無い場合でも名前引きは安全だが、インデックスアクセスはアサートされている。
    u16 numModel;
    u16 numTexture;
    u16 numSkeletalAnim;
    u16 numShaderParamAnim;
    u16 numColorAnim;
    u16 numTexSrtAnim;
    u16 numTexPatternAnim;
    u16 numBoneVisAnim;
    u16 numMatVisAnim;
    u16 numShapeAnim;
    u16 numSceneAnim;
    u16 numExternalFile;

    BinPtr pUserPtr;
};

//! @brief リソースファイルです。
class ResFile : private ResFileData
{
public:
    //! @briefprivate
    typedef ResFileData DataType;

    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'R', 'E', 'S') };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief ロードしたファイルを ResFile に変換します。
    //!
    //! PC 版では内部でエンディアン変換が行われます。
    //!
    static ResFile* ResCast(void* ptr);

    //! @brief リソースをセットアップします。
    //!
    //! 配下のモデル、テクスチャ、アニメーションの Setup を呼び出します。
    //! インスタンス構築前に呼び出してください。
    //!
    void Setup();

    //! @brief リソースをクリーンアップします。
    //!
    //! 配下のモデル、テクスチャ、アニメーションの Cleanup を呼び出します。
    //! インスタンス破棄後に呼び出してください。
    //!
    void Cleanup();

    //! @brief 初期状態から変化した値を初期化します。
    //!
    //! アプリケーションから明示的に変更された値は初期化しません。
    //! 配下のモデル、テクスチャ、アニメーションの Reset を呼び出します。
    //! インスタンス破棄後に呼び出してください。
    //!
    void Reset();

    //! @brief 外部参照されているリソースの参照解決を行います。
    //!
    //! 参照解決が行われないまま描画処理を行うと不適切なアクセスが発生する可能性があります。
    //! 必ず BindResult を用いて正しくバインドされていることを確認してください。
    //!
    //! @param pFile 参照解決に用いるファイルです。
    //! @param typeMask BindTarget の論理和です。
    //!
    //! @return 参照解決の結果を表します。
    //!
    BindResult Bind(const ResFile* pFile, bit32 typeMask);

    //! @brief 外部参照されているリソースの参照解決を行います。
    //!
    //! 参照解決が行われないまま描画処理を行うと不適切なアクセスが発生する可能性があります。
    //! 必ず BindResult を用いて正しくバインドされていることを確認してください。
    //!
    //! @param pFile 参照解決に用いるファイルです。
    //! @param typeMask BindTarget の論理和です。
    //! @param pCallback 関連付けが失敗した場合に呼ばれるコールバックです。
    //!
    //! @return 参照解決の結果を表します。
    //!
    BindResult Bind(const ResFile* pFile, bit32 typeMask, BindCallback* pCallback);

    //! @brief シーンアニメに関連付ける関数名のテーブルを解決します。
    BindResult Bind(const BindFuncTable& funcTable);

    //! @brief リソースの参照設定を解除します。
    void Release(bit32 typeMask); // typeMask は BindTarget の論理和

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief クラス名を取得します。
    static const char* GetClassName() { return "ResFile"; }

    //! @brief データへの参照を取得します。
    DataType& ref() { NW_G3D_VALIDITY_ASSERT; return *ptr(); }

    //! @brief データへの参照を取得します。
    const DataType& ref() const { NW_G3D_VALIDITY_ASSERT; return *ptr(); }

    //! @brief データのポインタを取得します。
    DataType* ptr() { return this; }

    //! @brief データのポインタを取得します。
    const DataType* ptr() const { return this; }

    //! @brief バイナリファイルが有効かどうかを取得します。
    static bool IsValid(const void* ptr);

    //! @brief リソースのヘッダを取得します。
    const BinaryFileHeader* GetFileHeader() const { return &ref().fileHeader; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)


    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //@}

    //----------------------------------------
    //! @name モデル
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResModel, Model, GetName())

    //@}

    //----------------------------------------
    //! @name テクスチャ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResTexture, Texture, GetName())

    //@}

    //----------------------------------------
    //! @name スケルタルアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResSkeletalAnim, SkeletalAnim, GetName())

    //@}

    //----------------------------------------
    //! @name シェーダパラメータアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShaderParamAnim, ShaderParamAnim, GetName())
    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShaderParamAnim, ColorAnim, GetName())
    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShaderParamAnim, TexSrtAnim, GetName())

    //@}

    //----------------------------------------
    //! @name テクスチャパターンアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResTexPatternAnim, TexPatternAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ビジビリティアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResVisibilityAnim, BoneVisAnim, GetName())
    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResVisibilityAnim, MatVisAnim, GetName())

    //@}

    //----------------------------------------
    //! @name シェイプアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShapeAnim, ShapeAnim, GetName())

    //@}

    //----------------------------------------
    //! @name シーンアニメ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResSceneAnim, SceneAnim, GetName())

    //@}

    //----------------------------------------
    //! @name 外部ファイル
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResExternalFile, ExternalFile, GetName())

    //@}

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ResFile);
};

//--------------------------------------------------------------------------------------------------

//! @brief バインド失敗時に呼ばれるコールバッククラスです。
class BindCallback
{
public:
    //! @brief コンストラクタです。
    BindCallback() : m_pFile(NULL), m_pModel(NULL), m_pMaterial(NULL), m_pTexPatternAnim(NULL) {}

    //! @brief デストラクタです。
    virtual ~BindCallback() {}

    //! @brief ResModel 内の ResTextureRef に対する関連付けが失敗した場合に呼ばれます。
    virtual BindResult ModelTexNotBound(ResTextureRef* textureRef)
    {
        (void)textureRef;
        return BindResult::NotBound(BIND_TEXTURE);
    }

    //! @brief ResTexPatternAnim 内の ResTextureRef に対する関連付けが失敗した場合に呼ばれます。
    virtual BindResult AnimTexNotBound(ResTextureRef* textureRef)
    {
        (void)textureRef;
        return BindResult::NotBound(BIND_TEXTURE);
    }

    //! @brief リソースファイルを取得します。
    ResFile* GetFile() { return m_pFile; }

    //! @brief モデルを取得します。
    ResModel* GetModel() { return m_pModel; }

    //! @brief マテリアルを取得します。
    ResMaterial* GetMaterial() { return m_pMaterial; }

    //! @brief テクスチャパターンアニメーションを取得します。
    ResTexPatternAnim* GetTexPatternAnim() { return m_pTexPatternAnim; }

private:
    ResFile* m_pFile;
    ResModel* m_pModel;
    ResMaterial* m_pMaterial;
    ResTexPatternAnim* m_pTexPatternAnim;

    friend class ResFile;
    friend class ResModel;
};

}}} // namespace nw::g3d

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESFILE_H_
