﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SHAPEOBJ_H_
#define NW_G3D_SHAPEOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/res/g3d_ResShape.h>
#include <nw/g3d/g3d_Sizer.h>

namespace nw { namespace g3d {

namespace math {

class Vec2;
class Mtx34;

} // namespace math

class SkeletonObj;

struct ShpBlock
{
    Mtx34 worldMtx;
    s32 vtxSkinCount;
    u32 reserved[3];
    union
    {
        float userFloat[48];
        s32 userInt[48];
        u32 userUInt[48];
    };
    enum
    {
        SYSTEM_AREA_SIZE = 64,
        USER_AREA_SIZE = 192
    };
};

typedef ShpBlock ShapeBlock;

//! @brief 交差判定用の球です。
struct Sphere
{
    //! @brief 指定した行列で球を変換した結果を設定します。
    void Transform(const Sphere& sphere, const Mtx34& mtx);

    //! @brief 2つの球を囲う球を設定します。。
    void Merge(const Sphere& lhs, const Sphere& rhs);

    Vec3 center;
    float radius;
};

//! @brief 交差判定用の AABB です。
struct AABB
{
    //! @brief 指定した頂点群を囲う AABB を設定します。
    void Set(const Vec3* pPointArray, int count);

    //! @brief 指定した行列でバウンディングボリュームを変換した結果を設定します。
    void Transform(const Bounding& aabb, const Mtx34& mtx);

    //! @brief 2つの AABB を囲う AABB を設定します。。
    void Merge(const AABB& lhs, const AABB& rhs);

    Vec3 min;
    Vec3 max;
};

typedef AABB Aabb;

//! @brief 交差判定用の平面です。
struct Plane
{
    //! @brief 3つの頂点を通る平面を設定します。
    void Set(const Vec3& p0, const Vec3& p1, const Vec3& p2);

    Vec3 normal;
    float dist; // dot( normal, pos ) + dist = 0 となる値です。符号に注意してください。
};

struct ViewVolume
{
    //! @brief ビューボリュームを設定します。
    void SetPerspective(float fovy, float aspect, float zNear, float zFar, const Mtx34& viewToWorld);
    //! @brief ビューボリュームを設定します。
    void SetPerspectiveOffset(float fovy, float aspect, float zNear, float zFar, const Mtx34& viewToWorld, const Vec2& offset);
    //! @brief ビューボリュームを設定します。
    void SetFrustum(float top, float bottom, float left, float right, float zNear, float zFar,
        const Mtx34& viewToWorld);
    //! @brief ビューボリュームを設定します。
    void SetOrtho(float top, float bottom, float left, float right, float zNear, float zFar,
        const Mtx34& viewToWorld);

    //! @brief 交差判定を行います。
    //!
    //! ビューボリュームに対して球が外側の場合に false を、それ以外の場合に true を返します。
    //!
    bool TestIntersection(const Sphere& sphere) const;

    //! @brief 交差判定を行います。
    //!
    //! ビューボリュームに対して球が外側の場合に負の値を、内側の場合に正の値を、
    //! 交差している場合に 0 を返します。
    //!
    int TestIntersectionEx(const Sphere& sphere) const;

    //! @brief 交差判定を行います。
    //!
    //! ビューボリュームに対して AABB が外側の場合に false を、それ以外の場合に true を返します。
    //!
    bool TestIntersection(const AABB& aabb) const;

    //! @brief 交差判定を行います。
    //!
    //! ビューボリュームに対して AABB が外側の場合に負の値を、内側の場合に正の値を、
    //! 交差している場合に 0 を返します。
    //!
    int TestIntersectionEx(const AABB& aabb) const;

    //! @brief AABB による事前判定を有効化します。
    void EnableAABB() { flag = 1; }
    //! @brief EnableAABB() の別名関数です。
    void SetAabbEnabled() { EnableAABB(); }
    //! @brief AABB による事前判定を無効化します。
    void DisableAABB() { flag = 0; }
    //! @brief DisableAABB() の別名関数です。
    void SetAabbDisabled() { EnableAABB(); }
    //! @brief AABB による事前判定が有効かどうかを取得します。
    bool IsAABBEnabled() const { return !!flag; }
    //! @brief IsAABBEnabled() の別名関数です。
    bool IsAabbEnabled() const { return IsAABBEnabled(); }

    AABB aabb; // ボリュームを囲う AABB です。AABB との交差判定で事前判定に使用することができます。
    Plane planes[6]; // ボリュームの外側が法線方向です。
    int numPlanes; // 特定の面をを判定から外す場合などに使用します。
    int flag;
};

class ShapeObj;

//! @briefprivate サブメッシュ LOD のカリングの際、LOD レベルの判定に用いる関数オブジェクトです。
//!
//!
//!
struct ICalcLodLevelFunctor
{
public:
    enum
    {
        INVALID_LOD_LEVEL = -1
    };

    virtual int operator()(const AABB& bounding, const ShapeObj& shape) = 0;
};

typedef ICalcLodLevelFunctor ICalculateLodLevelFunctor;

//! @brief カリングに用いるコンテクストです。
struct CullingContext
{
    CullingContext() : nodeIndex(0), nodeLodLevel(ICalcLodLevelFunctor::INVALID_LOD_LEVEL),
        submeshIndex(0), submeshCount(0), submeshLodLevel(0) {}

    int nodeIndex;
    int nodeLodLevel;
    int submeshIndex;
    int submeshCount;
    int submeshLodLevel;
};

//! @brief サブメッシュの範囲です。
struct SubMeshRange
{
    u16 index; //!< 範囲の先頭サブメッシュのインデックスです。
    u16 count; //!< 範囲のサブメッシュ数です。
    u16 lodLevel; //!< 範囲の LOD レベルです。

    //! @brief 2つのサブメッシュの範囲配列を合成して重複している範囲配列を計算します。
    //!
    //! pLHS と pRHS は index と count が共に 0 の SubMeshRange で終端されている必要があります。
    //! pDst は index と count が共に 0 の SubMeshRange で終端され、終端を含まない
    //! サブメッシュの個数が戻り値として返されます。
    //!合成の際、LOD レベルが異なる場合は、より削られるほう、すなわち大きいほうが採用されます。
    //!
    static int And(SubMeshRange* pDst, const SubMeshRange* pLHS, const SubMeshRange* pRHS);
};

//! @brief シェイプインスタンスです。
class ShapeObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = CACHE_BLOCK_SIZE,

        //! @brief SetupBlockBuffer() で渡すバッファの必要アライメントサイズです。
        BLOCK_BUFFER_ALIGNMENT  = GX2_UNIFORM_BLOCK_ALIGNMENT
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ShapeObj() : m_pRes(NULL), m_pUserPtr(NULL), m_pBufferPtr(NULL), m_pBlockBuffer(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock のサイズを計算します。
    size_t CalcBlockBufferSize() const;

    //! @brief CalcBlockBufferSize() の別名関数です。
    size_t CalculateBlockBufferSize() const
    {
        return CalcBlockBufferSize();
    }

    //! @brief UniformBlock を構築します。
    bool SetupBlockBuffer(void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock を破棄します。
    void CleanupBlockBuffer();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief シェイプのバウンディングを計算します。
    void CalcBounding(const SkeletonObj* pSkeleton)
    {
        CalculateBounding(pSkeleton, 0);
    }

    //! @brief CalcBounding() の別名関数です。
    void CalculateBounding(const SkeletonObj* pSkeleton)
    {
        CalculateBounding(pSkeleton, 0);
    }

    //! @brief 指定したメッシュのバウンディング球を計算します。
    //!
    //! @param[in] pSkeleton SkeletonObjへのポインタ。
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @pre
    //! - 構築時にバウンディング計算を有効にしている。
    //! - インデックスはメッシュの数の範囲内。
    //!
    //! @details
    //! 前提条件を満たさない場合、この関数は何も行いません。
    //!
    void CalculateBounding(const SkeletonObj* pSkeleton, int meshIndex);

    //! @brief サブメッシュのバウンディングを計算します。
    void CalcSubMeshBounding(const SkeletonObj* pSkeleton)
    {
        CalculateSubMeshBounding(pSkeleton, 0);
    }

    //! @brief CalcSubMeshBounding() の別名関数です。
    void CalculateSubMeshBounding(const SkeletonObj* pSkeleton)
    {
        CalculateSubMeshBounding(pSkeleton, 0);
    }

    //! @brief 指定したメッシュのサブメッシュのバウンディングボックスを計算します。
    //!
    //! @param[in] pSkeleton SkeletonObjへのポインタ。
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @pre
    //! - 構築時にバウンディング計算を有効にしている。
    //! - 剛体である。
    //! - インデックスはメッシュの数の範囲内。
    //!
    //! @details
    //! 前提条件を満たさない場合、この関数は何も行いません。
    //!
    void CalculateSubMeshBounding(const SkeletonObj* pSkeleton, int meshIndex);

    //! @brief シェイプに関する描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcShpBlock(int viewIndex, const Mtx34& worldMtx, int bufferIndex = 0);

    //! @brief CalcShpBlock() の別名関数です。
    void CalculateShape(int viewIndex, const Mtx34& worldMtx, int bufferIndex = 0)
    {
        CalcShpBlock(viewIndex, worldMtx, bufferIndex);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    ResShape* GetResource() { return m_pRes; }

    //! @brief リソースを取得します。
    const ResShape* GetResource() const { return m_pRes; }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief SetupBlockBuffer() 時に渡されたバッファを取得します。
    void* GetBlockBufferPtr() { return m_pBlockBuffer; }

    //! @brief UniformBlock が構築済みであるかどうかを取得します。
    bool IsBlockBufferValid() const { return (m_Flag & BLOCK_BUFFER_VALID) != 0; }

    //! @brief 参照する ResMaterial へのインデクスを取得します。
    int GetMaterialIndex() const { return m_pRes->GetMaterialIndex(); }

    //! @brief 参照する ResBone へのインデクスを取得します。
    int GetBoneIndex() const { return m_pRes->GetBoneIndex(); }

    //! @brief 参照する ResVertex へのインデクスを取得します。
    int GetVertexIndex() const { return m_pRes->GetVertexIndex(); }

    //! @brief スムーススキニングに必要なボーン数を取得します。
    int GetVtxSkinCount() const { return m_pRes->GetVtxSkinCount(); }

    //! @brief GetVtxSkinCount() の別名関数です。
    int GetVertexSkinCount() const { return GetVtxSkinCount(); }

    //! @brief 剛体のシェイプかどうかを取得します。
    bool IsRigidBody() const { return m_pRes->IsRigidBody(); }

    //! @brief リジッドスキニングのシェイプかどうかを取得します。
    bool IsRigidSkinning() const { return m_pRes->IsRigidSkinning(); }

    //! @brief スムーススキニングのシェイプかどうかを取得します。
    bool IsSmoothSkinning() const { return m_pRes->IsSmoothSkinning(); }

    //! @brief メッシュを取得します。
    ResMesh* GetResMesh(int meshIndex = 0) { return m_pRes->GetMesh(meshIndex); }

    //! @brief メッシュを取得します。
    const ResMesh* GetResMesh(int meshIndex = 0) const { return m_pRes->GetMesh(meshIndex); }

    //! @brief メッシュの数を取得します。
    int GetMeshCount() const { return m_pRes->GetMeshCount(); }

    //! @brief ビューの数を取得します。
    int GetViewCount() const { return m_NumView; }

    //! @brief サブメッシュの数を取得します。
    int GetSubMeshCount() const { return m_pRes->GetSubMeshCount(); }

    //! @brief 指定したメッシュのサブメッシュの数を取得します。
    //!
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return サブメッシュの数を返します。
    //!
    int GetSubMeshCount(int meshIndex) const
    {
        return m_pRes->GetSubMeshCount(meshIndex);
    }

    //! @brief シェイプがビューに依存するかどうかを取得します。
    bool IsViewDependent() const { return m_ViewDependent != 0; }

    //! @brief シェイプ単位の UniformBlock の数を取得します。
    int GetShpBlockCount() const { return m_NumShpBlock; }

    //! @brief GetShpBlockCount() の別名関数です。
    int GetShapeBlockCount() const { return GetShpBlockCount(); }

    //! @brief シェイプ単位の UniformBlock を取得します。
    GfxBuffer& GetShpBlock(int viewIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(viewIndex, m_NumView, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_pShpBlockArray[viewIndex * m_ViewDependent];
    }

    //! @brief シェイプ単位の UniformBlock を取得します。
    const GfxBuffer& GetShpBlock(int viewIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(viewIndex, m_NumView, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_pShpBlockArray[viewIndex * m_ViewDependent];
    }

    //! @brief UniformBlock をバッファリングする数を設定します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief UniformBlock のエンディアンスワップを行うように設定します。
    void EnableBlockSwap() { m_Flag |= BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行わないように設定します。
    void DisableBlockSwap() { m_Flag &= ~BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行うかどうかを取得します。
    bool IsBlockSwapEnabled() const { return (m_Flag & BLOCK_BUFFER_SWAP) != 0; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //! @brief ユーザエリアへのポインタを取得します。
    void* GetUserArea() { return m_pUserArea; }

    //! @brief ユーザエリアへのポインタを取得します。
    const void* GetUserArea() const { return m_pUserArea; }

    //! @brief ユーザエリアへのポインタを取得します。
    template <typename T>
    T* GetUserArea() { return static_cast<T*>(m_pUserArea); }

    //! @brief ユーザエリアへのポインタを取得します。
    template <typename T>
    const T* GetUserArea() const { return static_cast<const T*>(m_pUserArea); }

    //! @brief ユーザエリアのサイズを取得します。
    size_t GetUserAreaSize() const { return m_UserAreaSize; }

    //@}

    //----------------------------------------
    //! @name 頂点情報
    //@{

    //! @brief 参照する ResVertex を取得します。
    ResVertex* GetResVertex() { return m_pRes->GetVertex(); }

    //! @brief 参照する ResVertex を取得します。
    const ResVertex* GetResVertex() const { return m_pRes->GetVertex(); }

    //! @brief 参照する ResVertex の頂点属性の数を取得します。
    int GetVtxAttribCount() const { return GetResVertex()->GetVtxAttribCount(); }

    //! @brief GetVtxAttribCount() の別名関数です。
    int GetVertexAttrCount() const { return GetVtxAttribCount(); }

    //! @brief インデクス引きで頂点属性を取得します。
    //!
    //! インデクスはシェイプが持つ頂点属性数の範囲内とする必要があります。
    //!
    ResVtxAttrib* GetResVtxAttrib(int attribIndex)
    {
        return GetResVertex()->GetVtxAttrib(attribIndex);
    }

    //! @brief GetResVtxAttrib() の別名関数です。
    ResVertexAttr* GetResVertexAttr(int attribIndex)
    {
        return GetResVtxAttrib(attribIndex);
    }

    //! @brief インデクス引きで頂点属性を取得します。
    //!
    //! インデクスはシェイプが持つ頂点属性数の範囲内とする必要があります。
    //!
    const ResVtxAttrib* GetResVtxAttrib(int attribIndex) const
    {
        return GetResVertex()->GetVtxAttrib(attribIndex);
    }

    //! @brief GetResVtxAttrib() の別名関数です。
    const ResVertexAttr* GetResVertexAttr(int attribIndex) const
    {
        return GetResVtxAttrib(attribIndex);
    }

    //! @brief 名前引きで頂点属性を取得します。
    //!
    //! 指定した名前の頂点属性が存在しない場合は NULL を返します。
    //!
    ResVtxAttrib* GetResVtxAttrib(const char* name)
    {
        return GetResVertex()->GetVtxAttrib(name);
    }

    //! @brief GetResVtxAttrib() の別名関数です。
    ResVertexAttr* FindResVertexAttr(const char* name)
    {
        return GetResVtxAttrib(name);
    }

    //! @brief 名前引きで頂点属性を取得します。
    //!
    //! 指定した名前の頂点属性が存在しない場合は NULL を返します。
    //!
    const ResVtxAttrib* GetResVtxAttrib(const char* name) const
    {
        return GetResVertex()->GetVtxAttrib(name);
    }

    //! @brief GetResVtxAttrib() の別名関数です。
    const ResVertexAttr* FindResVertexAttr(const char* name) const
    {
        return GetResVtxAttrib(name);
    }

    //! @brief インデクスから頂点属性の名前を取得します。
    //!
    //! インデクスはシェイプが持つ頂点属性数の範囲内とする必要があります。
    //!
    const char* GetVtxAttribName(int attribIndex) const
    {
        return GetResVertex()->GetVtxAttribName(attribIndex);
    }

    //! @brief 頂点属性の名前からインデクスを取得します。
    //!
    //! 指定した名前の頂点属性が存在しない場合は -1 を返します。
    //!
    int GetVtxAttribIndex(const char* name) const
    {
        return GetResVertex()->GetVtxAttribIndex(name);
    }

    //! @brief 参照する ResVertex の頂点バッファの数を取得します。
    int GetVertexBufferCount() const { return GetResVertex()->GetVtxBufferCount(); }

    //! @brief インデクスから頂点バッファを取得します。
    //!
    //! インデクスはシェイプが持つ頂点バッファ数の範囲内とする必要があります。
    //!
    GfxBuffer& GetVertexBuffer(int bufferIndex)
    {
        return *GetResVertex()->GetVtxBuffer(bufferIndex)->GetGfxBuffer();
    }

    //! @brief インデクスから頂点バッファを取得します。
    //!
    //! インデクスはシェイプが持つ頂点バッファ数の範囲内とする必要があります。
    //!
    const GfxBuffer& GetVertexBuffer(int bufferIndex) const
    {
        return *GetResVertex()->GetVtxBuffer(bufferIndex)->GetGfxBuffer();
    }

    //@}

    //----------------------------------------
    //! @name バウンディング
    //@{

    //! @brief ワールド座標系におけるシェイプのバウンディング球を取得します。
    //!
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    Sphere* GetBounding() { return m_pBounding; }

    //! @brief ワールド座標系におけるシェイプのバウンディング球を取得します。
    //!
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    const Sphere* GetBounding() const { return m_pBounding; }

    //! @brief 指定したメッシュのワールド座標系におけるバウンディング球を取得します。
    //!
    //! param[in] meshIndex メッシュのインデックス。
    //!
    //! @return 指定したメッシュのワールド座標系におけるバウンディング球へのポインタを返します。
    //!
    //! @details
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    Sphere* GetBounding(int meshIndex)
    {
        if (m_pBounding)
        {
            return m_pBounding + meshIndex * NUM_COORD;
        }
        // LODレベルごとのバウンディングを持たない場合
        else
        {
            return NULL;
        }
    }

    //! @brief ワールド座標系におけるサブメッシュの AABB 配列を取得します。
    //!
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    const AABB* GetSubMeshBoundingArray() const { return m_pSubMeshBounding; }

     //! @brief ワールド座標系における指定したメッシュのサブメッシュの AABB 配列を取得します。
    //!
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return ワールド座標系におけるサブメッシュの AABB 配列へのポインタを返します。
    //!
    //! @details
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    const AABB* GetSubMeshBoundingArray(int meshIndex) const
    {
        int startBoundingIndex = 0;
        for (int index = 0; index < meshIndex; ++index)
        {
            startBoundingIndex += GetSubMeshCount(index);
        }
        return &m_pSubMeshBounding[startBoundingIndex];
    }

    //! @brief ビューボリュームと交差する連続したサブメッシュを取得します。
    //!
    //! コンテクストが保持するサブメッシュ列から探索し、次に交差または包含される
    //! 連続したサブメッシュをコンテクストに書き込みます。
    //! 交差または包含されるサブメッシュが存在する場合に true を返します。
    //!
    bool TestSubMeshIntersection(CullingContext* pCtx, const ViewVolume& volume) const
    {
        return TestSubMeshIntersection(pCtx, volume, 0);
    }

    //! @brief 指定したメッシュにおけるビューボリュームと交差する連続したサブメッシュを取得します。
    //!
    //! @param[in] pCtx サブメッシュ列を保持するコンテクストへのポインタ。
    //! @param[in] volume ビューボリュームへの参照。
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return 交差または包含されるサブメッシュが存在する場合はtrue、存在しない場合は false を返します。
    //!
    //! @pre
    //! - 構築時にバウンディング計算を有効にしている。
    //!
    //! @details
    //! コンテクストが保持するサブメッシュ列から探索し、次に交差または包含される
    //! 連続したサブメッシュをコンテクストに書き込みます。CullingContext::submeshIndex にビューボリューム内に入る
    //! 開始サブメッシュインデックス、CullingContext::submeshCount に開始サブメッシュインデックス
    //! からビューボリューム内に入る連続するサブメッシュ数が格納されます。
    //!
    //! 探索は最後のサブメッシュに到達するか、ビューボリューム内に入るサブメッシュを見つけた後に続くサブメッシュを
    //! 探索し、ビューボリューム内に入らないものを見つけた場合に終了し、結果を返します。pCtxのデータを保持し、続けて
    //! この関数を呼び出すことによって、ビューボリューム内に入るすべてのサブメッシュを知ることができます。
    //!
    //! 得られた CullingContext を元に nn::g3d::ResMesh::DrawSubMesh()を呼び出し、
    //! これを TestSubMeshIntersection() が false を返すまで繰り返すことによって、サブメッシュカリングを実現できます。
    //!
    bool TestSubMeshIntersection(CullingContext* pCtx, const ViewVolume& volume, int meshIndex) const;

    //! @briefprivate ビューボリュームと交差する LOD レベルの等しい連続したサブメッシュを取得します。
    //!
    //!
    //! コンテクストが保持するサブメッシュから探索し、次に交差または包含される
    //! LOD レベルの等しい連続したサブメッシュをコンテクストに書き込みます。
    //! 交差または包含されるサブメッシュが存在する場合に true を返します。
    //!
    bool TestSubMeshLodIntersection(CullingContext* pCtx, const ViewVolume& volume,
        ICalcLodLevelFunctor& calcLodFunctor) const;

    //! @brief ビューボリュームと交差する連続したサブメッシュの範囲配列を取得します。
    //!
    //! pRangeArray は index と count と lodLevel がそれぞれ 0 の SubMeshRange で終端され、
    //! 終端を含まないサブメッシュの個数が戻り値として返されます。
    //! シェイプのサブメッシュ数を N とすると SubMeshRange の最大数は (N+1)/2 です。（終端を除く）
    //!
    int MakeSubMeshRange(SubMeshRange* pRangeArray, const ViewVolume& volume) const
    {
        return MakeSubMeshRange(pRangeArray, volume, 0);
    }

    //! @brief メッシュのインデックスを指定して、ビューボリュームと交差する連続したサブメッシュの範囲配列を取得します。
    //!
    //! @param[in, out] pRangeArray SubMeshRange 配列へのポインタ。
    //! @param[in] volume ビューボリュームへの参照。
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return ビューボリュームと交差するサブメッシュの数を返します。
    //!
    //! @details
    //! pRangeArray は index と count と lodLevel がそれぞれ 0 の SubMeshRange で終端され、
    //! 終端を含まないサブメッシュの個数が戻り値として返されます。
    //! シェイプのサブメッシュ数を N とすると SubMeshRange の最大数は (N+1)/2 です。（終端を除く）
    //!
    int MakeSubMeshRange(SubMeshRange* pRangeArray, const ViewVolume& volume, int meshIndex) const;

    //! @briefprivate ビューボリュームと交差する LOD レベルの等しい連続したサブメッシュの範囲配列を取得します。
    //!
    //!
    //! pRangeArray は index と count と lodLevel がそれぞれ 0 の SubMeshRange で終端され、
    //! 終端を含まないサブメッシュの個数が戻り値として返されます。
    //! シェイプのサブメッシュ数を N とすると SubMeshRange の最大数は N です。（終端を除く）
    //!
    int MakeSubMeshLodRange(SubMeshRange* pRangeArray, const ViewVolume& volume,
        ICalcLodLevelFunctor& calcLodFunctor) const;

    //@}

    //----------------------------------------
    //! @name キーシェイプ
    //@{

    //! @brief キーシェイプの数を取得します。
    int GetKeyShapeCount() const { return m_pRes->GetKeyShapeCount(); }

    //! @brief インデクス引きでキーシェイプを取得します。
    //!
    //! インデクスはシェイプが持つキーシェイプの範囲内とする必要があります。
    //!
    ResKeyShape* GetResKeyShape(int keyShapeIndex)
    {
        return m_pRes->GetKeyShape(keyShapeIndex);
    }

    //! @brief インデクス引きでキーシェイプを取得します。
    //!
    //! インデクスはシェイプが持つキーシェイプの範囲内とする必要があります。
    //!
    const ResKeyShape* GetResKeyShape(int keyShapeIndex) const
    {
        return m_pRes->GetKeyShape(keyShapeIndex);
    }

    //! @brief 名前引きでキーシェイプを取得します。
    //!
    //! 指定した名前のキーシェイプが存在しない場合は NULL を返します。
    //!
    ResKeyShape* GetResKeyShape(const char* name)
    {
        return m_pRes->GetKeyShape(name);
    }

    //! @brief 名前引きでキーシェイプを取得します。
    //!
    //! 指定した名前のキーシェイプが存在しない場合は NULL を返します。
    //!
    const ResKeyShape* GetResKeyShape(const char* name) const
    {
        return m_pRes->GetKeyShape(name);
    }

    //! @brief インデクスからキーシェイプの名前を取得します。
    //!
    //! インデクスはシェイプが持つキーシェイプの範囲内とする必要があります。
    //!
    const char* GetKeyShapeName(int keyShapeIndex) const
    {
        return m_pRes->GetKeyShapeName(keyShapeIndex);
    }

    //! @brief キーシェイプの名前からインデクスを取得します。
    //!
    //! 指定した名前のキーシェイプが存在しない場合は -1 を返します。
    //!
    int GetKeyShapeIndex(const char* name) const
    {
        return m_pRes->GetKeyShapeIndex(name);
    }

    //! @brief キーシェイプが持つ頂点属性の数を取得します。
    int GetTargetAttribCount() const { return m_pRes->GetTargetAttribCount(); }

    //! @brief キーシェイプのブレンドウェイトを初期化します。
    void ClearBlendWeights();

    //! @brief キーシェイプのブレンドウェイトが有効かどうかを取得します。
    //!
    //! ClearBlendWeights() 呼び出し後に SetBlendWeight() を呼んでいた場合に true を返します。
    //!
    NW_G3D_DEPRECATED_FUNCTION(bool IsBlendWeightValid() const) { return 0 != (m_Flag & BLEND_WEIGHT_VALID); }

    //! @brief 有効なキーシェイプのブレンドウェイトをもつかどうかを取得します。
    //!
    //! ClearBlendWeights() 呼び出し後に SetBlendWeight() を呼んでいた場合に true を返します。
    //!
    bool HasValidBlendWeight() const { return 0 != (m_Flag & BLEND_WEIGHT_VALID); }

    //! @brief キーシェイプのブレンドウェイトを設定します。
    void SetBlendWeight(int keyShapeIndex, float weight)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(keyShapeIndex, GetKeyShapeCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_pBlendWeightArray[keyShapeIndex] = weight;
        m_Flag |= BLEND_WEIGHT_VALID;
        m_pBlendWeightValidFlags[keyShapeIndex >> 5] |= 0x01 << (keyShapeIndex & 0x1F);
    }

    //! @brief キーシェイプのブレンドウェイトを取得します。
    float GetBlendWeight(int keyShapeIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(keyShapeIndex, GetKeyShapeCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_pBlendWeightArray[keyShapeIndex];
    }

    //! @brief キーシェイプのブレンドウェイトが有効かどうかを取得します。
    //!
    //! ClearBlendWeights() 呼び出し後に SetBlendWeight(keyShapeIndex) を呼んでいた場合に true を返します。
    //!
    bool IsBlendWeightValid(int keyShapeIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(keyShapeIndex, GetKeyShapeCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return !!(m_pBlendWeightValidFlags[keyShapeIndex >> 5] & (0x01 << (keyShapeIndex & 0x1F)));
    }

    //! @brief 有効なキーシェイプのブレンドウェイトを数えます。
    int CountValidBlendWeight() const
    {
        int ret = 0;
        for (int idxFlag = 0, numFlag = Align(m_pRes->GetKeyShapeCount(), 32) >> 5;
            idxFlag < numFlag; ++idxFlag)
        {
            ret += CountOnes(m_pBlendWeightValidFlags[idxFlag]);
        }
        return ret;
    }

    //@}

protected:
    //! @brief シェイプの状態を表すフラグです。
    enum Flag
    {
        //! @briefprivate UniformBlock が構築済みであるかどうかを表すフラグです。
        //!
        //!
        BLOCK_BUFFER_VALID  = 0x1 << 0,

        //! @briefprivate UniformBlock のエンディアンスワップを行うかどうかを表すフラグです。
        //!
        //!
        BLOCK_BUFFER_SWAP   = 0x1 << 1,

        //! @briefprivate キーシェイプのブレンドウェイトが変更されたかどうかを表すフラグです。
        //!
        //!
        BLEND_WEIGHT_VALID  = 0x1 << 2
    };

    enum BoundingCoord
    {
        WORLD_COORD     = 0,
        LOCAL_COORD     = 1,
        NUM_COORD
    };

private:
    class Sizer;
    ResShape* m_pRes;

    bit32 m_Flag;

    u8 m_NumView;
    u8 m_ViewDependent; // 0 or 1
    u8 m_NumShpBlock;
    u8 m_NumBuffering;
    GfxBuffer* m_pShpBlockArray; // 行列パレット非使用時の変換行列
    float* m_pBlendWeightArray; // シェイプアニメにおけるキーシェイプ毎のウェイト
    bit32* m_pBlendWeightValidFlags; // 各キーシェイプのブレンドウェイトが有効かどうかのフラグ
    Sphere* m_pBounding;
    AABB* m_pSubMeshBounding;
    void* m_pUserArea;
    u32 m_UserAreaSize;

    void* m_pUserPtr;
    void* m_pBufferPtr;
    void* m_pBlockBuffer;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShapeObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ShapeObj のサイズを計算するためのクラスです。
//!
class ShapeObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ShapeObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        SUBBOUNDING_ARRAY,
        SHAPE_BLOCK_ARRAY,
        BLEND_WEIGHT_ARRAY,
        BLEND_WEIGHT_FLAGS,
        BOUNDING_ARRAY,
        USER_AREA,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//! @brief ShapeObj::Init() に渡して初期化を行うパラメータです。
class ShapeObj::InitArg
{
public:
    //! @brief コンストラクタです。
    explicit InitArg(ResShape* resource)
        : m_pRes(resource)
        , m_NumBuffering(1)
        , m_NumView(1)
        , m_ViewDependent(false)
        , m_BoundingEnabled(false)
        , m_UserAreaSize(0)
    {
        NW_G3D_ASSERT_NOT_NULL(resource);
    }

    //! @brief UniformBlock をバッファリングする数を設定します。
    void BufferingCount(int count) { m_NumBuffering = count; m_Sizer.Invalidate(); }

    //! @brief ビューの数を設定します。
    //!
    //! UniformBlock はビューの数に応じて複数構築されます。
    //!
    void ViewCount(int count) { m_NumView = count; m_Sizer.Invalidate(); }

    //! @brief シェイプがビューに依存することを指定します。
    //!
    //! UniformBlock はビューの数に応じて複数構築されます。
    //!
    void ViewDependent() { m_ViewDependent = true; m_Sizer.Invalidate(); }

    //! @brief シェイプがビューに依存しないことを指定します。
    //!
    //! UniformBlock はビュー間で共有されます。
    //!
    void ViewIndependent() { m_ViewDependent = false; m_Sizer.Invalidate(); }

    //! @brief バウンディング計算を有効に指定します。
    void EnableBounding() { m_BoundingEnabled = true; m_Sizer.Invalidate(); }

    //! @brief EnableBounding() の別名関数です。
    void SetBoundingEnabled() { EnableBounding(); }

    //! @brief バウンディング計算を無効に指定します。
    void DisableBounding() { m_BoundingEnabled = false; m_Sizer.Invalidate(); }

    //! @brief DisableBounding() の別名関数です。
    void SetBoundingDisabled() { DisableBounding(); }

    //! @brief ユーザエリアのサイズを設定します。
    void UserAreaSize(size_t size) { m_UserAreaSize = size; m_Sizer.Invalidate(); }

    //! @brief UniformBlock をバッファリングする数を取得します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief ビューの数を取得します。
    int GetViewCount() const { return m_NumView; }

    //! @brief シェイプがビュー依存するかどうかを取得します。
    bool IsViewDependent() const { return m_ViewDependent; }

    //! @brief バウンディング計算が有効かどうかを取得します。
    bool IsBoundingEnabled() const { return m_BoundingEnabled; }

    //! @brief ユーザエリアのサイズを取得します。
    size_t GetUserAreaSize() const { return m_UserAreaSize; }

    //! @brief リソースを取得します。
    ResShape* GetResource() const { return m_pRes; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ResShape* m_pRes;
    int m_NumBuffering;
    int m_NumView;
    bool m_ViewDependent;
    bool m_BoundingEnabled;
    size_t m_UserAreaSize;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief ShapeObj の構築を行うクラスです。
class ShapeObj::Builder : public ShapeObj::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ResShape* resource) : InitArg(resource), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ShapeObj を構築します。
    //!
    //! @param[in] pShapeObj ShapeObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ShapeObj* pShapeObj, void* pBuffer, size_t bufferSize) const
    {
        return pShapeObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ShapeObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ShapeObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ShapeObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief ShapeObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_SHAPEOBJ_H_
