﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_MODELOBJ_H_
#define NW_G3D_MODELOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/g3d_SkeletonObj.h>
#include <nw/g3d/g3d_ShapeObj.h>
#include <nw/g3d/g3d_MaterialObj.h>
#include <nw/g3d/g3d_Sizer.h>
#include <nw/g3d/ut/g3d_Flag.h>

namespace nw { namespace g3d {

//! @brief モデルインスタンスです。
class ModelObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = SkeletonObj::BUFFER_ALIGNMENT,

        //! @brief SetupBlockBuffer() で渡すバッファの必要アライメントサイズです。
        BLOCK_BUFFER_ALIGNMENT  = GX2_UNIFORM_BLOCK_ALIGNMENT
    };

    //! @briefprivate ボーンビジビリティ変更時に呼ばれるコールバックの型
    typedef void (*BoneVisibilityCallback)(ModelObj* pModel, int idxBone);

    //! @briefprivate マテリアルビジビリティ変更時に呼ばれるコールバックの型
    typedef void (*MaterialVisibilityCallback)(ModelObj* pModel, int idxMaterial);
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ModelObj() : m_pRes(NULL), m_pUserPtr(NULL), m_pBufferPtr(NULL), m_pBlockBuffer(NULL), m_pBoneVisibilityCallback(NULL), m_pMaterialVisibilityCallback(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock のサイズを計算します。
    //!
    //! 配下の ShapeObj, MaterialObj, SkeletonObj のバッファサイズも合わせて返します。
    //!
    size_t CalcBlockBufferSize();

    //! @brief CalcBlockBufferSize() の別名関数です。
    size_t CalculateBlockBufferSize()
    {
        return CalcBlockBufferSize();
    }

    //! @brief UniformBlock を構築します。
    bool SetupBlockBuffer(void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock を破棄します。
    void CleanupBlockBuffer();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief ローカル行列からワールド行列を計算します。
    void CalcWorld(const Mtx34& baseMtx);

    //! @brief  CalcWorld() の別名関数です。
    void CalculateWorld(const Mtx34& baseMtx)
    {
        CalcWorld(baseMtx);
    }

    //! @brief バウンディングを計算します。
    void CalcBounding()
    {
        CalculateBounding(0);
    }

    //! @brief  CalcBounding() の別名関数です。
    void CalculateBounding()
    {
        CalculateBounding(0);
    }

    //! @brief 指定したLODレベルのバウンディング球を計算します。
    //!
    //! @param[in] lodLevel LODレベル。
    //!
    //! @details
    //! すべての ShapeObj を走査し、モデルのバウンディング球を計算します。
    //! ShapeObj が指定した LOD レベルを持たない場合、LOD レベル 0 のデータで代用し、
    //! 計算を行います。
    //!
    void CalculateBounding(int lodLevel);

    //! @brief ワールド行列から行列パレットを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcMtxBlock(int bufferIndex = 0);

    //! @brief  CalcMtxBlock() の別名関数です。
    void CalculateSkeleton(int bufferIndex = 0)
    {
        CalcMtxBlock(bufferIndex);
    }

    //! @brief シェイプに関する描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcShape(int bufferIndex = 0);

    //! @brief  CalcShape() の別名関数です。
    void CalculateShape(int bufferIndex = 0)
    {
        CalcShape(bufferIndex);
    }

    //! @brief マテリアルに関する描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcMaterial(int bufferIndex = 0);

    //! @brief  CalcMaterial() の別名関数です。
    void CalculateMaterial(int bufferIndex = 0)
    {
        CalcMaterial(bufferIndex);
    }

    //! @brief ビューに関する描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcView(int viewIndex, const Mtx34& cameraMtx, int bufferIndex = 0);

    //! @brief  CalcView() の別名関数です。
    void CalculateView(int viewIndex, const Mtx34& cameraMtx, int bufferIndex = 0)
    {
        CalcView(viewIndex, cameraMtx, bufferIndex);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    ResModel* GetResource() { return m_pRes; }

    //! @brief リソースを取得します。
    const ResModel* GetResource() const { return m_pRes; }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief SetupBlockBuffer() 時に渡されたバッファを取得します。
    void* GetBlockBufferPtr() { return m_pBlockBuffer; }

    //! @brief モデル配下のオブジェクトの UniformBlock が構築済みであるかどうかを取得します。
    bool IsBlockBufferValid() const { return (m_Flag & BLOCK_BUFFER_VALID) != 0; }

    //! @brief モデル配下のオブジェクトに対して UniformBlock のエンディアンスワップを行うように設定します。
    //!
    //! 配下の SkeletonObj, ShapeObj, MaterialObj の EnableBlockSwap を呼び出します。
    //!
    void EnableBlockSwapAll();

    //! @brief モデル配下のオブジェクトに対して UniformBlock のエンディアンスワップを行なわないように設定します。
    //!
    //! 配下の SkeletonObj, ShapeObj, MaterialObj の DisableBlockSwap を呼び出します。
    //!
    void DisableBlockSwapAll();

    //! @brief ビューの数を取得します。
    //!
    //! ModelObj::InitArg の ViewCount で指定した数です。
    //!
    int GetViewCount() const { return m_NumView; }

    //! @brief モデルがビューに依存するかどうかを取得します。
    bool IsViewDependent() const { return !!m_ViewDependent; }

    //! @brief モデルがビューに依存するかどうかを更新します。
    //!
    //! ビューに依存するかどうかは初期化時に設定されるため、通常は呼ぶ必要はありません。
    //!
    void UpdateViewDependency();

    //! @brief ワールド座標系におけるモデルのバウンディング球を取得します。
    //!
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    Sphere* GetBounding()
    {
        return GetBounding(0);
    }

    //! @brief ワールド座標系におけるモデルのバウンディング球を取得します。
    //!
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    const Sphere* GetBounding() const
    {
        return GetBounding(0);
    }

    //! @brief ワールド座標系における指定した LOD のモデルのバウンディング球を取得します。
    //!
    //! @param[in] lodLevel lod レベル。
    //!
    //! @return Sphere へのポインタを返します。
    //!
    //! @details
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    Sphere* GetBounding(int lodLevel)
    {
        return &m_pBounding[lodLevel];
    }

    //! @brief ワールド座標系における指定した LOD のモデルのバウンディング球を取得します。
    //!
    //! @param[in] lodLevel lod レベル。
    //!
    //! @return Sphere へのポインタを返します。
    //!
    //! @details
    //! 構築時にバウンディングを無効に指定した場合は NULL を返します。
    //!
    const Sphere* GetBounding(int lodLevel) const
    {
        return &m_pBounding[lodLevel];
    }

    //! @brief モデルの持つ LOD 数を返します。
    //!
    //! @return モデルの持つ LOD 数。
    //!
    int GetLodCount() const
    {
        return m_LodCount;
    }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}

    //----------------------------------------
    //! @name ビジビリティ
    //@{

    //! @brief 指定したシェイプのビジビリティを取得します。
    //!
    //! IsBoneVisible() と IsMatVisible() を合わせたビジビリティを取得します。
    //!
    bool IsShapeVisible(int shapeIndex) const
    {
        const ShapeObj* pShape = GetShape(shapeIndex);
        return IsMatVisible(pShape->GetMaterialIndex()) &&
            IsBoneVisible(pShape->GetBoneIndex());
    }

    //! @brief 指定したボーンのビジビリティを取得します。
    bool IsBoneVisible(int boneIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(boneIndex, GetSkeleton()->GetBoneCount(),
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return IsBitOn<bool>(m_pBoneVisArray, boneIndex);
    }

    //! @brief 指定したボーンのビジビリティを設定します。
    void SetBoneVisibility(int boneIndex, bool visible)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(boneIndex, GetSkeleton()->GetBoneCount(),
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

        bool oldVisible = IsBoneVisible(boneIndex);
        SetBit(m_pBoneVisArray, boneIndex,  visible ? 1 : 0);
        if (m_pBoneVisibilityCallback && (oldVisible != visible))
        {
            m_pBoneVisibilityCallback(this, boneIndex);
        }
    }

    //! @brief SetBoneVisibility() の別名関数です。
    void SetBoneVisible(int boneIndex, bool visible)
    {
        SetBoneVisibility(boneIndex, visible);
    }

    //! @brief ボーンのビジビリティを表す配列を取得します。
    bit32* GetBoneVisArray() { return m_pBoneVisArray; }

    //! @brief ボーンのビジビリティを表す配列を取得します。
    const bit32* GetBoneVisArray() const { return m_pBoneVisArray; }

    //! @brief ボーンのビジビリティをリソースに基づいて初期化します。
    //!
    //! Init() 内で呼ばれるため、意図的に結果を初期化したい場合以外は呼ぶ必要はありません。
    //!
    void ClearBoneVisibility();

    //! @brief ClearBoneVisibility() の別名関数です。
    void ClearBoneVisible()
    {
        ClearBoneVisibility();
    }

    //! @briefprivate ボーンビジビリティ変更時に呼ばれるコールバックを設定します。
    void SetBoneVisibilityCallback(BoneVisibilityCallback pCallback)
    {
        m_pBoneVisibilityCallback = pCallback;
    }

    //! @briefprivate ボーンビジビリティ変更時に呼ばれるコールバックを取得します。
    BoneVisibilityCallback GetBoneVisibilityCallback() const
    {
        return m_pBoneVisibilityCallback;
    }

    //! @brief 指定したマテリアルのビジビリティを取得します。
    bool IsMatVisible(int matIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(matIndex, GetMaterialCount(),
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return IsBitOn<bool>(m_pMatVisArray, matIndex);
    }

    //! @brief IsMatVisible() の別名関数です。
    bool IsMaterialVisible(int matIndex) const
    {
        return IsMatVisible(matIndex);
    }

    //! @brief 指定したマテリアルのビジビリティを設定します。
    void SetMatVisibility(int matIndex, bool visible)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(matIndex, GetMaterialCount(),
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

        bool oldVisible = IsMatVisible(matIndex);
        SetBit(m_pMatVisArray, matIndex,  visible ? 1 : 0);
        if (m_pMaterialVisibilityCallback && (oldVisible != visible))
        {
            m_pMaterialVisibilityCallback(this, matIndex);
        }
    }

    //! @brief SetMatVisibility() の別名関数です。
    void SetMaterialVisible(int matIndex, bool visible)
    {
        SetMatVisibility(matIndex, visible);
    }

    //! @brief マテリアルのビジビリティを表す配列を取得します。
    bit32* GetMatVisArray() { return m_pMatVisArray; }

    //! @brief マテリアルのビジビリティを表す配列を取得します。
    const bit32* GetMatVisArray() const { return m_pMatVisArray; }

    //! @brief マテリアルのビジビリティをリソースに基づいて初期化します。
    //!
    //! Init() 内で呼ばれるため、意図的に結果を初期化したい場合以外は呼ぶ必要はありません。
    //!
    void ClearMatVisibility();

    //! @brief ClearMatVisibility() の別名関数です。
    void ClearMaterialVisible()
    {
        ClearMatVisibility();
    }

    //! @briefprivate マテリアルビジビリティ変更時に呼ばれるコールバックを設定します。
    void SetMaterialVisibilityCallback(MaterialVisibilityCallback pCallback)
    {
        m_pMaterialVisibilityCallback = pCallback;
    }

    //! @briefprivate マテリアルビジビリティ変更時に呼ばれるコールバックを取得します。
    MaterialVisibilityCallback GetMaterialVisibilityCallback() const
    {
        return m_pMaterialVisibilityCallback;
    }
    //@}

    //----------------------------------------
    //! @name スケルトン
    //@{

    //! @brief スケルトンを取得します。
    SkeletonObj* GetSkeleton() { return m_pSkeleton; }

    //! @brief スケルトンを取得します。
    const SkeletonObj* GetSkeleton() const { return m_pSkeleton; }

    //@}

    //----------------------------------------
    //! @name シェイプ
    //@{

    //! @brief モデル配下のシェイプの数を取得します。
    int GetShapeCount() const { return m_NumShape; }

    //! @brief インデクス引きでシェイプを取得します。
    //!
    //! インデクスはモデルが持つシェイプ数の範囲内とする必要があります。
    //!
    ShapeObj* GetShape(int shapeIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(shapeIndex, m_NumShape,
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return &m_pShapeArray[shapeIndex];
    }

    //! @brief インデクス引きでシェイプを取得します。
    //!
    //! インデクスはモデルが持つシェイプ数の範囲内とする必要があります。
    //!
    const ShapeObj* GetShape(int shapeIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(shapeIndex, m_NumShape,
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return &m_pShapeArray[shapeIndex];
    }

    //! @brief 名前引きでシェイプを取得します。
    //!
    //! 指定した名前のシェイプが存在しない場合は NULL を返します。
    //!
    ShapeObj* GetShape(const char* name)
    {
        int shapeIndex = m_pRes->GetShapeIndex(name);
        return shapeIndex >= 0 ? GetShape(shapeIndex) : NULL;
    }

    //! @brief GetShape() の別名関数です。
    ShapeObj* FindShape(const char* name)
    {
        return GetShape(name);
    }

    //! @brief 名前引きでシェイプを取得します。
    //!
    //! 指定した名前のシェイプが存在しない場合は NULL を返します。
    //!
    const ShapeObj* GetShape(const char* name) const
    {
        int shapeIndex = m_pRes->GetShapeIndex(name);
        return shapeIndex >= 0 ? GetShape(shapeIndex) : NULL;
    }

    //! @brief GetShape() の別名関数です。
    const ShapeObj* FindShape(const char* name) const
    {
        return GetShape(name);
    }

    //! @brief インデクスからシェイプの名前を取得します。
    //!
    //! インデクスはモデルが持つシェイプ数の範囲内とする必要があります。
    //!
    const char* GetShapeName(int shapeIndex) const { return m_pRes->GetShapeName(shapeIndex); }

    //! @brief シェイプの名前からインデクスを取得します。
    //!
    //! 指定した名前のシェイプが存在しない場合は -1 を返します。
    //!
    int GetShapeIndex(const char* name) const { return m_pRes->GetShapeIndex(name); }

    //! @brief GetShapeIndex() の別名関数です。
    int FindShapeIndex(const char* name) const { return GetShapeIndex(name); }

    //@}

    //----------------------------------------
    //! @name マテリアル
    //@{

    //! @brief モデル配下のマテリアル数を取得します。
    int GetMaterialCount() const { return m_NumMaterial; }

    //! @brief インデクス引きでマテリアルを取得します。
    //!
    //! インデクスはモデルが持つマテリアル数の範囲内とする必要があります。
    //!
    MaterialObj* GetMaterial(int matIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(matIndex, m_NumMaterial,
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return &m_pMaterialArray[matIndex];
    }

    //! @brief インデクス引きでマテリアルを取得します。
    //!
    //! インデクスはモデルが持つマテリアル数の範囲内とする必要があります。
    //!
    const MaterialObj* GetMaterial(int matIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(matIndex, m_NumMaterial,
                                        "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return &m_pMaterialArray[matIndex];
    }

    //! @brief 名前引きでマテリアルを取得します。
    //!
    //! 指定した名前のマテリアルが存在しない場合は NULL を返します。
    //!
    MaterialObj* GetMaterial(const char* name)
    {
        int matIndex = m_pRes->GetMaterialIndex(name);
        return matIndex >= 0 ? GetMaterial(matIndex) : NULL;
    }

    //! @brief GetMaterial() の別名関数です。
    MaterialObj* FindMaterial(const char* name)
    {
        return GetMaterial(name);
    }

    //! @brief 名前引きでマテリアルを取得します。
    //!
    //! 指定した名前のマテリアルが存在しない場合は NULL を返します。
    //!
    const MaterialObj* GetMaterial(const char* name) const
    {
        int matIndex = m_pRes->GetMaterialIndex(name);
        return matIndex >= 0 ? GetMaterial(matIndex) : NULL;
    }

    //! @brief GetMaterial() の別名関数です。
    const MaterialObj* FindMaterial(const char* name) const
    {
        return GetMaterial(name);
    }

    //! @brief インデクスからマテリアルの名前を取得します。
    //!
    //! インデクスはモデルが持つマテリアル数の範囲内とする必要があります。
    //!
    const char* GetMaterialName(int matIndex) const { return m_pRes->GetMaterialName(matIndex); }

    //! @brief マテリアルの名前からインデクスを取得します。
    //!
    //! 指定した名前のマテリアルが存在しない場合は -1 を返します。
    //!
    int GetMaterialIndex(const char* name) const { return m_pRes->GetMaterialIndex(name); }

    //! @brief GetMaterial() の別名関数です。
    int FindMaterialIndex(const char* name) const
    {
        return GetMaterialIndex(name);
    }

    //! @briefprivate テクスチャ差し替え時に呼ばれるコールバックを設定します。
    void SetTextureChangeCallback(MaterialObj::TextureChangeCallback pCallback);
    //@}

protected:
    class Sizer;

    //! @brief モデルの状態を表すフラグです。
    enum Flag
    {
        //! @briefprivate UniformBlock が構築済みであるかどうかを表すフラグです。
        //!
        //!
        BLOCK_BUFFER_VALID  = 0x1 << 0
    };

private:

    ResModel* m_pRes;

    bit32* m_pBoneVisArray; // ボーンのビジビリティビット配列
    bit32* m_pMatVisArray; // マテリアルのビジビリティビット配列
    u8 m_NumView;
    u8 m_ViewDependent;
    bit16 m_Flag;
    GfxBuffer* m_pViewBlockArray; // ビュー行列とビルボード行列用ブロックの配列

    u16 m_NumShape;
    u16 m_NumMaterial;

    SkeletonObj* m_pSkeleton;
    ShapeObj* m_pShapeArray;
    MaterialObj* m_pMaterialArray;
    Sphere* m_pBounding;

    void* m_pUserPtr;
    void* m_pBufferPtr;
    void* m_pBlockBuffer;

    BoneVisibilityCallback m_pBoneVisibilityCallback;
    MaterialVisibilityCallback m_pMaterialVisibilityCallback;
    int m_LodCount;
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ModelObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ModelObj のサイズを計算するためのクラスです。
//!
class ModelObj::Sizer : public nw::g3d::Sizer
{
public:

    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ModelObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        SKELETON_BUFFER,
        SHAPE_BUFFER,
        MATERIAL_BUFFER,
        SKELETON,
        SHAPE_ARRAY,
        MATERIAL_ARRAY,
        BONE_VIS_ARRAY,
        MAT_VIS_ARRAY,
        BOUNDING,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief ModelObj::Init() に渡して初期化を行うパラメータです。
class ModelObj::InitArg
{
public:
    //! コンストラクタです。
    explicit InitArg(ResModel* resource)
        : m_pRes(resource)
        , m_pSkeleton(NULL)
        , m_NumSkelBuffering(1)
        , m_NumShpBuffering(1)
        , m_NumMatBuffering(1)
        , m_NumView(1)
        , m_LodCount(1)
        , m_ShapeUserAreaSize(0)
        , m_BoundingEnabled(false)
    {
        NW_G3D_ASSERT_NOT_NULL(resource);
        // モデルのシェイプを走査し、LOD数を算出
        for (int shapeIndex = 0; shapeIndex < m_pRes->GetShapeCount(); ++shapeIndex)
        {
            const nw::g3d::ResShape* pResShape = m_pRes->GetShape(shapeIndex);
            m_LodCount = std::max(m_LodCount, pResShape->GetMeshCount());
        }
    }

    //! @brief 共有スケルトンです。
    //!
    //! 構築済み SkeletonObj を渡すことで他の ModelObj とスケルトンを共有することができます。
    //! 共有することで計算処理をスキップすることができます。
    //!
    void Skeleton(SkeletonObj* pSkeleton) { m_pSkeleton = pSkeleton; m_Sizer.Invalidate(); }

    //! @brief スケルトンの UniformBlock をバッファリングする数を設定します。
    void SkeletonBufferingCount(int count) { m_NumSkelBuffering = count; m_Sizer.Invalidate(); }

    //! @brief シェイプの UniformBlock をバッファリングする数を設定します。
    void ShapeBufferingCount(int count) { m_NumShpBuffering = count; m_Sizer.Invalidate(); }

    //! @brief マテリアルの UniformBlock をバッファリングする数を設定します。
    void MaterialBufferingCount(int count) { m_NumMatBuffering = count; m_Sizer.Invalidate(); }

    //! @brief ビューの数を設定します。
    //!
    //! ビューに依存する UniformBlock はビューの数に応じて複数構築されます。
    //!
    void ViewCount(int count) { m_NumView = count; m_Sizer.Invalidate(); }

    //! @brief シェイプのユーザエリアサイズを設定します。
    void ShapeUserAreaSize(size_t size) { m_ShapeUserAreaSize = size; }

    //! @brief バウンディング計算を有効に指定します。
    void EnableBounding() { m_BoundingEnabled = true; m_Sizer.Invalidate(); }

    //! @brief EnableBounding() の別名関数です。
    void SetBoundingEnabled() { EnableBounding(); }

    //! @brief バウンディング計算を無効に指定します。
    void DisableBounding() { m_BoundingEnabled = false; m_Sizer.Invalidate(); }

    //! @brief DisableBounding() の別名関数です。
    void SetBoundingDisabled() { DisableBounding(); }

    //! @brief リソースを取得します。
    ResModel* GetResource() const { return m_pRes; }

    //! @brief スケルトンを取得します。
    SkeletonObj* GetSkeleton() const { return m_pSkeleton; }

    //! @brief スケルトンの UniformBlock をバッファリングする数を取得します。
    int GetSkeletonBufferingCount() const { return m_NumSkelBuffering; }

    //! @brief シェイプの UniformBlock をバッファリングする数を取得します。
    int GetShapeBufferingCount() const { return m_NumShpBuffering; }

    //! @brief マテリアルの UniformBlock をバッファリングする数を取得します。
    int GetMaterialBufferingCount() const { return m_NumMatBuffering; }

    //! @brief ビュー用の数を取得します。
    int GetViewCount() const { return m_NumView; }

    //! @brief モデルのLODの数を取得します。
    //!
    //! @return Lodの数を返します。
    //!
    int GetLodCount() const
    {
        return m_LodCount;

    }

    //! @brief シェイプのユーザエリアサイズを取得します。
    size_t GetShapeUserAreaSize() const { return m_ShapeUserAreaSize; }

    //! @brief バウンディング計算が有効かどうかを取得します。
    bool IsBoundingEnabled() const { return m_BoundingEnabled; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ResModel* m_pRes;
    SkeletonObj* m_pSkeleton;
    int m_NumSkelBuffering;
    int m_NumShpBuffering;
    int m_NumMatBuffering;
    int m_NumView;
    int m_LodCount;
    size_t m_ShapeUserAreaSize;
    bool m_BoundingEnabled;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief ModelObj の構築を行うクラスです。
class ModelObj::Builder : public ModelObj::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ResModel* resource) : InitArg(resource), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ModelObj を構築します。
    //!
    //! @param[in] pModelObj ModelObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ModelObj* pModelObj, void* pBuffer, size_t bufferSize) const
    {
        return pModelObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ModelObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ModelObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ModelObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief ModelObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_MODELOBJ_H_
