﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GLUTILITY_H_
#define NW_G3D_FND_GLUTILITY_H_

#include <nw/g3d/g3d_config.h>

// TODO: レベル毎の名前を考える
#define NW_G3D_GL_LEVEL0 0
#define NW_G3D_GL_LEVEL1 1
#define NW_G3D_GL_LEVEL2 2
#define NW_G3D_GL_LEVEL3 3

#if defined(NW_G3D_IS_GL_ES)
#define NW_G3D_GL_PORTABILITY NW_G3D_GL_LEVEL2
#else
#define NW_G3D_GL_PORTABILITY NW_G3D_GL_LEVEL1
#endif

#if !NW_G3D_IS_GL && !defined(NW_STRIP_GL)
#define NW_STRIP_GL // GL 以外では NW_STRIP_GL を必ず定義する。
#endif

#ifndef NW_STRIP_GL

#if defined(NW_G3D_IS_GL_ES)
#if defined(ANDROID)
#include <GLES3/gl3.h>
#elif defined(__APPLE__)
#include <OpenGLES/ES3/glext.h>
#endif
#else
#include <gl/glew.h>
#endif

#if NW_G3D_IS_DEBUG || NW_G3D_IS_DEVELOP
#define NW_G3D_GL_ASSERT()  NW_G3D_GL_ASSERT_DETAIL("")
#define NW_G3D_GL_ASSERT_DETAIL(...)                                                              \
    do                                                                                            \
    {                                                                                             \
        GLenum err = glGetError();                                                                \
        if(GL_NO_ERROR != err)                                                                    \
        {                                                                                         \
            nw::g3d::DebugPrint("GL Error: %s (0x%x)n", nw::g3d::GetGLErrorString(err), err);     \
            NW_G3D_ASSERTMSG(0, __VA_ARGS__);                                                     \
        }                                                                                         \
    } while (NW_G3D_STATIC_CONDITION(0))
#else
#define NW_G3D_GL_ASSERT()              (void)0
#define NW_G3D_GL_ASSERT_DETAIL(...)    (void)0
#endif

namespace nw { namespace g3d { namespace fnd {

void SetSwapInterval(int interval);

bool IsScalarType(GLenum type);
bool IsVectorType(GLenum type);
bool IsMatrixType(GLenum type);
bool IsSamplerType(GLenum type);

const char* GetGLErrorString(GLenum err);
const char* GetGLDebugString(GLenum debug);
const char* GetGLTypeString(GLenum type);
const char* GetFramebufferStatusString(GLenum status);

GLuint CreateShaderProgram();
void AttachShader(GLuint hProgram, GLenum type, const GLchar* string);
void AttachShader(GLuint hProgram, GLenum type, const GLchar** strings, GLsizei numStrings);
void LinkShaderProgram(GLuint hProgram);
void DestroyAttachedShader(GLuint hProgram);
void DestroyShaderProgram(GLuint hProgram);

bool CheckShaderStatus(GLuint hShader);
bool CheckProgramStatus(GLuint hProgram);
bool CheckFramebufferStatus(GLuint hFramebuffer);

namespace detail {

class TextureBinder
{
public:
    TextureBinder(GLenum target, GLuint handle) : m_Target(target)
    {
        glBindTexture(m_Target, handle);
    }
    ~TextureBinder() { glBindTexture(m_Target, 0); }

private:
    GLenum m_Target;
};

class BufferBinder
{
public:
    BufferBinder(GLenum target, GLuint handle) : m_Target(target)
    {
        glBindBuffer(target, handle);
    }
    ~BufferBinder() { glBindBuffer(m_Target, 0); }

private:
    GLenum m_Target;
};

class VertexArrayBinder
{
public:
    VertexArrayBinder(GLuint handle)
    {
        glBindVertexArray(handle);
    }
    ~VertexArrayBinder() { glBindVertexArray(0); }
};

#if NW_G3D_GL_PORTABILITY >= NW_G3D_GL_LEVEL2
#define NW_G3D_GL_BIND_TEXTURE(target, handle) TextureBinder binder(target, handle)
#else
#define NW_G3D_GL_BIND_TEXTURE(target, handle) ((void)0)
#endif

class GL
{
public:

    static void BindTexture(GLenum unit, GLenum target, GLuint handle);

    static void TexParameter(GLuint hTexture, GLenum target, GLenum name, GLint value);

    static void TexParameter(GLuint hTexture, GLenum target, GLenum name, const GLint* value);

    static void TexImage(GLuint hTexture, GLenum target, GLint mipLevel, GLint detailFormat,
        GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type,
        GLboolean compressed, GLsizei imageSize, const GLvoid* data);

    static void TexSubImage(GLuint hTexture, GLenum target, GLint mipLevel, GLint detailFormat,
        GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type,
        GLboolean compressed, GLsizei imageSize, const GLvoid* data);

    static void GetTexImage(GLuint hTexture, GLenum target, GLint mipLevel,
        GLenum format, GLenum type, GLsizei imageSize, GLvoid* data);

    static void EnableVertexAttribArray(GLuint hVertexArray, GLuint location, GLboolean enabled);

    static void VertexAttribOffset(GLuint hVertexArray, GLuint hVertexBuffer, GLuint location,
        GLint size, GLenum type, GLboolean integer, GLboolean normalized,
        GLsizei stride, const GLvoid* offset);

    static void VertexAttribDivisor(GLuint hVertexArray, GLuint location, GLuint divisor);

    static void ProgramUniform(GLuint hProgram, GLint location, GLint value);

    static void ProgramUniform(GLuint hProgram, GLint location, GLfloat value);

    static void Enable(GLenum cap, bool enabled);

    static void Enable(GLenum cap, GLuint targetIndex, bool enabled);

    static void BlendEquation(GLuint targetIndex, GLenum modeRGB, GLenum modeAlpha);

    static void BlendFunc(GLuint targetIndex,
        GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha);

    static void ColorMask(GLuint targetIndex,
        GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha);

    static void FrameBufferDrawBuffers(GLuint handle, GLsizei num, const GLenum* bufs);

    static void FrameBufferTexture(
        GLuint hFramebuffer, GLenum attachment, GLuint hTexture, GLint mipLevel);

    static void FrameBufferTexture(
        GLuint hFramebuffer, GLenum attachment, GLuint hTexture, GLint mipLevel, GLint layer);
};

} // detail

}}} // nw::g3d::fnd

#else // NW_STRIP_GL

#define NW_G3D_GL_ASSERT() (void)0

#endif // NW_STRIP_GL

#endif // NW_G3D_FND_GLUTILITY_H_
