﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_RESOURCEFORMAT_H_
#define NW_FONT_RESOURCEFORMAT_H_

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma once
#endif

#include <nw/types.h>
#include <nw/ut/ut_BinaryFileFormat.h>
#include <nw/ut/res/ut_ResTypes.h>


#ifdef _MSC_VER // for VC
    #pragma warning( push )
    #pragma warning( disable: 4200 )
    #pragma warning( disable: 4201 )
#endif  //_MSC_VER



namespace nw {
namespace font {


const int   GlyphDataAlignment   = 128;


/* ------------------------------------------------------------------------
        シグネチャ定義
   ------------------------------------------------------------------------ */

// 通常フォントリソースファイルシグネチャ
const ut::SigWord BINFILE_SIG_FONT              = NW_UT_MAKE_SIGWORD('F','F','N','T');

// オフセット解決済みバイナリファイルシグネチャ
const ut::SigWord BINFILE_SIG_FONT_RESOLEVED    = NW_UT_MAKE_SIGWORD('F','F','N','U');

// アーカイブフォントリソースファイルシグネチャ
const ut::SigWord BINFILE_SIG_FONTA             = NW_UT_MAKE_SIGWORD('F','F','N','A');

// バイナリブロックシグネチャ
const ut::SigWord BINBLOCK_SIG_FINF             = NW_UT_MAKE_SIGWORD('F','I','N','F');
const ut::SigWord BINBLOCK_SIG_CGLP             = NW_UT_MAKE_SIGWORD('C','G','L','P');
const ut::SigWord BINBLOCK_SIG_TGLP             = NW_UT_MAKE_SIGWORD('T','G','L','P');
const ut::SigWord BINBLOCK_SIG_CWDH             = NW_UT_MAKE_SIGWORD('C','W','D','H');
const ut::SigWord BINBLOCK_SIG_CMAP             = NW_UT_MAKE_SIGWORD('C','M','A','P');
const ut::SigWord BINBLOCK_SIG_KRNG             = NW_UT_MAKE_SIGWORD('K','R','N','G');

const ut::SigWord BINBLOCK_SIG_GLGR             = NW_UT_MAKE_SIGWORD('G','L','G','R');
const ut::SigWord BINBLOCK_SIG_HTGL             = NW_UT_MAKE_SIGWORD('H','T','G','L');





/* ------------------------------------------------------------------------
        リリースバージョン定義
   ------------------------------------------------------------------------ */

const u32 FONT_FILE_VERSION         = NW_UT_MAKE_VERSION(3, 0, 0, 0);



/* ------------------------------------------------------------------------
        定数定義
   ------------------------------------------------------------------------ */
const u16 INVALID_CHAR_CODE   = 0xFFFF;
const u16 INVALID_GLYPH_INDEX = 0xFFFF;





/* ------------------------------------------------------------------------
        enum定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FontType

  Description:  フォントタイプ
 *---------------------------------------------------------------------------*/
enum FontType
{
    FONT_TYPE_GLYPH,            // BMP
    FONT_TYPE_TEXTURE,          // Texture
    FONT_TYPE_PACKED_TEXTURE,   // 1つのチャンネルに複数のチャンネルを詰めた形式のテクスチャ
    NUM_OF_FONT_TYPE
};

//---------------------------------------------------------------------------
//! @brief        文字コード
//---------------------------------------------------------------------------
enum CharacterCode
{
    CHARACTER_CODE_UNICODE = 1,  //!< Unicode
    CHARACTER_CODE_SJIS,         //!< ShiftJIS
    CHARACTER_CODE_CP1252,       //!< CP1252

    CHARACTER_CODE_MAX
};

/*---------------------------------------------------------------------------*
  Name:         FontMapMethod

  Description:  文字コードマッピングタイプ
 *---------------------------------------------------------------------------*/
enum FontMapMethod
{
    FONT_MAPMETHOD_DIRECT,      // インデックス = 文字コード - オフセット
    FONT_MAPMETHOD_TABLE,       // インデックス = mapInfo[文字コード - オフセット]
    FONT_MAPMETHOD_SCAN,        // インデックス = search(mapInfo, 文字コード)
    NUM_OF_FONT_MAPMETHOD
};


/*---------------------------------------------------------------------------*
  Name:         FontSheetFormat

  Description:  シートの形式 (テクスチャフォーマット)
 *---------------------------------------------------------------------------*/
enum FontSheetFormat
{
    FONT_SHEET_FORMAT_RGBA8,
    FONT_SHEET_FORMAT_RGB8,
    FONT_SHEET_FORMAT_RGB5A1,
    FONT_SHEET_FORMAT_RGB565,
    FONT_SHEET_FORMAT_RGBA4,
    FONT_SHEET_FORMAT_LA8,
    FONT_SHEET_FORMAT_LA4,
    FONT_SHEET_FORMAT_A4,
    FONT_SHEET_FORMAT_A8 = 8,

    FONT_SHEET_FORMAT_BC1,
    FONT_SHEET_FORMAT_BC2,
    FONT_SHEET_FORMAT_BC3,
    FONT_SHEET_FORMAT_BC4 = 12,
    FONT_SHEET_FORMAT_BC5,
    FONT_SHEET_FORMAT_RGBA8_SRGB,
    FONT_SHEET_FORMAT_BC1_SRGB,
    FONT_SHEET_FORMAT_BC2_SRGB,
    FONT_SHEET_FORMAT_BC3_SRGB,
    FONT_SHEET_FORMAT_QUANTITY,



    FONT_SHEET_FORMAT_MASK              = 0x3FFF,
    FONT_SHEET_FORMAT_LINEAR_FLAG       = 0x4000,   // 1 ならリニア形式なので、タイリング解除の処理は不要
    FONT_SHEET_FORMAT_COMPRESSED_FLAG   = 0x8000    // 1 なら圧縮されている(現在は使われていません)
};




/* ------------------------------------------------------------------------
        構造体定義
   ------------------------------------------------------------------------ */

//---------------------------------------------------------------------------
//! @brief        文字の各種幅を保持する構造体です。
//---------------------------------------------------------------------------
struct CharWidths
{
    ut::ResS8  left;                   //!< 文字の左スペースの幅
    ut::ResU8  glyphWidth;             //!< 文字のグリフ幅
    ut::ResU8  charWidth;              //!< 文字の幅 = 左スペース幅 + グリフ幅 + 右スペース幅
};

/*---------------------------------------------------------------------------*
  Name:         CMapScanEntry

  Description:  文字コードとグリフインデックスペア
 *---------------------------------------------------------------------------*/
struct CMapScanEntry
{
    ut::ResU16 ccode;                  // 文字コード
    ut::ResU16 index;                  // グリフインデックス
};

/*---------------------------------------------------------------------------*
  Name:         CMapInfoScan

  Description:  MAPMETHOD_SCAN の場合の FontCodeMap.mapInfo
 *---------------------------------------------------------------------------*/
struct CMapInfoScan
{
    ut::ResU16 num;             // entries の要素数
    CMapScanEntry entries[];    // 文字コードからグリフインデックスへのマッピングリスト
};

//---------------------------------------------------------------------------
//! @brief      一つ目の文字から二つ目のテーブルへのオフセットを
//!             引くためのテーブルの要素です。
//---------------------------------------------------------------------------
struct KerningFirstTableElem
{
    ut::ResU16 firstWord;       // 一つ目の文字
    ut::ResU16 offset;          // 二つ目のテーブルへのオフセット。値はFontKerningTableの先頭アドレスからのオフセットを2で割ったものとなります。
};

//---------------------------------------------------------------------------
//! @brief      二つ目のテーブルの要素を表します。
//---------------------------------------------------------------------------
struct KerningSecondTableElem
{
    ut::ResU16 secondWord;      // 二つ目の文字
    ut::ResS16 kerningValue;    // カーニングの値
};

//---------------------------------------------------------------------------
//! @brief      二つ目のテーブルを表します。
//---------------------------------------------------------------------------
struct KerningSecondTable
{
    ut::ResU16 secondWordNum;           // このテーブルに含まれる二つ目の文字の種類
    KerningSecondTableElem  elems[1];   // 二分探索するための要素の配列。実際にはsecondWordNum個あります。
};

/* ------------------------------------------------------------------------
        ブロック本体定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FontGlyphGroups

  Description:  フォントを部分ロードするための情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontGlyphGroups
{
    ut::ResU32 sheetSize;
    ut::ResU16 glyphsPerSheet;
    ut::ResU16 numSet;
    ut::ResU16 numSheet;
    ut::ResU16 numCWDH;
    ut::ResU16 numCMAP;

    ut::ResU16 nameOffsets[1];  // numSet
/*
    以下は可変長データが続きます

    (4 byte align)

    u32 sizeSheets[numSheet];       // 圧縮済みシートのサイズ
    u32 sizeCWDH[numCWDH];          // CWDH ブロックのサイズ
    u32 sizeCMAP[numCMAP];          // CMAP ブロックのサイズ
    u32 useSheets[numSet][numSheet/32];
    u32 useCWDH[numSet][numCWDH/32];
    u32 useCMAP[numSet][numCMAP/32];

    char names[numSet][name length];
*/
};



/*---------------------------------------------------------------------------*
  Name:         FontTextureGlyph

  Description:  フォントのグリフテクスチャを格納します。
 *---------------------------------------------------------------------------*/
struct FontTextureGlyph
{
    ut::ResU8  cellWidth;              // セル幅=最大グリフ幅
    ut::ResU8  cellHeight;             // セル高さ
    ut::ResU8  sheetNum;               // シート数
    ut::ResU8  maxCharWidth;           // 最大文字幅
    ut::ResU32 sheetSize;              // テクスチャシートデータサイズ(byte単位)
    ut::ResS16 baselinePos;           // ベースライン位置
    ut::ResU16 sheetFormat;            // FontSheetFormat
    ut::ResU16 sheetRow;               // シート内の横方向セル数
    ut::ResU16 sheetLine;              // シート内の縦方向セル数
    ut::ResU16 sheetWidth;             // シート幅(ピクセル単位)
    ut::ResU16 sheetHeight;            // シート高さ(ピクセル単位)
    ut::ResU8* sheetImage;             // テクスチャデータへのポインタ
};



/*---------------------------------------------------------------------------*
  Name:         FontWidth

  Description:  各文字の文字幅情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontWidth
{
    ut::ResU16          indexBegin;     // widthTable の最初のエントリが対応するグリフインデックス
    ut::ResU16          indexEnd;       // widthTable の最後のエントリが対応するグリフインデックス
    FontWidth*          pNext;          // 次の FontWidth へのポインタ
    CharWidths          widthTable[];   // 幅情報の配列
};



/*---------------------------------------------------------------------------*
  Name:         FontCodeMap

  Description:  文字コードから、文字コードに対応するグリフの
                グリフイメージ配列中のインデックスへのマッピングを規定します。
 *---------------------------------------------------------------------------*/
struct FontCodeMap
{
    ut::ResU16          ccodeBegin;     // このブロックが担当する文字コード範囲の最初の文字コード
    ut::ResU16          ccodeEnd;       // このブロックが担当する文字コード範囲の最後の文字コード
    ut::ResU16          mappingMethod;  // マッピング方法 (FontMappingMethod型)
    ut::ResU16          reserved;       // 予約
    FontCodeMap*        pNext;          // 次の FontCodeMap へのポインタ
    ut::ResU16          mapInfo[];      // 文字コードマッピング情報 具体的な内容は mappingMethod に依る
};



/*---------------------------------------------------------------------------*
  Name:         FontKerningTable

  Description:  カーニングの値を格納します。
 *---------------------------------------------------------------------------*/
struct FontKerningTable
{
    ut::ResU16          firstWordNum;       // 最初の文字が何パターンあるか
    KerningFirstTableElem   firstTable[1];  // 二分探索するための要素の配列。実際にはfirstWordNum個あります。
/*
    以下はサイズ可変のデータが続きます。

    (2 byte align)

    KerningSecondTable      secondTables[firstWordNum];
*/
};



/*---------------------------------------------------------------------------*
  Name:         FontInformation

  Description:  フォント全体に渡る情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontInformation
{
    ut::ResU8           fontType;       // グリフデータタイプ (FontType型)
    ut::ResU8           height;         // フォントの高さ
    ut::ResU8           width;          // フォントの幅
    ut::ResU8           ascent;         // アセント
    ut::ResS16          linefeed;       // (*)== leading
    ut::ResU16          alterCharIndex; // (*)フォントに含まれない文字用のグリフのグリフインデックス
    CharWidths          defaultWidth;   // (*)文字幅情報を持たないグリフ用の文字幅情報
    ut::ResU8           characterCode;  // 対応する文字コード (CharacterCode型)
    FontTextureGlyph*   pGlyph;         // 唯一の FontGlyph へのポインタ
    FontWidth*          pWidth;         // 最初の FontWidth へのポインタ
    FontCodeMap*        pMap;           // 最初の FontCodeMap へのポインタ
};



/* ------------------------------------------------------------------------
        ブロック定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         BinaryBlock

  Description:  NintendoWare 標準バイナリブロック
 *---------------------------------------------------------------------------*/
template <typename BlockBodyType>
struct BinaryBlock
{
    ut::BinaryBlockHeader   header;     // ブロックヘッダ
    BlockBodyType           body;       // ブロック本体
};

typedef BinaryBlock<FontGlyphGroups>    FontGlyphGroupsBlock;
typedef BinaryBlock<FontInformation>    FontInformationBlock;
typedef BinaryBlock<FontTextureGlyph>   FontTextureGlyphBlock;
typedef BinaryBlock<FontWidth>          FontWidthBlock;
typedef BinaryBlock<FontCodeMap>        FontCodeMapBlock;



} // namespace font
} // namespace nw


#ifdef _MSC_VER
    #pragma warning( pop )
#endif  //_MSC_VER


#endif //  NW_FONT_RESOURCEFORMAT_H_
