﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_RECTDRAWER_H_
#define NW_FONT_RECTDRAWER_H_

#include <nw/assert.h>
#include <nw/font/font_Font.h>
#include <nw/ut/ut_Range.h>
#include <nw/ut/ut_Color.h>
#include <nw/gfnd/gfnd_DisplayList.h>
#include <nw/gfnd/gfnd_ShaderHelper.h>
#include <nw/font/font_DispStringBuffer.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    #include <gl/glew.h>
#elif defined(NW_PLATFORM_CAFE)
    #include <cafe/gx2.h>
#endif

namespace nw {
namespace font {

//---------------------------------------------------------------------------
//! @brief 描画内容です。
//---------------------------------------------------------------------------
struct DrawContent
{
    //! コンストラクタです。
    DrawContent()
    : dispStringBuffer(NULL)
    , projectionMatrix(NULL)
    , viewMatrix(NULL)
    , localMatrix(NULL)
    , interpolateBlack(0.0f, 0.0f, 0.0f, 0.0f)
    , interpolateWhite(1.0f, 1.0f, 1.0f, 1.0f)
    , shadowInterpolateBlack(0.0f, 0.0f, 0.0f, 0.0f)
    , shadowInterpolateWhite(1.0f, 1.0f, 1.0f, 1.0f)
    , flags(0)
    {}

    //! 白黒補間の黒カラー値とアルファ値を設定します。
    NW_INLINE void SetInterpolateBlackWithAlpha(const ut::Color4u8 color, u8 alpha);

    //! 白黒補間の白カラー値とアルファ値を設定します。
    NW_INLINE void SetInterpolateWhiteWithAlpha(const ut::Color4u8 color, u8 alpha);

    //! 影用の白黒補間の黒カラー値とアルファ値を設定します。
    NW_INLINE void SetShadowInterpolateBlackWithAlpha(const ut::Color4u8 color, u8 alpha);

    //! 影用の白黒補間の白カラー値とアルファ値を設定します。
    NW_INLINE void SetShadowInterpolateWhiteWithAlpha(const ut::Color4u8 color, u8 alpha);

    DispStringBuffer* dispStringBuffer; //!< DispStringBuffer オブジェクトへのポインタです。

    const math::MTX44* projectionMatrix; //!< プロジェクション行列です。
    const math::MTX34* viewMatrix; //!< ビューマトリックスです。
    const math::MTX34* localMatrix; //!< ローカルマトリクスです。
    ut::Color4f interpolateBlack; //!< 黒補完の値です。
    ut::Color4f interpolateWhite; //!< 白補完の値です。
    ut::Color4f shadowInterpolateBlack; //!< 影用の黒補完の値です。
    ut::Color4f shadowInterpolateWhite; //!< 影用の白補完の値です。

    enum Flags
    {
        INVISIBLE_BORDER = 1 << 0,
    };
    u32 flags; //!< フラグです。
};

//---------------------------------------------------------------------------
//! @brief フォント影の設定内容です。
//---------------------------------------------------------------------------
struct ShadowParameter
{
    ut::Color4u8 shadowUpperColor; //!< 影の上端カラー値です。
    ut::Color4u8 shadowLowerColor; //!< 影の下端カラー値です。
    math::VEC2 shadowOffset; //!< 影のオフセット値です。
    math::VEC2 shadowScale; //!< 影のスケール値です。
    f32 shadowItalicOffset; //!< 影のイタリック傾き値です。
};

//---------------------------------------------------------------------------
//! @brief アニメーション情報です。
//---------------------------------------------------------------------------
struct PerCharacterTransformInfo
{
    PerCharacterTransformInfo()
    {
        for (s32 i = 0; i < 3; i++)
        {
            RotationCos[i] = 1.0f;
            RotationSin[i] = 0.0f;
            Translation[i] = 0.0f;
        }
        for (s32 i = 0; i < 4; i++)
        {
            LT[i] = 255;
            LB[i] = 255;
        }
    }

    f32 RotationCos[3]; //!< 回転の余弦成分です。
    f32 RotationSin[3]; //!< 回転の正弦成分です。
    f32 Translation[3]; //!< 平行移動です。
    u8 LT[4]; //!< 上端カラーです。
    u8 LB[4]; //!< 下端カラーです。
};

//---------------------------------------------------------------------------
//! @brief      描画クラスです。
//---------------------------------------------------------------------------
class RectDrawer
{
public:
    enum
    {
        INDEX_NUMBER_BY_LETTER = 4, //!< 文字単位のインデックス数です。
        VERTEX_ATTRIBUTE_NUMBER = 3 //!< 頂点属性の数です。
    };

    //! @brief      RectDrawer が必要なワークバッファのサイズを取得します。
    //!
    //! @param[in] charNum 1度に描画できる文字数です。
    #if defined(NW_PLATFORM_CAFE)
    //! @param[in] shaderBinary シェーダバイナリのアドレスです。
    #endif
    //!
    //! @return     頂点バッファ・コマンドバッファのメモリサイズを返します。
    //!
    #if defined(NW_PLATFORM_CAFE)
    static u32          GetWorkBufferSize(u32 charNum, const ut::MemoryRange& shaderBinary);
    #else
    static u32          GetWorkBufferSize(u32 charNum);
    #endif

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! コンストラクタです。
                        RectDrawer();

    //! @brief      デストラクタです。
    //!
    //! @details    Finalize() を呼び出します。
    //!
    //! @sa         Finalize
    //!
    virtual             ~RectDrawer();

    //@}

    //----------------------------------------
    //! @name 初期化／後処理
    //@{

    //! @brief      初期化を行います。
    //!
    //! @param[in] workBuffer  頂点用バッファ
    //! @param[in] charNum 文字数です。
    #if defined(NW_PLATFORM_CAFE)
    //! @param[in] shaderBinary シェーダバイナリのアドレスです。
    #endif
    //!
    //! @sa         Finalize
    #if defined(NW_PLATFORM_CAFE)
    void Initialize(void* workBuffer, u32 charNum, const ut::MemoryRange& shaderBinary);
    #else
    void Initialize(void* workBuffer, u32 charNum);
    #endif

    //! @brief      開放を行います。
    //!
    //! @sa         Initialize
    virtual void        Finalize();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief      文字列用の頂点配列を作成します。
    //!
    //! @param[in,out] dispStringBuffer DispStringBuffer オブジェクトへのポインタです。
    //! @param[in] shadowParam フォント影の設定内容です。
    //! @param[in] perCharacterTransformInfos 文字ごとのアニメーション情報です。
    //! @param[in] perCharacterTransformCenter アニメーションの回転原点です。
    //!
    void BuildVertexElements(
        DispStringBuffer* dispStringBuffer,
        const ShadowParameter* shadowParam = NULL,
        const PerCharacterTransformInfo* perCharacterTransformInfos = NULL,
        u8 perCharacterTransformCenter = 0);

    //! @brief        描画を行います。
    //!
    //! @param[in]  content 描画内容です。
    //!
    void Draw(const DrawContent& content);

    //@}

    //----------------------------------------
    //! @name 取得／設定
    //@{

    //! @brief      初期化時に与えたバッファを取得します。
    //!
    //! @return     バッファです。
    //!
    void*               GetBuffer() { return static_cast<void*>(m_IndexBuffer); }

    //@}

protected:

    //! @brief        テクスチャ使用情報を構築します。
    //!
    //! @param[in]    pDispStringBuffer 表示文字情報です。
    //! @param[in]    shadowEnabled 影が有効な場合に true とします。
    //!
    void                BuildTextureUseInfos(
        DispStringBuffer* pDispStringBuffer,
        bool shadowEnabled);

    //! @brief      グリフデータの持つテクスチャをロードします。
    //!
    //! @param[in]  pTexObj ロードすべきテクスチャ情報を持つグリフデータ。
    //!
    void                LoadTexture(const internal::TextureObject* pTexObj);

    //! @brief      インデックス生成を行います。
    //!
    //! @param[in]  indices インデックスデータを格納するバッファです。
    //! @param[in]  charNum 文字数です。
    //!
    static void        CreateIndices(u16* indices, u32 charNum);

    static const u32 DISPLAY_LIST_BUFFER_SIZE = 512;
    enum BitFlags
    {
        COLOR_BLACK_WHITE_INTERPOLATION_ENABLED = (1 << 0),
        BORDER_EFFECT_ENABLED = (1 << 1),
    };
    enum ShaderVariation
    {
        NORMAL_SHADER,
        BORDER_SHADER,
        INVISIBLE_BORDER_SHADER,
        SHADER_QUANTITY
    };

    struct Shader
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        GLuint m_VertexShader; //!< 頂点シェーダです。
        GLuint m_FragmentShader; //!< フラグメントシェーダです。
        GLuint m_ShaderProgram; //!< プログラムです。
        GLuint m_AttrVertex; //!< 頂点属性です。
        GLuint m_AttrColor; //!< フォント色です。
        GLuint m_AttrTexCoord0; //!< テクスチャ座標です。
#elif defined(NW_PLATFORM_CAFE)
        GX2VertexShader* m_VertexShader;
        GX2PixelShader* m_PixelShader;
        GX2FetchShader m_FetchShader;
        void* m_FetchShaderBuffer;
        u32 m_ParamTexture;
        u32 m_AttributePosition;
        u32 m_AttributeColor;
        u32 m_AttributeTexCoord0;
        GX2AttribStream m_Attributes[VERTEX_ATTRIBUTE_NUMBER];
        gfnd::DisplayList m_DisplayList;
#endif

        gfnd::ShaderUniformId m_ParamUser;
        gfnd::ShaderUniformId m_ParamInterpolateOffset; //!< 白黒補完オフセットのユニフォームロケーションです。
        gfnd::ShaderUniformId m_ParamInterpolateWidth; //!< 白黒補完幅のユニフォームロケーションです。
        gfnd::ShaderUniformId m_ParamShadowInterpolateOffset; //!< 影用の白黒補完オフセットのユニフォームロケーションです。
        gfnd::ShaderUniformId m_ParamShadowInterpolateWidth; //!< 影用の白黒補完幅のユニフォームロケーションです。

        Shader();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        void* Initialize(void* workBuffer, int idx);
        void Finalize();

        void Activate();

        bool CreateShader(GLuint shader, const char* source);

#elif defined(NW_PLATFORM_CAFE)
        void* Initialize(void* workBuffer, int idx, const ut::MemoryRange& shaderBinary);
        void Finalize();

        void Activate();

        //! @brief 頂点シェーダを解析して生成します。
        //!
        //! @param[out] ppShader 生成されたシェーダを返します。
        //! @param[in] index シェーダのインデックス番号です。
        //! @param[in] data シェーダバイナリのデータです。
        //! @param[in] buffer バッファの先頭です。
        //!
        //! @return 使用したバッファの末尾を返します。生成できない場合 NULL を返します。
        //!
        void* ReadVertexShader(GX2VertexShader** ppShader, u32 index, const void* data, void* buffer);

        //! @brief ピクセルシェーダシェーダを解析して生成します。
        //!
        //! @param[out] ppShader 生成されたシェーダを返します。
        //! @param[in] index シェーダのインデックス番号です。
        //! @param[in] data シェーダバイナリのデータです。
        //! @param[in] buffer バッファの先頭です。
        //!
        //! @return 使用したバッファの末尾を返します。生成できない場合 NULL を返します。
        //!
        void* ReadPixelShader(GX2PixelShader** ppShader, u32 index, const void* data, void* buffer);
#endif
    };

    u32 m_MaxCharNum;
    Shader m_Shader[SHADER_QUANTITY];
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    GLuint m_Sampler;
#elif defined(NW_PLATFORM_CAFE)
    GX2Sampler m_Sampler;
#endif
    u16* m_IndexBuffer; //!< インデックス配列

};

//----------------------------------------
NW_INLINE void DrawContent::SetInterpolateWhiteWithAlpha(const ut::Color4u8 color, u8 alpha)
{
    const f32 floatAlphaMax = ut::Color4u8::ALPHA_MAX;

    interpolateWhite.r = color.r / floatAlphaMax;
    interpolateWhite.g = color.g / floatAlphaMax;
    interpolateWhite.b = color.b / floatAlphaMax;
    interpolateWhite.a = (color.a / floatAlphaMax) * (alpha / floatAlphaMax);
}

//----------------------------------------
NW_INLINE void DrawContent::SetInterpolateBlackWithAlpha(const ut::Color4u8 color, u8 alpha)
{
    const f32 floatAlphaMax = ut::Color4u8::ALPHA_MAX;

    interpolateBlack.r = color.r / floatAlphaMax;
    interpolateBlack.g = color.g / floatAlphaMax;
    interpolateBlack.b = color.b / floatAlphaMax;
    // 黒補完のアルファは使用しない。グローバルアルファをそのまま入れる
    interpolateBlack.a = alpha / floatAlphaMax;
}

//----------------------------------------
NW_INLINE void DrawContent::SetShadowInterpolateWhiteWithAlpha(const ut::Color4u8 color, u8 alpha)
{
    const f32 floatAlphaMax = ut::Color4u8::ALPHA_MAX;

    shadowInterpolateWhite.r = color.r / floatAlphaMax;
    shadowInterpolateWhite.g = color.g / floatAlphaMax;
    shadowInterpolateWhite.b = color.b / floatAlphaMax;
    shadowInterpolateWhite.a = (color.a / floatAlphaMax) * (alpha / floatAlphaMax);
}

//----------------------------------------
NW_INLINE void DrawContent::SetShadowInterpolateBlackWithAlpha(const ut::Color4u8 color, u8 alpha)
{
    const f32 floatAlphaMax = ut::Color4u8::ALPHA_MAX;

    shadowInterpolateBlack.r = color.r / floatAlphaMax;
    shadowInterpolateBlack.g = color.g / floatAlphaMax;
    shadowInterpolateBlack.b = color.b / floatAlphaMax;
    // 黒補完のアルファは使用しない。グローバルアルファをそのまま入れる
    shadowInterpolateBlack.a = alpha / floatAlphaMax;
}

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_RECTDRAWER_H_
