﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eftvw2_CmdReceiver.h>
#include <nw/eft/eftvw2_CmdSender.h>
#include <nw/eft/eftvw2_PacketProcedure.h>
#include <nw/eft/eftvw2_ResourceManager.h>
#include <nw/eft/eftvw2_EffectPreview.h>
#include <nw/eft/eftvw2_TextWriter.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nw/eft/eftvw2_ToolConnecter.h>

namespace nw     {
namespace eftvw2 {

//---------------------------------------------------------------------------
//! @brief        エフェクトビューアがモデル情報を取得するためのインターフェース。
//---------------------------------------------------------------------------
class IEftModelEnumerator
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    IEftModelEnumerator() {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~IEftModelEnumerator() {}

public:
    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual s32			         GetNumModel() = 0;

    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @param[in] ix     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual const char*          GetModelName( s32 ix ) = 0;

    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @param[in] modelName     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual s32                  GetNumBone ( const char* modelName ) = 0;

    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @param[in] modelName     TBD
    //! @param[in] ix     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual const char*          GetBoneName( const char* modelName , s32 ix ) = 0;

    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @param[in] boneMatrix     TBD
    //! @param[in] modelName     TBD
    //! @param[in] ix     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual void GetBoneMtx ( nw::math::Matrix34* boneMatrix, const char* modelName , s32 ix ) = 0;

    //---------------------------------------------------------------------------
    //! @brief        TBD
    //! @param[in] rootMatrix     TBD
    //! @param[in] modelName     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual void GetModelRootMtx( nw::math::Matrix34* rootMatrix, const char* modelName ) = 0;
};

//---------------------------------------------------
//! @brief  モデルプレビュー生成コールバックの定義です。
//! @param[in] heap     TBD
//! @param[in] viewerSystem     TBD
//! @param[in] binary     TBD
//! @param[in] binarySize     TBD
//! @param[in] guid     TBD
//---------------------------------------------------
typedef nw::eftvw2::Preview* (*CreateModelPreviewCallback)( nw::eft2::Heap* heap, nw::eftvw2::ViewerSystem* viewerSystem, void* binary, u32 binarySize, nw::eftcom::Guid guid );

//---------------------------------------------------
//! @brief  モデルプレビュー破棄コールバックの定義です。
//! @param[in] heap     TBD
//! @param[in] preview     TBD
//---------------------------------------------------
typedef void (*DestroyModelPreviewCallback)( nw::eft2::Heap* heap, nw::eftvw2::Preview* preview );

//---------------------------------------------------------------------------
//! @brief        エフェクトビューアシステムです。
//---------------------------------------------------------------------------
class ViewerSystem
{
public:
    //---------------------------------------------------------------------------
    //! @brief  ビューア定数。
    //---------------------------------------------------------------------------
    enum
    {
        EFT_VWR_BIND_ESET_MAX = 16      //!< リソースバインド可能な最大数
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //! @param[in] heap     TBD
    //! @param[in] connectionHeap     TBD
    //! @param[in] system     TBD
    //---------------------------------------------------------------------------
#if EFT_IS_WIN
    explicit ViewerSystem( nw::eft2::Heap* heap, nw::eft2::Heap* connectionHeap, nw::eft2::System* system, bool noUseInternalThread = false, int priority = THREAD_PRIORITY_LOWEST );
#else
    explicit ViewerSystem( nw::eft2::Heap* heap, nw::eft2::Heap* connectionHeap, nw::eft2::System* system, bool noUseInternalThread = false, int priority = 16 );
#endif

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~ViewerSystem();

    //---------------------------------------------------------------------------
    //! @brief        アプリケーションの初期化完了を通知します。
    //---------------------------------------------------------------------------
    void SetAppIsInitialized();

    //---------------------------------------------------------------------------
    //! @brief      エフェクトメーカーと接続済みかどうかを取得します。
    //! @return 接続済みかどうか
    //---------------------------------------------------------------------------
    bool IsConnected()
    {
        return mIsConnected;
    }

    //---------------------------------------------------------------------------
    //! @brief        再生します。
    //---------------------------------------------------------------------------
    void Play()     { mIsPaused = false; }

    //---------------------------------------------------------------------------
    //! @brief        一時停止します。
    //---------------------------------------------------------------------------
    void Pause()    { mIsPaused = true; }

    //---------------------------------------------------------------------------
    //! @brief        一時停止中かチェックします。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsPause()  { return mIsPaused; }

    //---------------------------------------------------------------------------
    //! @brief        停止します。
    //---------------------------------------------------------------------------
    void Stop()
    {
        ResetTime( false );
        mIsPaused = true;
    }

    //---------------------------------------------------------------------------
    //! @brief        コマ送りします。
    //---------------------------------------------------------------------------
    void StepFrame() { mStep = true; }

    //---------------------------------------------------------------------------
    //! @brief        強制的にフェードさせます。
    //---------------------------------------------------------------------------
    void ForceFade();

    //---------------------------------------------------------------------------
    //! @brief        定期処理を実行します。
    //! @param[in] frameRate     TBD
    //! @param[in] view     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool ProcCalc( f32 frameRate, const nw::math::MTX34& view );

    //---------------------------------------------------------------------------
    //! @brief        コマンド処理を実行します。
    //---------------------------------------------------------------------------
    void ExecuteCommand();

    //------------------------------------------------------------------------------
    //! @brief        タイムを0に戻します。
    //! @param[in] bFade     TBD
    //------------------------------------------------------------------------------
    void ResetTime( bool bFade );

    //------------------------------------------------------------------------------
    //! @brief        タイムを取得します。
    //! @return TBD
    //------------------------------------------------------------------------------
    f32 GetTime() const { return mTime; }

    //------------------------------------------------------------------------------
    //! @brief        終了タイムを取得します。
    //! @return TBD
    //------------------------------------------------------------------------------
    f32 GetEndTime() const { return mEndTime; }

    //---------------------------------------------------------------------------
    //! @brief        ループしているかを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsLoop() const { return mIsLoop; }

    //---------------------------------------------------------------------------
    //! @brief        再生範囲とループの設定をします。
    //! @param[in] startFrame     TBD
    //! @param[in] endFrame     TBD
    //! @param[in] isLoop     TBD
    //---------------------------------------------------------------------------
    void SetFrameRange( f32 startFrame, f32 endFrame, bool isLoop )
    {
        EFT_UNUSED_VARIABLE( startFrame );
        //mStartFrame     = startFrame;
        mEndTime       = endFrame;
        mIsLoop         = isLoop;
    }

    //---------------------------------------------------------------------------
    //! @brief        リソースマネージャを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    ResourceManager* GetResManager() { return &mResManager; }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムのポインタを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    nw::eft2::System* GetEftSystem() { return mEftSystem; }

    //---------------------------------------------------------------------------
    //! @brief        ToolReceiverスレッドのポインタを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    void* GetToolReceiver() {  return (void*)(&ToolConnecter::threadToolReceiver); }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットバイナリの読み込みを行います。
    //! @param[in] pPath     TBD
    //! @param[in] guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool LoadEffectData( const char* pPath, eftcom::Guid guid );

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットバイナリの破棄を行います。
    //! @param[in] guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool UnLoadEffectData( eftcom::Guid guid );

    //---------------------------------------------------------------------------
    //! @brief エフェクトプレビューを生成します。
    //! @param[in] resGuid     TBD
    //! @param[in] prevGuid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CreateEffectPreview( eftcom::Guid resGuid, eftcom::Guid prevGuid );

    //---------------------------------------------------------------------------
    //! @brief ビューアデータのコールバックを設定します。
    //! @param[in] pThis     TBD
    //! @param[in] callback     TBD
    //---------------------------------------------------------------------------
    void SetViewerDataCallBack( void* pThis, ViewerDataCallback callback )
    {
        mPacketProcedure.SetViewerDataCallback( pThis, callback );
    }

    //---------------------------------------------------------------------------
    //! @brief ビューアメッセージのコールバックを設定します。
    //! @param[in] pThis     TBD
    //! @param[in] callback     TBD
    //---------------------------------------------------------------------------
    void SetViewerMessageCallBack( void* pThis, ViewerMessageCallback callback )
    {
        mPacketProcedure.SetViewerMessageCallback( pThis, callback );
    }
    //---------------------------------------------------------------------------
    //! @brief モデルプレビュー生成コールバックを設定します。
    //! @param[in] callback     TBD
    //---------------------------------------------------------------------------
    void SetCreateModelPreviewCallback( CreateModelPreviewCallback callback )
    {
        mCreateModelPreviewCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief モデルプレビュー破棄コールバックを設定します。
    //! @param[in] callback     TBD
    //---------------------------------------------------------------------------
    void SetDestroyModelPreviewCallback( DestroyModelPreviewCallback callback )
    {
        mDestroyModelPreviewCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief モデルプレビューを生成します。
    //! @param[in] guid     TBD
    //! @param[in] pData     TBD
    //! @param[in] dataSize     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CreateModelPreview( eftcom::Guid guid, void* pData, u32 dataSize );

    //---------------------------------------------------------------------------
    //! @brief モデルプレビューを破棄します。
    //! @param[in] guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool DestroyModelPreview( eftcom::Guid guid);

    //---------------------------------------------------------------------------
    //! @brief エフェクトプレビューリソースをエフェクトプレビューに設定します。
    //! @param[in] prevGuid     TBD
    //! @param[in] resource     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool SetResEffectPreview( eftcom::Guid prevGuid, ResEffectPreviewPack* resource );

    //---------------------------------------------------------------------------
    //! @brief モデルプレビューリソースをモデルプレビューに設定します。
    //! @param[in] prevGuid     TBD
    //! @param[in] resource     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool SetResModelPreview( eftcom::Guid prevGuid, ResModelPreviewPack* resource );

    //---------------------------------------------------------------------------
    //! @brief  指定リソースをバインドします。
    //! @param[in] resGuid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindResource( eftcom::Guid resGuid );

    //---------------------------------------------------------------------------
    //! @brief  指定リソースをアンバインドします。
    //! @param[in] resGuid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool UnbindResource( eftcom::Guid resGuid );

    //---------------------------------------------------------------------------
    //! @brief  指定リソースをアンバインドします。
    //! @param[in] resID     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool UnbindResource( s32 resID );

    //---------------------------------------------------------------------------
    //! @brief エフェクトプレビューリソースを更新します。
    //! @param[in] prevGuid     TBD
    //! @param[in] offset     TBD
    //! @param[in] overrideData     TBD
    //! @param[in] overrideDataSize     TBD
    //! @param[in] endian     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool UpdateResEffectPreview( eftcom::Guid   prevGuid,
                                 u32            offset,
                                 void*          overrideData,
                                 u32            overrideDataSize,
                                 u8             endian );

    //---------------------------------------------------------------------------
    //! @brief モデルプレビューリソースを更新します。
    //! @param[in] prevGuid     TBD
    //! @param[in] offset     TBD
    //! @param[in] overrideData     TBD
    //! @param[in] overrideDataSize     TBD
    //! @param[in] endian     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool UpdateResModelPreview( eftcom::Guid   prevGuid,
                                u32            offset,
                                void*          overrideData,
                                u32            overrideDataSize,
                                u8             endian );

    //------------------------------------------------
    //! @brief  指定されたemitterの変数を更新します。
    //! @param[in] resGuid     TBD
    //! @param[in] emitterName     TBD
    //! @param[in] offset     TBD
    //! @param[in] overrideData     TBD
    //! @param[in] overrideDataSize     TBD
    //! @param[in] withReset     TBD
    //! @param[in] endian     TBD
    //------------------------------------------------
    void UpdateAssetEmitter( eftcom::Guid resGuid,
                             const char* emitterName, u32 offset,
                             void* overrideData,      u32 overrideDataSize,
                             const bool withReset,    u8  endian );

    //------------------------------------------------
    //! @brief  指定されたエミッタセットのビジビリティを設定します。
    //! @param[in] resGuid     TBD
    //! @param[in] visible     TBD
    //------------------------------------------------
    void SetEmitterSetVisibility( eftcom::Guid resGuid, bool visible );

    //------------------------------------------------
    //! @brief  指定されたエミッタセットのビジビリティを設定します。
    //! @param[in] resGuid     TBD
    //! @param[in] emitterName     TBD
    //! @param[in] visible     TBD
    //------------------------------------------------
    void SetEmitterVisibility( eftcom::Guid resGuid, const char* emitterName, bool visible );

    //------------------------------------------------
    //! @brief  指定されたプレビューのビジビリティを設定します。
    //! @param[in] prevGuid     TBD
    //! @param[in] visible     TBD
    //------------------------------------------------
    void SetPreviewVisibility( eftcom::Guid prevGuid, bool visible );

    //---------------------------------------------------------------------------
    //! @brief  プレビューをリストに追加します。
    //! @param[in] preview     TBD
    //---------------------------------------------------------------------------
    void AddPreview( Preview* preview )
    {
        mPreviewHead.AddPreview(preview);
    }

    //---------------------------------------------------------------------------
    //! @brief  プレビューをリストに追加します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Preview* GetPreviewHead()
    {
        return &mPreviewHead;
    }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットを作成します。
    //! @param[in] emitterSetName     TBD
    //---------------------------------------------------------------------------
    void CreateViewSysEmitterSet( const char* emitterSetName );

    //---------------------------------------------------------------------------
    //! @brief  全てのプレビューを破棄します。
    //---------------------------------------------------------------------------
    void DestroyAllPreview();

    //---------------------------------------------------------------------------
    //! @brief  全てのリソースを破棄します。
    //---------------------------------------------------------------------------
    void DestroyAllResource();

    //---------------------------------------------------------------------------
    //! @brief  ビューアセンターを設定します。
    //! @param[in] center     TBD
    //---------------------------------------------------------------------------
    void SetViewerCenter( nw::math::MTX34& center )
    {
        mViewerCenter = center;
    }

    //---------------------------------------------------------------------------
    //! @brief  ビューアセンターを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const nw::math::MTX34& GetViewerCenter() const { return mViewerCenter; }

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 共通処理
    //! @param[in] type     TBD
    //! @param[in] emitterSetName     TBD
    //! @param[in] duplicateEmitterSetName     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestEset(  u32 type, const char* emitterSetName, const char* duplicateEmitterSetName = NULL );

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 エミッタセットファイルオープン
    //! @param[in] emitterSetFilePath     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestEsetFileOpen( const char* emitterSetFilePath );

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 エミッタセットファイルクローズ
    //! @param[in] emitterSetFilePath     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestEsetFileClose( const char* emitterSetFilePath );

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 新規エミッタセット作成
    //! @param[in] emitterSetFilePath     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestEsetFileCreate( const char* emitterSetFilePath );

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 エミッタセット複製
    //! @param[in] srcEmitterSetName     TBD
    //! @param[in] duplicateEmitterSetName     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestEsetFileDuplicate( const char* srcEmitterSetName, const char* duplicateEmitterSetName );

/*
    ～メッセージ未実装 ここから～
*/
    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 カスタムシェーダのパスを設定する
    //! @param[in] customShaderPath     TBD
    //! @param[in] forceAssign     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestSetCustomShaderPath( const char* customShaderPath, bool forceAssign )
    {
        EFT_UNUSED_VARIABLE( customShaderPath );
        EFT_UNUSED_VARIABLE( forceAssign );
    }

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 カスタムアクションのパスを設定する
    //! @param[in] customActionPath     TBD
    //! @param[in] forceAssign     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestSetCustomActionPath( const char* customActionPath, bool forceAssign )
    {
        EFT_UNUSED_VARIABLE( customActionPath );
        EFT_UNUSED_VARIABLE( forceAssign );
    }

    //-----------------------------------------------------------------------------------------------------
    //! @brief		エフェクトメーカー操作 ユーザーコンフィグのパスを設定する
    //! @param[in] userConfigPath     TBD
    //! @param[in] forceAssign     TBD
    //-----------------------------------------------------------------------------------------------------
    void RequestSetUserConfigPath( const char* userConfigPath, bool forceAssign )
    {
        EFT_UNUSED_VARIABLE( userConfigPath );
        EFT_UNUSED_VARIABLE( forceAssign );
    }


    //---------------------------------------------------------------------------
    //! @brief        現在のビューアの状況をデバッグ表示します。
    //! @param[in] textWriter     TBD
    //! @param[in] posX     TBD
    //! @param[in] posY     TBD
    //! @param[in] spaceY     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 DrawDebugInfo( nw::eftvw2::TexWriter* textWriter, f32 posX, f32 posY, f32 spaceY )
    {
        EFT_UNUSED_VARIABLE( textWriter );
        EFT_UNUSED_VARIABLE( posX );
        EFT_UNUSED_VARIABLE( posY );
        EFT_UNUSED_VARIABLE( spaceY );
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        カスタムシェーダパスを切り替える。
    //! @param[in] filepath     TBD
    //---------------------------------------------------------------------------
    void SetCustomShaderPath( const char* filepath )
    {
        return SetCustomShaderDefinitionFile( filepath );
    }

    //---------------------------------------------------------------------------
    //! @brief        カスタムシェーダ定義ファイルパスを切り替える。
    //! @param[in] filepath     TBD
    //---------------------------------------------------------------------------
    void SetCustomShaderDefinitionFile( const char* filepath )
    {
        EFT_UNUSED_VARIABLE( filepath );
#if 0
#if EFT_IS_CAFE
        strcpy( mCustomShaderPath, filepath );
#endif
#if EFT_IS_WIN
        strcpy_s( mCustomShaderPath, ProtocolEM4F::NString::MAX_STRING, filepath );
#endif
        mCustomShaderPathChanged = true;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        カスタムアクションパスを切り替える。
    //! @param[in] filepath     TBD
    //---------------------------------------------------------------------------
    void SetCustomActionPath( const char* filepath )
    {
        EFT_UNUSED_VARIABLE( filepath );
#if 0
#if EFT_IS_CAFE
        strcpy( mCustomActionPath, filepath );
#endif
#if EFT_IS_WIN
        strcpy_s( mCustomActionPath, ProtocolEM4F::NString::MAX_STRING, filepath );
#endif
        mCustomActionPathChanged = true;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ユーザーコンフィグファイルを切り替える。
    //! @param[in] filepath     TBD
    //---------------------------------------------------------------------------
    void SetUserConfigPath( const char* filepath )
    {
        return SetUserConfigFile( filepath );
    }

    //---------------------------------------------------------------------------
    //! @brief        ユーザーコンフィグファイルを切り替える。
    //! @param[in] filepath     TBD
    //---------------------------------------------------------------------------
    void SetUserConfigFile( const char* filepath )
    {
        EFT_UNUSED_VARIABLE( filepath );
#if 0
#if EFT_IS_CAFE
        strcpy( mUserConfigPath, userConfigPath );
#endif
#if EFT_IS_WIN
        strcpy_s( mUserConfigPath, ProtocolEM4F::NString::MAX_STRING, filepath );
#endif
        mUserConfigPathChanged = true;
#endif
    }
/*
    ～メッセージ未実装 ここまで～
*/

    //---------------------------------------------------------------------------
    //! @brief      モデル情報更新
    //! @param[in] modelEnumrator     TBD
    //---------------------------------------------------------------------------
    void UpdateModelInfo( IEftModelEnumerator* modelEnumrator );

    //---------------------------------------------------------------------------
    //! @brief        ClearPreviewModelV2E を送信する。
    //---------------------------------------------------------------------------
    void SendClearPreviewModelV2E()
    {
        nw::eftcom::ModelHeaderMessage header;
        header.type = nw::eftcom::MODEL_MESSAGE_TYPE_CLEAR_PREVIEWMODEL;
        header.size = 0;

        nw::eftvw2::CmdSender::gCmdSender->SendCommand(
            nw::eftcom::MESSAGE_TYPE_MODEL_INFO,
            &header, sizeof(nw::eftcom::ModelHeaderMessage) );
    }

    //---------------------------------------------------------------------------
    //! @brief        アプリケーション初期化完了メッセージ  を送信する。
    //---------------------------------------------------------------------------
    void SendAppIsInitialized()
    {
        nw::eftcom::ViewerHeaderMessage msg;
        msg.type = nw::eftcom::VIEWER_MESSAGE_TYPE_APP_INITIALIZED;
        msg.size = 0;

        nw::eftvw2::CmdSender::gCmdSender->SendCommand(
            nw::eftcom::MESSAGE_TYPE_SEND_BINARY,
            &msg, sizeof(nw::eftcom::ViewerHeaderMessage) );
    }

    //---------------------------------------------------------------------------
    //! @brief        BeginSendModelInfoV2E  を送信する。
    //---------------------------------------------------------------------------
    void SendBeginPreviewModelV2E()
    {
        nw::eftcom::ModelHeaderMessage header;
        header.type = nw::eftcom::MODEL_MESSAGE_TYPE_BEGIN_SENDMODELINFO;
        header.size = 0;

        nw::eftvw2::CmdSender::gCmdSender->SendCommand(
            nw::eftcom::MESSAGE_TYPE_MODEL_INFO,
            &header, sizeof(nw::eftcom::ModelHeaderMessage) );
    }

    //---------------------------------------------------------------------------
    //! @brief        SendModelInfoV2E  を送信する。
    //! @param[in] sendPacket     TBD
    //! @param[in] packetSize     TBD
    //---------------------------------------------------------------------------
    void SendModelInfoV2E( void* sendPacket, u32 packetSize )
    {
        EFT_UNUSED_VARIABLE( sendPacket );
        EFT_UNUSED_VARIABLE( packetSize );
        nw::eftvw2::CmdSender::gCmdSender->SendCommand(
            nw::eftcom::MESSAGE_TYPE_MODEL_INFO, sendPacket, packetSize );
    }

    //---------------------------------------------------------------------------
    //! @brief        EndSendModelInfoV2E  を送信する。
    //---------------------------------------------------------------------------
    void SendEndPreviewModelV2E()
    {
        nw::eftcom::ModelHeaderMessage header;
        header.type = nw::eftcom::MODEL_MESSAGE_TYPE_END_SEND_MODELINFO;
        header.size = 0;

        nw::eftvw2::CmdSender::gCmdSender->SendCommand(
            nw::eftcom::MESSAGE_TYPE_MODEL_INFO,
            &header, sizeof(nw::eftcom::ModelHeaderMessage) );
    }

    //---------------------------------------------------------------------------
    //! @brief        SendLinkedEmitterSetV2E  を送信する。
    //! @param[in] emitterSetName     TBD
    //---------------------------------------------------------------------------
    void SendLinkedEmitterSetV2E( const char* emitterSetName );

    //---------------------------------------------------------------------------
    //! @brief        SendUnLinkedEmitterSetV2E  を送信する。
    //! @param[in] emitterSetName     TBD
    //---------------------------------------------------------------------------
    void SendUnLinkedEmitterSetV2E( const char* emitterSetName );


private:
    //---------------------------------------------------------------------------
    //! @brief      指定リソースIDのエフェクトプレビューを削除。
    //! @param[in] resID     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool RemoveEffectPreviewFromResID( s32 resID );


#if 0 // Test用Send Dummy ModelData
    void UploadDummyModelDataInfo();
    void UploadDummyToolModelDataInfo();
#endif

private:
    nw::eft2::Heap*                 mViewerHeap;        //!< ビューアヒープ
    nw::eft2::Heap*                 mConnectionHeap;    //!< コネクションヒープ
    PacketProcedure                 mPacketProcedure;   //!< TBD
    CmdReceiver*                    mCmdReceiver;       //!< EffectMakerからメッセージを受信する為のクラス
    CmdSender*                      mCmdSender;         //!< EffectMakerへメッセージを送信する為のクラス
    nw::eft2::System*               mEftSystem;         //!< ビューアが再生に利用するエフェクトシステム
    ResourceManager                 mResManager;        //!< リソースマネージャ
    Preview                         mPreviewHead;       //!< プレビューの先頭
    f32                             mTime;              //!< ビューアシステムのタイム
    f32                             mFrameRate;         //!< 再生フレームレート
    f32                             mEndTime;           //!< ループ再生中の終了フレーム
    bool                            mIsLoop;            //!< ループ再生中かどうか。
    bool                            mIsPaused;          //!< ポーズ中かどうか。
    bool                            mStep;              //!< コマ送り
    bool                            mIsAppInitialized;  //!< アプリケーションが初期化済みかどうか
    bool                            mIsConnected;       //!< 接続中かどうか
    bool                            mIsResetTime;       //!< タイムをリセットするかどうか
    nw::math::MTX34                 mViewerCenter;      //!< ビューアのセンター

    IEftModelEnumerator*            mModelEnumrator;    //!< モデル情報取得の為のインターフェースクラス

    CreateModelPreviewCallback      mCreateModelPreviewCallback;   //!< モデルプレビュー生成コールバック
    DestroyModelPreviewCallback     mDestroyModelPreviewCallback;  //!< モデルプレビュー破棄コールバック

#if EFT_IS_CAFE
    nw::ut::IAllocator*             mFileSystemAllocator;   //!< ファイルシステム用アロケータ
#endif
};

} // namespace eftvw2
} // namespace nw
