﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eftvw2_Preview.h>
#include <nw/ut/ut_Color.h>

namespace nw     {
namespace eftvw2 {

//! @brief エフェクトプレビューが利用するグループID
enum
{
    EFT_VWR_USE_GROUP_ID = ( nw::eft2::EFT_GROUP_MAX - 1 ), //!< TBD
};

//---------------------------------------------------------------------------
//! @brief  エフェクトプレビュークラスです。
//---------------------------------------------------------------------------
class EffectPreview : public Preview
{
public:

    //----------------------------------------
    //! @name 生成/破棄 処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief  EffectPreview クラスのインスタンスを生成します。
    //! @param[in] heap     TBD
    //! @param[in] system     TBD
    //! @param[in] guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    static EffectPreview* CreateEffectPreview( nw::eft2::Heap* heap, nw::eft2::System* system, eftcom::Guid guid );

    //---------------------------------------------------------------------------
    //! @brief  EffectPreview クラスのインスタンスを破棄します。
    //! @param[in] eftHeap     TBD
    //! @param[in] preview     TBD
    //---------------------------------------------------------------------------
    static void DestroyEffectPreview( nw::eft2::Heap* eftHeap, Preview* preview );

    //@}


    //----------------------------------------
    //! @name 取得/設定 処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エフェクトプレビューリソースを設定します。
    //!
    //! @param[in]    resource エフェクトプレビューリソース。
    //---------------------------------------------------------------------------
    void SetResEffectPreview( const ResEffectPreview* resource )
    {
        mResEffectPreview = *resource;
        mIsSetResEffectPreview = true;
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトプレビューリソースを取得します。
    //!
    //! @return       エフェクトプレビューリソースを返します。
    //---------------------------------------------------------------------------
    ResEffectPreview* GetResEffectPreview() { return &mResEffectPreview; }

    //@}


    //----------------------------------------
    //! @name 定期 処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief        計算処理です。
    //! @param[in] bPause     TBD
    //! @param[in] frameRate     TBD
    //! @param[in] centerMatrix     TBD
    //! @param[in] viewMatrix     TBD
    //---------------------------------------------------------------------------
    void Calc( bool bPause, f32 frameRate, const nw::math::MTX34& centerMatrix, const nw::math::MTX34& viewMatrix );

    //@}

    //----------------------------------------
    //! @name エミッタセット操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットを生成します。
    //! @param[in] resId     TBD
    //! @param[in] startFrame     TBD
    //! @param[in] forceCalc     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CreateEmitterSet( s32 resId, u32 startFrame, u32 forceCalc );

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットを破棄します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool DestroyEmitterSet();

    //---------------------------------------------------------------------------
    //! @brief  指定フレーム再生位を進めます。
    //! @param[in] frame     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool ForceCalc( u32 frame );

    //---------------------------------------------------------------------------
    //! @brief  再生中のエミッタセットに対してフェードを呼び出します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Fade();

    //---------------------------------------------------------------------------
    //! @brief  プレビュー再生をリセットします。
    //! @param[in] bFade     TBD
    //---------------------------------------------------------------------------
    void ResetPreview( bool bFade );

    //---------------------------------------------------------------------------
    //! @brief  マトリクスを取得します。
    //! @param[in] matrix     TBD
    //! @param[in] idx     TBD
    //---------------------------------------------------------------------------
    void GetMatrix( nw::math::MTX34* matrix, s32 idx )
    {
        EFT_UNUSED_VARIABLE( idx );
        *matrix = mDrawMatrix;
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットハンドルを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    nw::eft2::Handle* GetEmitterSetHandle() { return &mCurrentHandle; }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトリソースIDを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    s32 GetResourceID() { return mResId; }
    s32 GetResourceId() { return mResId; }

    //---------------------------------------------------------------------------
    //! @brief        描画のOn/Offを設定します。
    //! @param[in] visible     TBD
    //---------------------------------------------------------------------------
    void SetVisible( bool visible );

    //@}

protected:
    ResEffectPreview mResEffectPreview;        //!< エフェクトプレビューリソース
    bool             mIsSetResEffectPreview;   //!< エフェクトプレビューリソースがセットされたか？

private:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    EffectPreview();

    //---------------------------------------------------------------------------
    //! @brief  エフェクトプレビューを初期化します。
    //! @param[in] system     TBD
    //---------------------------------------------------------------------------
    void InitializeEffectPreview( nw::eft2::System* system );

    //---------------------------------------------------------------------------
    //! @brief        描画カラーを更新する。
    //---------------------------------------------------------------------------
    void updateDrawColor();

private:
    //---------------------------------------------------------------------------
    //! @brief TBD
    //---------------------------------------------------------------------------
    enum
    {
        EFT_PREV_FADE_CTRL_MAX  = 32,       //!< Fade処理中のエミッタセット最大保持数
    };

    nw::eft2::System*       mSystem;                                            //!< TBD
    nw::eft2::Handle        mCurrentHandle;                                     //!< TBD
    s32                     mResId;                                             //!< TBD
    u32                     mWiatFrame;                                         //!< TBD
    u32                     mForceCalcFrame;                                    //!< TBD
    nw::math::MTX34         mCenter;                                            //!< TBD
    nw::math::MTX34         mOffsetMatrix;                                      //!< TBD
    nw::ut::Color4f         mFunctionColor;                                     //!< TBD
    nw::ut::Color4f         mViewerColor;                                       //!< TBD
    bool                    mEmitterSetVisible;                                 //!< TBD
    bool                    mEmitterVisible[nw::eft2::EFT_EMITTER_INSET_NUM];   //!< TBD
    nw::eft2::EmitterSet*   mFadeEmitterSet[EFT_PREV_FADE_CTRL_MAX];            //!< TBD


#if 0
    //---------------------------------------------------------------------------
    //! @brief        カラーを設定します。
    //---------------------------------------------------------------------------
    void SetColor( nw::ut::Color4f* color );

    //---------------------------------------------------------------------------
    //! @brief        ビューアカラーを設定します。
    //---------------------------------------------------------------------------
    void SetViewerColor( nw::ut::Color4f* color );

    //---------------------------------------------------------------------------
    //! @brief        パーティクルスケールを設定します。
    //---------------------------------------------------------------------------
    void SetParticleScale( nw::math::VEC2* particle, nw::math::VEC2* emisison );

    //---------------------------------------------------------------------------
    //! @brief        エミッタスケールを設定します。
    //---------------------------------------------------------------------------
    void SetEmitterScale( nw::math::VEC3* emitter );

    //---------------------------------------------------------------------------
    //! @brief        パーティクルライフスケールを設定します。
    //---------------------------------------------------------------------------
    void SetParticleLifeScale( f32 lifeScale );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル速度スケールを設定します。
    //---------------------------------------------------------------------------
    void SetParticleVelocityScale( f32 allDirectionalVec, f32 directionalVec );

    //---------------------------------------------------------------------------
    //! @brief        パーティクルワールド加算速度を設定します。
    //---------------------------------------------------------------------------
    void SetParticleAddWorldVelocity( nw::math::VEC3* velocity );

    //---------------------------------------------------------------------------
    //! @brief        パーティクルワールド加算速度を設定します。
    //---------------------------------------------------------------------------
    void SetParticleRandomVelocity( f32 random );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル指定方向速度のスケール値を設定します。
    //---------------------------------------------------------------------------
    void SetParticleDirectionalVel( f32 velScale );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出スケールを設定します。
    //---------------------------------------------------------------------------
    void SetParticleEmissionScale( f32 emissionRate, f32 emissionInterval );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル エミッタカラー０を設定します。
    //---------------------------------------------------------------------------
    void SetParticleEmitterColor0( nw::ut::Color4f* color );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル エミッタカラー０を設定します。
    //---------------------------------------------------------------------------
    void SetParticleEmitterColor1( nw::ut::Color4f* color );

    //---------------------------------------------------------------------------
    //! @brief        エミッタの描画のOn/Offを設定します。
    //---------------------------------------------------------------------------
    void SetEmitterVisible( u32 emitterID, bool visible );
#endif
};

} // namespace eftvw2
} // namespace nw
