﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Data.h>
#include <nw/eft/eft2_Enum.h>

namespace nw   {
namespace eft2 {

struct ResTexture;

//---------------------------------------------------
//! @brief テクスチャをバインド
//! @param[in] texture      テクスチャインターフェース
//! @param[in] textureType  テクスチャタイプ
//! @param[in] fragLoc      フラグメントシェーダのロケーション
//! @param[in] vertLoc      頂点シェーダのロケーション
//! @param[in] slot         スロット
//---------------------------------------------------
bool BindTexture( TextureInterface texture, TextureType textureType, u32 fragLoc, u32 vertLoc, s32 slot );

//---------------------------------------------------------------------------
//! @brief  テクスチャベースクラス
//---------------------------------------------------------------------------
class TextureBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    TextureBase()
    {
#ifdef EFT_OGL
        m_Texture = (TextureInterface)EFT_INVALID_TEXTURE_ID;
#endif
#if EFT_GX2
        m_Texture = NULL;
#endif
        m_TextureType = EFT_TEXTURE_TYPE_2D;
    }

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    virtual void Finalize(){}

    //---------------------------------------------------------------------------
    //! @brief              テクスチャを有効にします。
    //! @param[in] fragLoc  フラグメントシェーダのロケーション
    //! @param[in] vertLoc  頂点シェーダのロケーション
    //! @param[in] slot     スロット
    //! @return             成功した場合trueを返す。
    //---------------------------------------------------------------------------
    virtual bool Bind( u32 fragLoc, u32 vertLoc, TextureSlot slot )
    {
        EFT_UNUSED_VARIABLE( fragLoc );
        EFT_UNUSED_VARIABLE( vertLoc );
        EFT_UNUSED_VARIABLE( slot );
        return false;
    }

protected:
    TextureInterface    m_Texture;      //!< テクスチャインターフェース
    TextureType         m_TextureType;  //!< テクスチャタイプ
};


//---------------------------------------------------------------------------
//! @brief  外部テクスチャ
//---------------------------------------------------------------------------
class TextureExt : public TextureBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief              初期化
    //! @param[in] texture  テクスチャインターフェース
    //! @param[in] type     テクスチャタイプ
    //---------------------------------------------------------------------------
    void Initialize( TextureInterface texture, TextureType type )
    {
        m_Texture       = texture;
        m_TextureType   = type;
    }

    //---------------------------------------------------------------------------
    //! @brief              テクスチャを有効にします。
    //! @param[in] fragLoc  フラグメントシェーダのロケーション
    //! @param[in] vertLoc  頂点シェーダのロケーション
    //! @param[in] slot     スロット
    //! @return             成功した場合trueを返す。
    //---------------------------------------------------------------------------
    virtual bool Bind( u32 fragLoc, u32 vertLoc, TextureSlot slot );
};


//---------------------------------------------------------------------------
//! @brief  リソースから初期化されるテクスチャ
//---------------------------------------------------------------------------
class TextureResource : public TextureBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief              TextureResourceの初期化
    //! @param[in] texuture テクスチャインターフェース
    //! @param[in] globalId グローバルID
    //! @param[in] texRes   リソースへのポインタ
    //---------------------------------------------------------------------------
    void Initialize( TextureInterface texuture, u64 globalId, ResTexture* texRes );

    //---------------------------------------------------------------------------
    //! @brief                  GX2イメージの初期化
    //! @param[in] id           ID
    //! @param[in] texRes       テクスチャリソースへのポインタ
    //! @param[in] resource     リソースへのポインタ
    //! @param[in] resourceSize リソースサイズ
    //---------------------------------------------------------------------------
    void InitializeGx2bImage( u64 id, ResTexture* texRes, void* resource, u32 resourceSize );

    //---------------------------------------------------------------------------
    //! @brief                  オリジナルイメージの初期化
    //! @param[in] id           ID
    //! @param[in] texRes       テクスチャリソースへのポインタ
    //! @param[in] resource     リソースへのポインタ
    //! @param[in] resourceSize リソースサイズ
    //---------------------------------------------------------------------------
    void InitializeOrgbImage( u64 id, ResTexture* texRes, void* resource, u32 resourceSize );

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief              テクスチャを有効にします。
    //! @param[in] fragLoc  フラグメントシェーダのロケーション
    //! @param[in] vertLoc  頂点シェーダのロケーション
    //! @param[in] slot     スロット
    //! @return             成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool Bind( u32 fragLoc, u32 vertLoc, TextureSlot slot );

    //---------------------------------------------------------------------------
    //! @brief  guid を取得します。
    //! @return GUID
    //---------------------------------------------------------------------------
    u64 GetGrobalID() const { return m_TextureGlobalId; }
    u64 GetGrobalId() const { return m_TextureGlobalId; }
    u64 GetGlobalID() const { return m_TextureGlobalId; }
    u64 GetGlobalId() const { return m_TextureGlobalId; }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャを取得します。
    //! @return テクスチャインターフェース
    //---------------------------------------------------------------------------
    TextureInterface GetTexture() const { return m_Texture; }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャサイズXを取得します。
    //! @return テクスチャサイズX
    //---------------------------------------------------------------------------
    u32 GetSizeX() const { return m_TextureRes->width; }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャサイズYを取得します。
    //! @return テクスチャサイズY
    //---------------------------------------------------------------------------
    u32 GetSizeY() const { return m_TextureRes->height; }

private:
    //---------------------------------------------------------------------------
    //! @brief                  24Bit/32Bit/Float テクスチャの初期化
    //! @param[in] resource     リソースへのポインタ
    //! @param[in] resourceSize リソースサイズ
    //---------------------------------------------------------------------------
    void InitializeNativeTexture( void* resource, u32 resourceSize );

private:
    u64             m_TextureGlobalId;  //!< グローバルID
    ResTexture*     m_TextureRes;       //!< リソースへのポインタ
#if EFT_GX2
    GX2Texture      m_GxTexture;        //!< テクスチャ
#endif
};

} // namespace nw
} // namespace eft2
