﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Callback.h>

namespace nw    {
namespace eft2  {

//---------------------------------------------------------------------------
//! @brief 連結式ストライプの端処理タイプ
//---------------------------------------------------------------------------
enum StripeTailType
{
    EFT_STRIPE_TAIL_TYPE_DEFAULT    = 0,    //!< 「通常」
    EFT_STRIPE_TAIL_TYPE_LOOP       = 1,    //!< 「先端に繋ぐ」
    EFT_STRIPE_TAIL_TYPE_EMITTER    = 2,    //!< 「エミッタに繋ぐ」
};

//---------------------------------------------------------------------------
//! @brief ストライプ展開方向
//---------------------------------------------------------------------------
enum StripePolygonType
{
    EFT_STRIPE_POLYGON_TYPE_BILLBOARD       = 0,    //!< 「ビルボード」
    EFT_STRIPE_POLYGON_TYPE_EMITTER_MATRIX  = 1,    //!< 「エミッタ行列」
    EFT_STRIPE_POLYGON_TYPE_EMITTER_UPRIGHT = 2,    //!< 「エミッタ上下に伸ばす」
    EFT_STRIPE_POLYGON_TYPE_RIBBON          = 3,    //!< 「リボン」
};


//---------------------------------------------------------------------------
//! @brief ストライプ頂点バッファ
//---------------------------------------------------------------------------
struct ConnectionStripeVertexBuffer
{
    nw::math::VEC4*  pos;                                   //!< 背骨の位置 xyz / a
    nw::math::VEC4*  dir;                                   //!< 背骨の向き xyz / empty
    nw::math::VEC4*  emat;                                  //!< 背骨のエミッタマトリクス
};

//------------------------------------------------------------------------------
//! @brief  ストライプ
//------------------------------------------------------------------------------
struct ConnectionStripe
{
    ConnectionStripeVertexBuffer    vertexBuffer;           //!< 頂点バッファ
    nw::math::VEC4                  random;                 //!< ランダム
    nw::math::VEC3                  interpolateNextDir;     //!< 補間ベクトル
    ConnectionStripe*               next;                   //!< 遅延終了処理用のリスト
    u32                             numHistory;             //!< ヒストリが何個分あるか？
    u32                             vertexBufferNum;        //!< 頂点バッファに書き込んだ数
    bool                            used;                   //!< 利用中かどうか
};



//---------------------------------------------------
//! @brief  連結式ストライプを管理するクラスです。
//---------------------------------------------------
class ConnectionStripeSystem
{
public:
    static const EmitterPluginResourceIndex  PluginId = EFT_EMITTER_PLUGIN_ID_1;   //!< エミッタプラグインID
public:
    //------------------------------------------------------------------------------
    //! @brief                  連結式ストライプシステムの初期化処理を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプ最大数
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //------------------------------------------------------------------------------
    //! @brief          連結式ストライプシステムの終了処理を行います。
    //! @param[in] heap ヒープへのポインタ
    //------------------------------------------------------------------------------
    static void Finalize( nw::eft2::Heap* heap );

    //------------------------------------------------------------------------------
    //! @brief  連結式ストライプシステムで確保したワークサイズを取得します。
    //! @return ワークサイズ
    //------------------------------------------------------------------------------
    static u32 GetWorkSize();

    //------------------------------------------------------------------------------
    //! @brief          エミッタ生成後コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合はtrue
    //------------------------------------------------------------------------------
    static bool _EmitterInitializeCallback( EmitterInitializeArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル生成コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合はtrue
    //------------------------------------------------------------------------------
    static bool _ParticleEmitCallback( ParticleEmitArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル削除コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合はtrue
    //------------------------------------------------------------------------------
    static bool _ParticleRemoveCallback( ParticleRemoveArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル計算コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _ParticleCalcCallback( ParticleCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理前コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPreCalcCallback( EmitterPreCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPostCalcCallback( EmitterPostCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ描画コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合はtrue
    //------------------------------------------------------------------------------
    static bool _EmitterDrawCallback( EmitterDrawArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ破棄後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterFinalizeCallback( EmitterFinalizeArg& arg );

private:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief                  連結式ストライプシステムの生成を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプ最大数
    //---------------------------------------------------------------------------
    explicit ConnectionStripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //---------------------------------------------------------------------------
    //! @brief        連結式ストライプシステムの破棄を行います。
    //---------------------------------------------------------------------------
    virtual ~ConnectionStripeSystem();

    //---------------------------------------------------------------------------
    //! @brief              バッファをスワップする。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void SwapBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief                  エミッタ描画処理を行います。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] shaderType   シェーダタイプ
    //! @param[in] userParam    ユーザーパラメータへのポインタ
    //---------------------------------------------------------------------------
    void Draw( Emitter* emitter, ShaderType shaderType, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief              ストライプのポリゴン生成を行います。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void MakeStripeVertexBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              ポリゴン用ワークを確保します。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void AllocStripeVertexBuffer( Emitter* emitter );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return 確保したワークサイズ
    //------------------------------------------------------------------------------
    u32 _GetWorkSize() const { return mStripeWorkSize; }

    //@}

private:
    nw::eft2::System*       mSystem;            //!< エフェクトシステム
    nw::eft2::Heap*         mHeap;              //!< ヒープ
    ConnectionStripe*       mStripeArray;       //!< ストライプ配列
    u32                     mStripeArrayNum;    //!< ストライプ配列数
    u32                     mStripeArrayIdx;    //!<
    u32                     mStripeNum;         //!< ストライプ数
    BufferMode              mBufferMode;        //!< バッファモード
    u32                     mStripeWorkSize;    //!< ストライプワークサイズ
};



} // namespace eft2
} // namespace nw
