﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Callback.h>

namespace nw    {
namespace eft2  {

class System;
struct Emitter;

//------------------------------------------------------------------------------
//! @brief  ストライプ定数
//------------------------------------------------------------------------------
enum StripeConst
{
    EFT_STRIPE_HISTRY_NUM = 256,        //!< 最大履歴数
    EFT_STRIPE_NORMAL_MODE = 0,         //!< 通常ストライプ
    EFT_STRIPE_CROSS_MODE = 1,          //!< クロスストライプ
};

//------------------------------------------------------------------------------
//! @brief  ストライプ履歴
//------------------------------------------------------------------------------
struct StripeHistory
{
    nw::math::VEC4          pos;                            //!< エミッタ（or ワールド）座標系での座標xyz/ スケール(w)
    nw::math::VEC3          dir;                            //!< 進行方向
    nw::math::VEC3          emitterSRT_Y;                   //!< エミッタSRT行列のローカルY軸成分
};

//---------------------------------------------------------------------------
// ストライプ頂点バッファ
//---------------------------------------------------------------------------
struct StripeVertexBuffer
{
    nw::math::VEC4*  pos;                                    //!< 背骨の位置 xyz / a
    nw::math::VEC4*  dir;                                    //!< 背骨の向き xyz / empty
    nw::math::VEC4*  emat;                                   //!< 背骨のエミッタマトリクス
};

//------------------------------------------------------------------------------
//! @brief  ストライプ
//------------------------------------------------------------------------------
struct Stripe
{
    StripeHistory           hist[ EFT_STRIPE_HISTRY_NUM ];  //!< ヒストリ
    StripeVertexBuffer      vertexBuffer;                   //!< 頂点バッファ
    nw::math::VEC4          random;                         //!< ランダム
    nw::math::VEC4          color0;                         //!< カラー0
    nw::math::VEC4          color1;                         //!< カラー1
    nw::math::VEC3          interpolateNextDir;             //!< 補間ベクトル
    Stripe*                 next;                           //!< 遅延終了処理用のリスト
    u32                     numHistory;                     //!< ヒストリが何個分あるか？
    u32                     vertexBufferNum;                //!< 頂点バッファに書き込んだ数
    f32                     time;                           //!< 生成時間
    f32                     life;                           //!< 寿命
    bool                    used;                           //!< 利用中かどうか
};

//---------------------------------------------------
//! @brief  ストライプ用UniformBlock構造体です。
//---------------------------------------------------
struct StripeUniformBlockParam
{
    nw::math::VEC4 random;          //!< random
    nw::math::VEC4 param0;          //!< 先端α / 末端α/ ... / ...
    nw::math::VEC4 color0;          //!< Color0
    nw::math::VEC4 color1;          //!< Color1
    nw::math::VEC4 param1;          //!< ストライプtime / 総頂点数 / CrossMode / ストライプ寿命
};

//---------------------------------------------------
//! @brief  ストライプを管理するクラスです。
//---------------------------------------------------
class StripeSystem
{
public:
    static const EmitterPluginResourceIndex  PluginId = EFT_EMITTER_PLUGIN_ID_2;   //!< エミッタプラグインID
public:
    //------------------------------------------------------------------------------
    //! @brief                  ストライプシステムの初期化処理を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプ最大数
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //------------------------------------------------------------------------------
    //! @brief          ストライプシステムの終了処理を行います。
    //! @param[in] heap ヒープへのポインタ
    //------------------------------------------------------------------------------
    static void Finalize( nw::eft2::Heap* heap );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return ワークサイズ
    //------------------------------------------------------------------------------
    static u32 GetWorkSize();

    //------------------------------------------------------------------------------
    //! @brief      ワンタイムエミッタ用の追加のエミッタ寿命を取得
    //! @param[in]  pEmitter    エミッタへのポインタ
    //! @return                 ワンタイムエミッタ用の追加のエミッタ寿命
    //------------------------------------------------------------------------------
    static s32 GetExtendedEndTimeForOneTimeEmitter( nw::eft2::Emitter* pEmitter );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ生成後コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合true
    //------------------------------------------------------------------------------
    static bool _EmitterInitializeCallback( EmitterInitializeArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル生成コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合true
    //------------------------------------------------------------------------------
    static bool _ParticleEmitCallback( ParticleEmitArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル削除コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合true
    //------------------------------------------------------------------------------
    static bool _ParticleRemoveCallback( ParticleRemoveArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル計算コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _ParticleCalcCallback( ParticleCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理前コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPreCalcCallback( EmitterPreCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPostCalcCallback( EmitterPostCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ描画コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功した場合true
    //------------------------------------------------------------------------------
    static bool _EmitterDrawCallback( EmitterDrawArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ破棄後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterFinalizeCallback( EmitterFinalizeArg& arg );

private:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief                  トライプシステムの生成を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプ最大数
    //---------------------------------------------------------------------------
    explicit StripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //---------------------------------------------------------------------------
    //! @brief        ストライプシステムの破棄を行います。
    //---------------------------------------------------------------------------
    virtual ~StripeSystem();

    //---------------------------------------------------------------------------
    //! @brief  空きストライプインスタンスを確保します。
    //! @return ストライプへのポインタ
    //---------------------------------------------------------------------------
    Stripe* Alloc();

    //---------------------------------------------------------------------------
    //! @brief              ストライプインスタンスを返却します。
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void Free( Stripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief              ストライプインスタンスを遅延リストへ追加します。
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void AddDelayList( Emitter* emitter, Stripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief              バッファをスワップする。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void SwapBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              ポリゴン用ワークを確保します。
    //! @param[in] emitter  エミッタへのポインタ
    //! @return             成功した場合 true
    //---------------------------------------------------------------------------
    bool AllocStripeVertexBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              遅延解放ストライプ計算処理を行います。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void CalcDelayStripe( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              遅延解放ストライトを破棄します。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void FreeDelayStripeList( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              ストライプ計算処理を行います。
    //! @param[in] arg      コールバック引数
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void CalcStripe( ParticleCalcArg& arg, Stripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief                  エミッタ描画処理を行います。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] shaderType   シェーダタイプ
    //! @param[in] userParam    ユーザーパラメータへのポインタ
    //---------------------------------------------------------------------------
    void Draw( Emitter* emitter, ShaderType shaderType, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief        現在のストライプ数取得を行います。
    //! @return 現在のストライプ数
    //---------------------------------------------------------------------------
    u32 GetProcessingStripeNum() const { return mStripeNum; }

    //---------------------------------------------------------------------------
    //! @brief                      ストライプのポリゴン生成を行います。
    //! @param[in] stripe           ストライプへのポインタ
    //! @param[in] stripeData       ストライプ設定へのポインタ
    //! @param[in] vertexBuffer     頂点バッファへのポインタ
    //! @param[in] textureBaseValue テクスチャ座標計算のための値
    //! @param[in] scaleFadeValue   スケールフェードのための値
    //---------------------------------------------------------------------------
    void MakeStripeVertexBuffer( Stripe* stripe, ResStripeHistory* stripeData, StripeVertexBuffer* vertexBuffer, f32 textureBaseValue, f32 scaleFadeValue );

    //---------------------------------------------------------------------------
    //! @brief                      ストライプのポリゴン生成を行います。（ストライプ分割版）
    //! @param[in] stripe           ストライプへのポインタ
    //! @param[in] stripeData       ストライプ設定へのポインタ
    //! @param[in] vertexBuffer     頂点バッファへのポインタ
    //! @param[in] textureBaseValue テクスチャ座標計算のための値
    //! @param[in] scaleFadeValue   スケールフェードのための値
    //---------------------------------------------------------------------------
    void MakeStripeVertexBufferForStripeDivision( Stripe* stripe, ResStripeHistory* stripeData, StripeVertexBuffer* vertexBuffer, f32 textureBaseValue, f32 scaleFadeValue );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return ワークサイズ
    //------------------------------------------------------------------------------
    u32 _GetWorkSize() const { return mStripeWorkSize; }

    //@}
private:
    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return ワークサイズ
    //------------------------------------------------------------------------------
    static u32 GetVertexBufferElementNum()
    {
        // 4バイト（ポインタのサイズ）で割ると要素数になる
        static const u32 elementNum = sizeof( StripeVertexBuffer ) / sizeof( nw::math::VEC4* );
        return elementNum;
    }

private:
    nw::eft2::System*       mSystem;            //!< エフェクトシステム
    nw::eft2::Heap*         mHeap;              //!< ヒープ
    Stripe*                 mStripeArray;       //!< ストライプ配列
    u32                     mStripeArrayNum;    //!< ストライプ配列数
    u32                     mStripeArrayIdx;    //!< ストライプ配列インデックス
    u32                     mStripeNum;         //!< ストライプ数
    BufferMode              mBufferMode;        //!< バッファモード
    u32                     mStripeWorkSize;    //!< ストライプワークサイズ

    static StripeSystem*    sStripeSystem;      //!< ストライプシステムへのポインタ
};

} // namespace eft2
} // namespace nw

namespace nw{
namespace eft2{

//------------------------------------------------------------------------------
//! @brief パーティクル計算コールバック
//------------------------------------------------------------------------------
inline void StripeSystem::_ParticleCalcCallback( ParticleCalcArg& arg )
{
    Stripe* stripe = reinterpret_cast<Stripe*>( arg.emPluginData );
    if ( stripe )
    {
        sStripeSystem->CalcStripe( arg, stripe );
    }
}

//------------------------------------------------------------------------------
//! @brief エミッタ計算処理前コールバック
//------------------------------------------------------------------------------
inline void StripeSystem::_EmitterPreCalcCallback( EmitterPreCalcArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        sStripeSystem->SwapBuffer( arg.emitter );
    }
}

//------------------------------------------------------------------------------
//! @brief エミッタ計算処理後コールバック
//------------------------------------------------------------------------------
inline void StripeSystem::_EmitterPostCalcCallback( EmitterPostCalcArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        sStripeSystem->CalcDelayStripe( arg.emitter );
    }
}

//------------------------------------------------------------------------------
//! @brief エミッタ描画コールバック
//------------------------------------------------------------------------------
inline bool StripeSystem::_EmitterDrawCallback( EmitterDrawArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        sStripeSystem->Draw( arg.emitter, arg.shaderType, arg.userParam );
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//! @brief 描画設定後コールバック ダミー用
//------------------------------------------------------------------------------
inline bool _StripeRenderStateSetCallback( nw::eft2::RenderStateSetArg& arg )
{
    EFT_UNUSED_VARIABLE( arg );
    return true;
}

//---------------------------------------------------------------------------
//! @brief バッファをスワップする。
//---------------------------------------------------------------------------
inline void StripeSystem::SwapBuffer( Emitter* emitter )
{
    emitter->stripeVertexBufferCurIdx = emitter->stripeVertexBufferStartIdx[emitter->bufferID];
    return;
}

}
}
