﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_VertexBuffer.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw   {
namespace eft2 {


//---------------------------------------------------
//! @brief プリミティブクラス
//---------------------------------------------------
struct Primitive
{
public:
    //---------------------------------------------------------------------------
    //! @brief                  初期化処理。
    //! @param[in] heap         TBD
    //! @param[in] guid         TBD
    //! @param[in] indexNum     TBD
    //! @param[in] vertexNum    TBD
    //! @param[in] posSrc       TBD
    //! @param[in] norSrc       TBD
    //! @param[in] tanSrc       TBD
    //! @param[in] colSrc       TBD
    //! @param[in] texSrc       TBD
    //! @param[in] idxSrc       TBD
    //---------------------------------------------------------------------------
    bool Initialize( Heap* heap,
                     u64 guid, u32 indexNum, u32 vertexNum,
                     void* posSrc,
                     void* norSrc,
                     void* tanSrc,
                     void* colSrc,
                     void* texSrc,
                     void* idxSrc )
    {
        m_PrimitivePosition.Initialize();
        m_PrimitiveNormal.Initialize();
        m_PrimitiveTangent.Initialize();
        m_NeedFreeTangent = false;
        m_PrimitiveColor.Initialize();
        m_NeedFreeColor = false;
        m_PrimitiveTexCrd.Initialize();
        m_PrimitiveIndex.Initialize();

        m_PrimitiveIndexNum     = indexNum;
        m_Guid                  = guid;
        m_Initialized           = false;

        if ( !posSrc || !texSrc || !idxSrc )
        {
            // 想定外のプリミティブが入力された
            Warning( NULL, EFT_WARNING_UNSAFE_PRIMITIVE_LOAD );
            return false;
        }

        u32 bufferSize = vertexNum * sizeof( nw::math::VEC4 );

        // 位置
        m_PrimitivePosition.SetVertexBuffer( posSrc, bufferSize, 4 );
        m_PrimitivePosition.Validate();

        // テクスチャ座標
        m_PrimitiveTexCrd.SetVertexBuffer( texSrc, bufferSize, 4 );
        m_PrimitiveTexCrd.Validate();

        // インデックス
        m_PrimitiveIndex.SetVertexBuffer( idxSrc, indexNum * sizeof(u32), 1 );
        m_PrimitiveIndex.Validate();

        // 法線
        if ( norSrc )
        {
            m_PrimitiveNormal.SetVertexBuffer( norSrc, bufferSize, 4 );
            m_PrimitiveNormal.Validate();
        }

        // 接線
        if ( tanSrc )
        {
            m_PrimitiveTangent.SetVertexBuffer( tanSrc, bufferSize, 4 );
            m_PrimitiveTangent.Validate();
        }
        else
        {
            // 接線が無い場合は偽装する
            nw::math::VEC4* tan = static_cast<nw::math::VEC4*>(
                m_PrimitiveTangent.AllocateVertexBuffer( heap, m_PrimitiveIndexNum * sizeof( nw::math::VEC4 ), 4 ) );
            if( tan == NULL )
            {
                // 確保に失敗
                return false;
            }
            for( u32 i = 0; i < m_PrimitiveIndexNum; ++i )
            {
                tan->x = 1.0f;
                tan->y = 0.0f;
                tan->z = 0.0f;
                tan->w = 1.0f;
                tan++;
            }
            m_PrimitiveTangent.Validate();
            m_NeedFreeTangent = true;
        }

        // カラー
        if ( colSrc )
        {
            m_PrimitiveColor.SetVertexBuffer( colSrc, bufferSize, 4 );
            m_PrimitiveColor.Validate();
        }
        else
        {
            // カラーが無い場合は偽装する
            nw::math::VEC4* col = static_cast<nw::math::VEC4*>(
                m_PrimitiveColor.AllocateVertexBuffer( heap, m_PrimitiveIndexNum * sizeof( nw::math::VEC4 ), 4 ) );
            if( col == NULL )
            {
                // 確保に失敗
                m_PrimitiveTangent.Finalize( heap );    // 直前に確保していたバッファを解放
                return false;
            }
            for( u32 i = 0; i < m_PrimitiveIndexNum; ++i )
            {
                col->x = 1.0f;
                col->y = 1.0f;
                col->z = 1.0f;
                col->w = 1.0f;
                col++;
            }
            m_PrimitiveColor.Validate();
            m_NeedFreeColor = true;
        }

        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  終了処理。
    //! @param[in] heap TBD
    //---------------------------------------------------------------------------
    void Finalize( Heap* heap )
    {
        // 開放が必要なのは偽装したバッファだけ
        if (m_NeedFreeTangent)
        {
            m_PrimitiveTangent.Finalize( heap );
        }

        if (m_NeedFreeColor)
        {
            m_PrimitiveColor.Finalize( heap );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  位置バッファをバインド。
    //! @param[in] attr TBD
    //---------------------------------------------------------------------------
    void BindPosionBuffer( u32 attr )
    {
        if ( ( attr != EFT_INVALID_SHADER_ATTRIBUTE ) && m_PrimitivePosition.GetVertexBuffer() )
        {
            m_PrimitivePosition.BindBuffer( attr, m_PrimitivePosition.GetVertexBufferSize(), sizeof(nw::math::VEC4) );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  法線バッファをバインド。
    //! @param[in] attr TBD
    //---------------------------------------------------------------------------
    void BindNormalBuffer( u32 attr )
    {
        if ( ( attr != EFT_INVALID_SHADER_ATTRIBUTE ) && m_PrimitiveNormal.GetVertexBuffer() )
        {
            m_PrimitiveNormal.BindBuffer( attr, m_PrimitiveNormal.GetVertexBufferSize(), sizeof(nw::math::VEC4) );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  接線バッファをバインド。
    //! @param[in] attr TBD
    //---------------------------------------------------------------------------
    void BindTangentBuffer( u32 attr )
    {
        if ( ( attr != EFT_INVALID_SHADER_ATTRIBUTE ) && m_PrimitiveTangent.GetVertexBuffer() )
        {
            m_PrimitiveTangent.BindBuffer( attr, m_PrimitiveTangent.GetVertexBufferSize(), sizeof(nw::math::VEC4) );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  カラーバッファをバインド。
    //! @param[in] attr TBD
    //---------------------------------------------------------------------------
    void BindColorBuffer( u32 attr )
    {
        if ( ( attr != EFT_INVALID_SHADER_ATTRIBUTE ) && m_PrimitiveColor.GetVertexBuffer() )
        {
            m_PrimitiveColor.BindBuffer( attr, m_PrimitiveColor.GetVertexBufferSize(), sizeof(nw::math::VEC4) );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャ座標バッファをバインド。
    //! @param[in] attr TBD
    //---------------------------------------------------------------------------
    void BindTextureCrdBuffer( u32 attr ) const
    {
        if ( ( attr != EFT_INVALID_SHADER_ATTRIBUTE ) && m_PrimitiveTexCrd.GetVertexBuffer() )
        {
            m_PrimitiveTexCrd.BindBuffer( attr, m_PrimitiveTexCrd.GetVertexBufferSize(), sizeof(nw::math::VEC4) );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを取得。
    //! @return TBD
    //---------------------------------------------------------------------------
    void* GetIndexBuffer() const { return m_PrimitiveIndex.GetVertexBuffer(); }

    //---------------------------------------------------------------------------
    //! @brief  頂点数を取得。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetVertexNum() const { return m_PrimitivePosition.GetVertexBufferSize()/16; }

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を取得。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetIndexNum() const { return m_PrimitiveIndexNum; }

    //---------------------------------------------------------------------------
    //! @brief  初期化済みかどうか。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_Initialized; }

    //---------------------------------------------------------------------------
    //! @brief  座標頂点バッファを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const VertexBuffer& GetPositionVertexBuffer() const { return m_PrimitivePosition; }
    //---------------------------------------------------------------------------
    //! @brief  法線バッファを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const VertexBuffer& GetNoramlVertexBuffer()   const { return m_PrimitiveNormal; }
    //---------------------------------------------------------------------------
    //! @brief  頂点カラーバッファを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const VertexBuffer& GetColorVertexBuffer()    const { return m_PrimitiveColor; }

    //---------------------------------------------------------------------------
    //! @brief  guid を取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u64 GetGrobalID() const { return m_Guid; }
    u64 GetGrobalId() const { return m_Guid; }
    u64 GetGlobalID() const { return m_Guid; }
    u64 GetGlobalId() const { return m_Guid; }

private:
    bool                            m_Initialized;              //!< 初期化済みかどうか
    bool                            m_NeedFreeTangent;          //!< 接線バッファの解放が必要かどうか
    bool                            m_NeedFreeColor;            //!< カラーバッファの解放が必要かどうか
    u64                             m_Guid;                     //!< プリミティブグローバルID
    u32                             m_PrimitiveIndexNum;        //!< インデックス数
    VertexBuffer                    m_PrimitivePosition;        //!< プリミティブ座標VBO
    VertexBuffer                    m_PrimitiveNormal;          //!< 法線座標VBO
    VertexBuffer                    m_PrimitiveTangent;         //!< 接線座標VBO
    VertexBuffer                    m_PrimitiveColor;           //!< カラー座標VBO
    VertexBuffer                    m_PrimitiveTexCrd;          //!< プリミティブテクスチャ座標VBO
    VertexBuffer                    m_PrimitiveIndex;           //!< インデックスVBO
};


} // namespace eft2
} // namespace nw
