﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>

namespace nw    {
namespace eft2  {


//----------------------------------------
//! @name システム系
//@{


//---------------------------------------------------------------------------
//! @brief      デフォルト描画プライオリティ
//!
//!             エミッタセットソート描画を行う際の描画優先度デフォルト値です。
//!             描画ソートは、[描画プライオリティ(8Bit)|Z距離(24Bit)]のキーを元に行われます。
//---------------------------------------------------------------------------
enum { EFT_DEFAULT_DRAW_PRIORITY = 128 };

//---------------------------------------------------------------------------
//! @brief        システム定数定義。
//---------------------------------------------------------------------------
enum SystemConstants
{
    EFT_BINARY_VERSION                  = 0x00000015,           //!< 現在のバージョン
    EFT_EMITTER_INSET_NUM               = 16,                   //!< エミッタセットの中の最大エミッタ個数
    EFT_TEXTURE_PATTERN_NUM             = 32 ,                  //!< テクスチャーパターンテーブル数
    EFT_INFINIT_LIFE                    = 0x0fffffff,           //!< 寿命無限の時のライフ ( 414日 )
    EFT_GROUP_MAX                       = 64,                   //!< 最大グループ数
    EFT_VIEWER_GROUP                    = EFT_GROUP_MAX - 1,    //!< ビューアが利用するグループID
    EFT_CALLBACK_MAX                    = 8,                    //!< コールバック最大登録数
    EFT_PTCL_BINARY_ALIGNMENT           = (16 * 1024),          //!< PTCLバイナリメモリアライメント
    EFT_CUSTOM_SHADER_PARAM_NUM         = 32,                   //!< カスタムシェーダパラメータ数
};

//---------------------------------------------------------------------------
//! @brief        CPUコアです。デフォルトでは EFT_CPU_CORE_1 が指定されます。
//---------------------------------------------------------------------------
enum CpuCore
{
    EFT_CPU_CORE_0 = 0,                                         //!< コア0(サブ)
    EFT_CPU_CORE_1 = 1,                                         //!< コア1(標準)
    EFT_CPU_CORE_2 = 2,                                         //!< コア2(サブ)
    EFT_CPU_CORE_MAX = 3,                                       //!< 最大CPUコア数
};

//---------------------------------------------------------------------------
//! @brief        バッファモード。
//---------------------------------------------------------------------------
enum BufferMode
{
    EFT_BUFFER_MODE_DOUBLE  = 2,    //!< ダブルバッファリング
    EFT_BUFFER_MODE_TRIPLE  = 3,    //!< トリプルバッファリング
};

//---------------------------------------------------------------------------
//! @brief        バッファID。
//---------------------------------------------------------------------------
enum BufferID
{
    EFT_BUFFER_ID_0 = 0,                                         //!< コア0(サブ)
    EFT_BUFFER_ID_1 = 1,                                         //!< コア1(標準)
    EFT_BUFFER_ID_DOUBLE_MAX = 2,                                //!< 最大CPUコア数
    EFT_BUFFER_ID_2 = 2,                                         //!< コア2(サブ)
    EFT_BUFFER_ID_TRIPLE_MAX = 3,                                //!< 最大CPUコア数
};

//@}



//----------------------------------------
//! @name エミッタセット系
//@{

//---------------------------------------------------------------------------
//! @brief      無効なエミッタセットIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_EMITTER_SET_ID = 0xFFFFFFFF,    //!< 無効なエミッタセット
};

//---------------------------------------------------------------------------
//! @brief      無効なエミッタIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_EMITTER_ID = 0xFFFFFFFF,    //!< 無効なエミッタ
};

//---------------------------------------------------------------------------
//! @brief      エミッタ計算処理の返り値で、後処理の種類を指定する値
//---------------------------------------------------------------------------
enum EmitterCalculationResult
{
    EmitterCalculationResult_Alive  = 0,    //!< エミッタ計算処理結果：成功（生存状態）
    EmitterCalculationResult_Kill   = 1,    //!< エミッタ計算処理結果：Kill() して終了
    EmitterCalculationResult_Fade   = 2,    //!< エミッタ計算処理結果：Fade() して終了
};

//@}


//----------------------------------------
//! @name パーティクル系
//@{

//---------------------------------------------------------------------------
//! @brief        パーティクルソートタイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_SORT_TYPE_NONE = 0,                             //!< ソートなし
    EFT_SORT_TYPE_ORDER = 3,                            //!< 昇順
    EFT_SORT_TYPE_REV_ORDER = 1,                        //!< 降順
    EFT_SORT_TYPE_Z_SORT = 2,                           //!< Zソート
};
typedef u8 ParticleSortType;


//---------------------------------------------------------------------------
//! @brief        パーティクル形状タイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_PARTCILE_FIGURE_TYPE_PARTICLE   = 0 ,           //!< パーティクル
    EFT_PARTICLE_FIGURE_TYPE_PRIMITIVE,                 //!< プリミティブ
};
typedef u8 ParticleFigureType;

//---------------------------------------------------------------------------
//! @brief  ビルボードタイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_PARTICLE_TYPE_BILLBOARD = 0,                    //!< ビルボード
    EFT_PARTICLE_TYPE_COMPLEX_BILLBOARD,                //!< 高機能ビルボード
    EFT_PARTICLE_TYPE_Y_BILLBOARD,                      //!< Y軸ビルボード
    EFT_PARTICLE_TYPE_POLYGON_XY,                       //!< XY平面のポリゴン
    EFT_PARTICLE_TYPE_POLYGON_XZ,                       //!< XZ平面のポリゴン
    EFT_PARTICLE_TYPE_VEL_LOOK,                         //!< 速度方向を向くビルボード
    EFT_PARTICLE_TYPE_VEL_LOOK_POLYGON,                 //!< 速度方向を向くポリゴン
    EFT_PARTICLE_TYPE_Y_BILLBOARD_PARALLEL,             //!< Y軸ビルボード（カメラに平行）
    EFT_PARTICLE_TYPE_WITH_SCALE_Z,                     //!< 立体風ビルボード
    EFT_PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL,         //!< 立体風ビルボード（ディレクショナル版）

    EFT_BILLBOARD_TYPE_STRIPE,                          //!< 履歴式ストライプ

    EFT_BILLBOARD_TYPE_MAX,                             //!< ビルボードタイプの総数
    EFT_BILLBOARD_TYPE_FORCE_4BYTE = 0x80000000         //!< TBD
};
typedef u8 ParticleType;

//---------------------------------------------------------------------------
//! @brief        パーティクル回転タイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_PARTCILE_ROTATE_TYPE_NONE   = 0 ,               //!< 回転しない
    EFT_PARTICLE_ROTATE_TYPE_ROTX,                      //!< X軸回転
    EFT_PARTICLE_ROTATE_TYPE_ROTY,                      //!< Y軸回転
    EFT_PARTICLE_ROTATE_TYPE_ROTZ,                      //!< Z軸回転
    EFT_PARTICLE_ROTATE_TYPE_ROTYZX,                    //!< YZX軸回転
    EFT_PARTICLE_ROTATE_TYPE_ROTXYZ,                    //!< XYZ軸回転
    EFT_PARTICLE_ROTATE_TYPE_ROTZXY,                    //!< ZXY軸回転
    EFT_PARTICLE_ROTATE_TYPE_MAX,
    EFT_PARTICLE_ROTATE_TYPE_FORCE_4BYTE = 0x80000000
};
typedef u8 ParticleRotType;

//------------------------------------------------------------------------------
//! @brief      パーティクル カラー/アルファ 計算タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_PARTICLE_COLOR_CALC_TYPE_FIXED           = 0,   //!< 固定
    EFT_PARTICLE_COLOR_CALC_TYPE_RANDOM          = 3,   //!< 8カラーランダム
    EFT_PARTICLE_COLOR_CALC_TYPE_8KEY            = 2,   //!< 8キーアニメ
};
typedef u8 ParticleColorCalcType;

//@}





//----------------------------------------
//! @name テクスチャ系
//@{

//---------------------------------------------------------------------------
//! @brief        テクスチャスロット
//---------------------------------------------------------------------------
enum TextureSlot
{
    EFT_TEXTURE_SLOT_0              = 0,                //!< スロット0 テクスチャ0
    EFT_TEXTURE_SLOT_1              = 1,                //!< スロット1 テクスチャ1
    EFT_TEXTURE_SLOT_2              = 2,                //!< スロット2 テクスチャ2
    EFT_TEXTURE_SLOT_MAX            = 3,                //!< スロット  最大数
    EFT_TEXTURE_SLOT_EXT_FB         = 3,                //!< フレームバッファ テクスチャ
    EFT_TEXTURE_SLOT_EXT_DB         = 4,                //!< デプスバッファ   テクスチャ
    EFT_TEXTURE_SLOT_EXT_CN         = 5,                //!< カールノイズ   テクスチャ
};

//---------------------------------------------------------------------------
//! @brief        カスタムシェーダ用テクスチャスロット
//---------------------------------------------------------------------------
enum CustomShaderTextureSlot
{
    EFT_CUSTOM_TEXTURE_SLOT_0       = 6,                //!< スロット0
    EFT_CUSTOM_TEXTURE_SLOT_1       = 7,                //!< スロット1
    EFT_CUSTOM_TEXTURE_SLOT_2       = 8,                //!< スロット2
    EFT_CUSTOM_TEXTURE_SLOT_3       = 9,                //!< スロット3
    EFT_CUSTOM_TEXTURE_SLOT_MAX     = 10,               //!< スロット3
};

//---------------------------------------------------------------------------
//! @brief        テクスチャタイプ
//---------------------------------------------------------------------------
enum TextureType
{
    EFT_TEXTURE_TYPE_2D              = 0,                //!< 2D               テクスチャ
    EFT_TEXTURE_TYPE_2D_ARRAY        = 1,                //!< 2D Array         テクスチャ
    EFT_TEXTURE_TYPE_3D              = 2,                //!< 3D               テクスチャ
    EFT_TEXTURE_TYPE_CUBE_MAP        = 3,                //!< Cube Map         テクスチャ
    EFT_TEXTURE_TYPE_CUBE_MAP_ARRAY  = 4,                //!< Cube Map Array   テクスチャ
    EFT_TEXTURE_TYPE_SHADOW          = 5,                //!< Shadow Map       テクスチャ
    EFT_TEXTURE_TYPE_SHADOW_ARRAY    = 6,                //!< Shadow Map Array テクスチャ
};


//---------------------------------------------------------------------------
//! @brief        カスタムテクスチャ ID
//---------------------------------------------------------------------------
enum CustomShaderTextureID
{
    EFT_CUSTOM_SHADER_TEXTURE_ID_0                  = 0,            //!< カスタムシェーダ ID 0
    EFT_CUSTOM_SHADER_TEXTURE_ID_1                  = 1,            //!< カスタムシェーダ ID 1
    EFT_CUSTOM_SHADER_TEXTURE_ID_2                  = 2,            //!< カスタムシェーダ ID 2
    EFT_CUSTOM_SHADER_TEXTURE_ID_3                  = 3,            //!< カスタムシェーダ ID 3
    EFT_CUSTOM_SHADER_TEXTURE_ID_MAX                = 4,            //!< カスタムシェーダ IDの総数
};


//---------------------------------------------------------------------------
//! @brief        カスタムテクスチャ Type
//---------------------------------------------------------------------------
enum CustomShaderTextureType
{
    EFT_CUSTOM_SHADER_TEXTURE_2D_0                  = 0,            //!< カスタムシェーダ 2D テクスチャ 0
    EFT_CUSTOM_SHADER_TEXTURE_2D_1                  = 1,            //!< カスタムシェーダ 2D テクスチャ 1
    EFT_CUSTOM_SHADER_TEXTURE_2D_2                  = 2,            //!< カスタムシェーダ 2D テクスチャ 2
    EFT_CUSTOM_SHADER_TEXTURE_2D_3                  = 3,            //!< カスタムシェーダ 2D テクスチャ 3
    EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_0            = 4,            //!< カスタムシェーダ 2D ARRAY テクスチャ 0
    EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_1            = 5,            //!< カスタムシェーダ 2D ARRAY テクスチャ 1
    EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_2            = 6,            //!< カスタムシェーダ 2D ARRAY テクスチャ 2
    EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_3            = 7,            //!< カスタムシェーダ 2D ARRAY テクスチャ 3
    EFT_CUSTOM_SHADER_TEXTURE_3D_0                  = 8,            //!< カスタムシェーダ 3D テクスチャ ID 0
    EFT_CUSTOM_SHADER_TEXTURE_3D_1                  = 9,            //!< カスタムシェーダ 3D テクスチャ ID 1
    EFT_CUSTOM_SHADER_TEXTURE_3D_2                  = 10,           //!< カスタムシェーダ 3D テクスチャ ID 2
    EFT_CUSTOM_SHADER_TEXTURE_3D_3                  = 11,           //!< カスタムシェーダ 3D テクスチャ ID 3
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_0            = 12,           //!< カスタムシェーダ CUBE MAP テクスチャ 0
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_1            = 13,           //!< カスタムシェーダ CUBE MAP テクスチャ 1
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_2            = 14,           //!< カスタムシェーダ CUBE MAP テクスチャ 2
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_3            = 15,           //!< カスタムシェーダ CUBE MAP テクスチャ 3
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_0      = 16,           //!< カスタムシェーダ CUBE MAP ARRAY テクスチャ 0
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_1      = 17,           //!< カスタムシェーダ CUBE MAP ARRAY テクスチャ 1
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_2      = 18,           //!< カスタムシェーダ CUBE MAP ARRAY テクスチャ 2
    EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_3      = 19,           //!< カスタムシェーダ CUBE MAP ARRAY テクスチャ 3
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_0              = 20,           //!< カスタムシェーダ SHADOW テクスチャ ID 0
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_1              = 21,           //!< カスタムシェーダ SHADOW テクスチャ ID 1
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_2              = 22,           //!< カスタムシェーダ SHADOW テクスチャ ID 2
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_3              = 23,           //!< カスタムシェーダ SHADOW テクスチャ ID 3
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_0        = 24,           //!< カスタムシェーダ SHADOW ARRAY テクスチャ ID 0
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_1        = 25,           //!< カスタムシェーダ SHADOW ARRAY テクスチャ ID 1
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_2        = 26,           //!< カスタムシェーダ SHADOW ARRAY テクスチャ ID 2
    EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_3        = 27,           //!< カスタムシェーダ SHADOW ARRAY テクスチャ ID 3

    EFT_CUSTOM_SHADER_TEXTURE_MAX                   = 28,           //!< カスタムシェーダ テクスチャタイプの総数
};


//---------------------------------------------------------------------------
//! @brief        テクスチャフォーマット。
//---------------------------------------------------------------------------
enum
{
    EFT_TEXTURE_FORMAT_NONE = 0,                        //!< NONE
    EFT_TEXTURE_FORMAT_24BIT_COLOR,                     //!< 24bitカラー(win/cafe)
    EFT_TEXTURE_FORMAT_32BIT_COLOR,                     //!< 32bitカラー(win/cafe)
    EFT_TEXTURE_FORMAT_UNORM_BC1,                       //!< UNORM BC1(cafe)
    EFT_TEXTURE_FORMAT_SRGB_BC1,                        //!< SRGB  BC1(cafe)
    EFT_TEXTURE_FORMAT_UNORM_BC2,                       //!< UNORM BC2(cafe)
    EFT_TEXTURE_FORMAT_SRGB_BC2,                        //!< SRGB  BC2(cafe)
    EFT_TEXTURE_FORMAT_UNORM_BC3,                       //!< UNORM BC3(cafe)
    EFT_TEXTURE_FORMAT_SRGB_BC3,                        //!< SRGB  BC3(cafe)
    EFT_TEXTURE_FORMAT_UNORM_BC4,                       //!< UNORM BC4(cafe)
    EFT_TEXTURE_FORMAT_SNORM_BC4,                       //!< SNORM BC4(cafe)
    EFT_TEXTURE_FORMAT_UNORM_BC5,                       //!< UNORM BC5(cafe)
    EFT_TEXTURE_FORMAT_SNORM_BC5,                       //!< SNORM BC5(cafe)
    EFT_TEXTURE_FORMAT_UNORM_8,                         //!< UNORM 8(cafe)
    EFT_TEXTURE_FORMAT_UNORM_8_8,                       //!< UNORM 8_8(cafe)
    EFT_TEXTURE_FORMAT_SRGB_8_8_8_8,                    //!< SNORM 8_8_8_8(cafe)
    EFT_TEXTURE_FORMAT_SNORM_8,                         //!< SNORM 8(cafe)
    EFT_TEXTURE_FORMAT_UNORM_4_4,                       //!< UNORM 4_4(cafe)
    EFT_TEXTURE_FORMAT_FLOAT_11_11_10,                  //!< FLOAT 11_11_10(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_16,                        //!< 16bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_16_16,                     //!< 16bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_16_16_16_16,               //!< 16bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_32,                        //!< 32bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_32_32,                     //!< 32bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_FLOAT_32_32_32_32,               //!< 32bit Float カラー(win/cafe)
    EFT_TEXTURE_FORMAT_UNORM_5_6_5,                     //!< UNORM 5_6_5(cafe)
    EFT_TEXTURE_FORMAT_UNORM_5_5_5_1,                   //!< UNORM 5_5_5_1(cafe)

    EFT_TEXTURE_FORMAT_ETC1_A4,                         //!< ETC1_A4(CTGA)
    EFT_TEXTURE_FORMAT_ETC1,                            //!< ETC1(CTGA)
    EFT_TEXTURE_FORMAT_HILO8,                           //!< HILO8(CTGA)
    EFT_TEXTURE_FORMAT_L4,                              //!< L4(CTGA)
    EFT_TEXTURE_FORMAT_A4,                              //!< A4(CTGA)
    EFT_TEXTURE_FORMAT_L8,                              //!< L8(CTGA)
    EFT_TEXTURE_FORMAT_A8,                              //!< A8(CTGA)
    EFT_TEXTURE_FORMAT_LA4,                             //!< LA4(CTGA)
    EFT_TEXTURE_FORMAT_LA8,                             //!< LA8(CTGA)
    EFT_TEXTURE_FORMAT_RGB5_A1,                         //!< RGB5_A1(CTGA)
    EFT_TEXTURE_FORMAT_RGBA4,                           //!< RGBA4(CTGA)
    EFT_TEXTURE_FORMAT_RGBA8,                           //!< RGBA8(CTGA)
    EFT_TEXTURE_FORMAT_RGB8,                            //!< RGB8(CTGA)
    EFT_TEXTURE_FORMAT_RGB565,                          //!< RGB565(CTGA)

    EFT_TEXTURE_FORMAT_UNORM_PVRTC1_2BPP,               //!< RGB_PVRTC_2BPPV1
    EFT_TEXTURE_FORMAT_UNORM_PVRTC1_4BPP,               //!< RGB_PVRTC_4BPPV1
    EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_2BPP,         //!< RGBA_PVRTC_2BPPV1
    EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_4BPP,         //!< RGBA_PVRTC_4BPPV1
    EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_2BPP,         //!< RGBA_PVRTC_2BPPV2
    EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_4BPP,         //!< RGBA_PVRTC_4BPPV2
    EFT_TEXTURE_FORMAT_SRGB_PVRTC1_2BPP,                //!< SRGB_PVRTC_2BPPV1
    EFT_TEXTURE_FORMAT_SRGB_PVRTC1_4BPP,                //!< SRGB_PVRTC_4BPPV1
    EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_2BPP,          //!< SRGBA_PVRTC_2BPPV1
    EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_4BPP,          //!< SRGBA_PVRTC_4BPPV1
    EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_2BPP,          //!< SRGBA_PVRTC_2BPPV2
    EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_4BPP           //!< SRGBA_PVRTC_4BPPV2
};
typedef u8 TextureFormat;


//---------------------------------------------------------------------------
//! @brief        テクスチャフィルタ
//---------------------------------------------------------------------------
enum
{
    EFT_TEXTURE_FILTER_TYPE_LINEAR,                 //!< リニア
    EFT_TEXTURE_FILTER_TYPE_NEAR,                   //!< ニア
};
typedef u8 TextureFilterMode;

//---------------------------------------------------------------------------
//! @brief        テクスチャラップ
//---------------------------------------------------------------------------
enum
{
    EFT_TEXTURE_WRAP_TYPE_MIRROR,                   //!< ミラー
    EFT_TEXTURE_WRAP_TYPE_REPEAT,                   //!< リピート
    EFT_TEXTURE_WRAP_TYPE_CLAMP,                    //!< クランプ
    EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE,              //!< ミラーワンス
};
typedef u8 TextureWrapMode;

//---------------------------------------------------------------------------
//! @brief      無効なアセットGUID
//!
//!             無効なアセットGUIDを表す値。
//---------------------------------------------------------------------------
#if !EFT_GX2
enum : u64
{
    EFT_INVALID_ASSET_GUID = 0xFFFFFFFFFFFFFFFF,    //!< 無効なアセットGUID
};
#endif

//---------------------------------------------------------------------------
//! @brief      無効なテクスチャID
//!
//!             無効なテクスチャIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_TEXTURE_ID = 0xFFFFFFFF,    //!< 無効なテクスチャID
};

//---------------------------------------------------------------------------
//! @brief      無効なテクスチャサンプラ
//!
//!             無効なテクスチャサンプラーを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_TEXTURE_SAMPLER = 0xFFFFFFFF,   //!< 無効なテクスチャサンプラー
};

//---------------------------------------------------------------------------
//! @brief      無効なテクスチャロケーション
//!
//!             無効なテクスチャロケーションを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_TEXTURE_LOCATION = 0xFFFFFFFF,  //!< 無効なテクスチャロケーション
};

//---------------------------------------------------------------------------
//! @brief        テクスチャパタンアニメタイプ
//---------------------------------------------------------------------------
enum
{
    EFT_TEXTURE_PTN_ANIM_TYPE_NONE      = 0,         //!< 利用しない
    EFT_TEXTURE_PTN_ANIM_TYPE_LIFE_FIT  = 1,         //!< ライフフィット
    EFT_TEXTURE_PTN_ANIM_TYPE_CLAMP     = 2,         //!< クランプ
    EFT_TEXTURE_PTN_ANIM_TYPE_LOOP      = 3,         //!< ループ
    EFT_TEXTURE_PTN_ANIM_TYPE_RANDOM    = 4,         //!< ランダム
    EFT_TEXTURE_PTN_ANIM_TYPE_SURFACE   = 5,         //!< サーフェイスアニメ
};
typedef u8 TexturePtnAnimType;


//@}


//----------------------------------------
//! @name コンバイナ系
//@{

//---------------------------------------------------------------------------
//! @brief        カラー計算式タイプ
//---------------------------------------------------------------------------
enum
{
    EFT_COMBINER_COLOR_PROC_COLOR                 = 0,        //!< Color0
    EFT_COMBINER_COLOR_PROC_TEXTURE               = 1,        //!< Color0 * Texture
    EFT_COMBINER_COLOR_PROC_TEXTURE_INTERPOLATE   = 2,        //!< Color0 * Texture + Color1 * ( 1 - Texture )
    EFT_COMBINER_COLOR_PROC_TEXTURE_ADD           = 3,        //!< Color0 * Texture + Color1
};
typedef u8 CombinerColorProcess;

//------------------------------------------------------------------------------
//! @brief        アルファ計算式タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_COMBINER_ALPHA_PROC_MOD                     = 0,	//!< CombinedA * Alpha0
    EFT_COMBINER_ALPHA_PROC_MOD_ALPHA0_ALPHA1       = 1,	//!< CombinedA * Alpha0 * Alpha1
    EFT_COMBINER_ALPHA_PROC_SUB                     = 2,	//!< ( CombinedA - ( 1 - Alpha0 ) ) * 2
    EFT_COMBINER_ALPHA_PROC_SUB_ALPHA0_MOD_ALPHA1   = 3,	//!< ( CombinedA - Alpha0 ) * Alpha1
    EFT_COMBINER_ALPHA_PROC_DIST_FIELD              = 4,	//!< clamp( ( CombinedA - Alpha0 ) *4 ) * Alpha1
};
typedef u8 CombinerAlphaProcess;

//------------------------------------------------------------------------------
//! @brief        コンバイナブレンドタイプ
//------------------------------------------------------------------------------
enum
{
    EFT_COMBINER_BLEND_TYPE_MOD = 0,                        //!< 乗算ブレンド
    EFT_COMBINER_BLEND_TYPE_ADD = 1,                        //!< 加算ブレンド
    EFT_COMBINER_BLEND_TYPE_SUB = 2,                        //!< 減算ブレンド
};
typedef u8 CombinerBlendType;

//------------------------------------------------------------------------------
//! @brief        コンバイナ カラー 入力　タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_COMBINER_COLOR_INPUT_TYPE_SRC = 0,                        //!< 入力をそのまま利用
    EFT_COMBINER_COLOR_INPUT_TYPE_ONE = 1,                        //!< 入力を 1.0 に置き換え
    EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_RGB = 2,              //!< 入力を (1.0 - RGB) に置き換え
    EFT_COMBINER_COLOR_INPUT_TYPE_ALPHA = 3,                      //!< 入力を Alpha に置き換え
    EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_ALPHA = 4,            //!< 入力を (1.0 - Alpha)に置き換え
};
typedef u8 CombinerColorInputType;

//------------------------------------------------------------------------------
//! @brief        コンバイナ アルファ 入力　タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_COMBINER_ALPHA_INPUT_TYPE_SRC = 0,                        //!< 入力をそのまま利用
    EFT_COMBINER_ALPHA_INPUT_TYPE_RED = 1,                        //!< 入力を Red に置き換え(アルファのみ)
    EFT_COMBINER_ALPHA_INPUT_TYPE_ONE = 2,                        //!< 入力を 1.0 に置き換え
    EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_ALPHA = 3,            //!< 入力を (1.0 - Alpha) に置き換え
    EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_RED = 4,              //!< 入力を (1.0 - Red) に置き換え
};
typedef u8 CombinerAlphaInputType;

//@}




//---------------------------------------------------------------------------
//! @brief      無効なリソースID
//!
//!             無効なリソースIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_RESOURCE_ID = 0xFFFFFFFF,       //!< 無効なリソースID
};


//---------------------------------------------------------------------------
//! @brief      無効なプリミティブ Index
//!
//!             無効なプリミティブIndexを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_PRIMITIVE_INDEX = 0xFFFFFFFF,   //!< 無効なプリミティブIndex
};





//----------------------------------------
//! @name エミッタ系
//@{

//---------------------------------------------------------------------------
//! @brief      エミッタ追従タイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_EMITTER_FOLLOW_TYPE_ALL = 0,                            //!< 完全追従
    EFT_EMITTER_FOLLOW_TYPE_NONE,                               //!< 追従しない
    EFT_EMITTER_FOLLOW_TYPE_POS,                                //!< 位置だけ追従
    EFT_EMITTER_FOLLOW_TYPE_MAX,                                //!< エミッタ追従タイプ総数
};
typedef u8 EmitterFollowType;


//---------------------------------------------------------------------------
//! @brief      エミッタボリューム形状
//---------------------------------------------------------------------------
enum
{
    EFT_EMITTER_VOLUME_TYPE_POINT = 0,                          //!< 点エミッタ
    EFT_EMITTER_VOLUME_TYPE_CIRCLE,                             //!< 円
    EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE,                 //!< 円(分割)
    EFT_EMITTER_VOLUME_TYPE_CIRCLE_FILL,                        //!< 円(フィル)
    EFT_EMITTER_VOLUME_TYPE_SPHERE,                             //!< 球
    EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE,                 //!< 球(分割)
    EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE64,               //!< 球(分割64)
    EFT_EMITTER_VOLUME_TYPE_SPHERE_FILL,                        //!< 球(フィル)
    EFT_EMITTER_VOLUME_TYPE_CYLINDER,                           //!< 円柱
    EFT_EMITTER_VOLUME_TYPE_CYLINDER_FILL,                      //!< 円柱(フィル)
    EFT_EMITTER_VOLUME_TYPE_BOX,                                //!< 立方体
    EFT_EMITTER_VOLUME_TYPE_BOX_FILL,                           //!< 立方体(フィル)
    EFT_EMITTER_VOLUME_TYPE_LINE,                               //!< 線
    EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE,                   //!< 線(分割)
    EFT_EMITTER_VOLUME_TYPE_RECTANGLE,                          //!< 矩形
    EFT_EMITTER_VOLUME_TYPE_PRIMITIVE,                          //!< プリミティブ
    EFT_EMITTER_VOLUME_TYPE_MAX,                                //!< エミッタ形状総数
};
typedef u8 EmitterVolumeType;

//---------------------------------------------------------------------------
//! @brief  プリミティブエミッタ放出タイプ
//---------------------------------------------------------------------------
enum
{
    EFT_EMIT_FROM_PRIMITIVE_UNISON      = 0,                    //!< 一斉放出
    EFT_EMIT_FROM_PRIMITVE_RANDOM       = 1,                    //!< ランダムな頂点から放出
    EFT_EMIT_FROM_PRIMITIVE_INDEX_ORDER = 2,                    //!< インデックス順に放出
    EFT_EMIT_FROM_PRIMITIVE_INVALID     = 0xFFFFFFFF,           //!< プリミティブエミッタではない
};
typedef u32 PrimitiveEmissionType;



//@}


//----------------------------------------
//! @name パーティクル挙動系
//@{

//------------------------------------------------------------------------------
//! @brief        パーティクル　オフセットタイプ
//------------------------------------------------------------------------------
enum
{
    EFT_PARTICLE_OFFSET_TYPE_DEPTH             = 0,             //!< デプス方向
    EFT_PARTICLE_OFFSET_TYPE_CAMERA            = 1,             //!< カメラ方向
    EFT_PARTICLE_OFFSET_TYPE_DISABLE           = 2,             //!< 無効
    EFT_PARTICLE_OFFSET_TYPE_CAMERA_FIXED_SIZE = 3,             //!< カメラ方向（サイズ固定）
};
typedef u8 OffsetType;

//------------------------------------------------------------------------------
//! @brief        パーティクル揺らぎ 波形タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_FLUCTUATION_WAVE_TYPE_SIN = 0,                           //!< Sin波
    EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH = 1,                     //!< のこぎり波
    EFT_FLUCTUATION_WAVE_TYPE_RECT = 2,                          //!< 矩形波
};
typedef u8 FluctuationWaveType;

//------------------------------------------------------------------------------
//! @brief        エミッタ挙動計算 タイプ
//------------------------------------------------------------------------------
enum
{
    EFT_EMITTER_CALC_TYPE_CPU          = 0,                    //!< CPUで挙動計算
    EFT_EMITTER_CALC_TYPE_GPU          = 1,                    //!< GPUで挙動計算
    EFT_EMITTER_CALC_TYPE_GPU_SO       = 2,                    //!< GPU+StreamOutで挙動計算
};
typedef u8 EmitterCalcType;

//@}



//----------------------------------------
//! @name 描画ステート系
//@{

//---------------------------------------------------------------------------
//! @brief        フレームバッファとのブレンドタイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_BLEND_TYPE_NORMAL = 0,              //!< 通常のブレンド
    EFT_BLEND_TYPE_ADD,                     //!< 加算ブレンド
    EFT_BLEND_TYPE_SUB,                     //!< 減算ブレンド
    EFT_BLEND_TYPE_MULT,                    //!< 乗算ブレンド
    EFT_BLEND_TYPE_SCREEN,                  //!< スクリーンブレンド
    EFT_BLEND_TYPE_NONE,                    //!< ブレンドしない
};
typedef u8 BlendType;

//---------------------------------------------------------------------------
//! @brief        Ｚバッファー、αテスト関連。
//---------------------------------------------------------------------------
enum
{
    EFT_ZBUFF_ATEST_TYPE_NORMAL = 0,     //!< ブレンド有り 通常の状態（Ｚ比較あり、Ｚ書き込みなし、αテスト０より大きいが合格）
    EFT_ZBUFF_ATEST_TYPE_ZIGNORE,        //!< ブレンド有り Ｚ無視　　（Ｚ比較なし、Ｚ書き込みなし、αテスト０より大きいが合格）
    EFT_ZBUFF_ATEST_TYPE_ENTITY,         //!< ブレンド無し 実体　　　（Ｚ比較あり、Ｚ書き込み有り、αテスト0.5以上が合格）
    EFT_ZBUFF_ATEST_TYPE_OPAQUE,         //!< ブレンド無し 不透明　　（Ｚ比較あり、Ｚ書き込み有り、αテスト無し）
    EFT_ZBUFF_ATEST_TYPE_MAX,               //!< バリエーションの総数
    EFT_ZBUFF_ATEST_TYPE_FORCE_4BYTE = 0x80000000   //!< TBD
};

typedef u8 ZBufATestType;

//---------------------------------------------------------------------------
//! @brief        表示面のタイプ。
//---------------------------------------------------------------------------
enum
{
    EFT_DISPLAYSIDETYPE_BOTH = 0,                       //!< 両面
    EFT_DISPLAYSIDETYPE_FRONT,                          //!< 表
    EFT_DISPLAYSIDETYPE_BACK,                           //!< 裏
};
typedef u8 DisplaySideType;


//---------------------------------------------------------------------------
//! @brief        アルファ/デプス 比較式。
//---------------------------------------------------------------------------
enum
{
    EFT_COMPARE_NEVER       = 0,              //!< 常に通過しない
    EFT_COMPARE_LESS        = 1,              //!< ＜ 基準値
    EFT_COMPARE_EQUAL       = 2,              //!< ＝ 基準値
    EFT_COMPARE_LEQUAL      = 3,              //!< ≦ 基準値
    EFT_COMPARE_GREATER     = 4,              //!< ＞ 基準値
    EFT_COMPARE_NOTEQUAL    = 5,              //!< ≠ 基準値
    EFT_COMPARE_GEQUAL      = 6,              //!< ≧ 基準値
    EFT_COMPARE_ALWAYS      = 7,              //!< 常に通過
};
typedef u8 CompareFunc;

//@}





//----------------------------------------
//! @name 描画系
//@{


//---------------------------------------------------------------------------
//! @brief        描画パス最大数
//---------------------------------------------------------------------------
enum
{
    EFT_DRAW_PATH_MAX = 32,
};

//---------------------------------------------------------------------------
//! @brief        描画パスフラグ
//!
//!               描画パスを指定して描画する際に利用するフラグです。
//---------------------------------------------------------------------------
enum DrawPathFlag
{
    EFT_DRAW_PATH_FLAG_0   = 0x01 << 0,       //!< 描画パス0
    EFT_DRAW_PATH_FLAG_1   = 0x01 << 1,       //!< 描画パス1
    EFT_DRAW_PATH_FLAG_2   = 0x01 << 2,       //!< 描画パス2
    EFT_DRAW_PATH_FLAG_3   = 0x01 << 3,       //!< 描画パス3
    EFT_DRAW_PATH_FLAG_4   = 0x01 << 4,       //!< 描画パス4
    EFT_DRAW_PATH_FLAG_5   = 0x01 << 5,       //!< 描画パス5
    EFT_DRAW_PATH_FLAG_6   = 0x01 << 6,       //!< 描画パス6
    EFT_DRAW_PATH_FLAG_7   = 0x01 << 7,       //!< 描画パス7
    EFT_DRAW_PATH_FLAG_8   = 0x01 << 8,       //!< 描画パス8
    EFT_DRAW_PATH_FLAG_9   = 0x01 << 9,       //!< 描画パス9
    EFT_DRAW_PATH_FLAG_10  = 0x01 << 10,      //!< 描画パス10
    EFT_DRAW_PATH_FLAG_11  = 0x01 << 11,      //!< 描画パス11
    EFT_DRAW_PATH_FLAG_12  = 0x01 << 12,      //!< 描画パス12
    EFT_DRAW_PATH_FLAG_13  = 0x01 << 13,      //!< 描画パス13
    EFT_DRAW_PATH_FLAG_14  = 0x01 << 14,      //!< 描画パス14
    EFT_DRAW_PATH_FLAG_15  = 0x01 << 15,      //!< 描画パス15
    EFT_DRAW_PATH_FLAG_16  = 0x01 << 16,      //!< 描画パス16
    EFT_DRAW_PATH_FLAG_17  = 0x01 << 17,      //!< 描画パス17
    EFT_DRAW_PATH_FLAG_18  = 0x01 << 18,      //!< 描画パス18
    EFT_DRAW_PATH_FLAG_19  = 0x01 << 19,      //!< 描画パス19
    EFT_DRAW_PATH_FLAG_20  = 0x01 << 20,      //!< 描画パス20
    EFT_DRAW_PATH_FLAG_21  = 0x01 << 21,      //!< 描画パス21
    EFT_DRAW_PATH_FLAG_22  = 0x01 << 22,      //!< 描画パス22
    EFT_DRAW_PATH_FLAG_23  = 0x01 << 23,      //!< 描画パス23
    EFT_DRAW_PATH_FLAG_24  = 0x01 << 24,      //!< 描画パス24
    EFT_DRAW_PATH_FLAG_25  = 0x01 << 25,      //!< 描画パス25
    EFT_DRAW_PATH_FLAG_26  = 0x01 << 26,      //!< 描画パス26
    EFT_DRAW_PATH_FLAG_27  = 0x01 << 27,      //!< 描画パス27
    EFT_DRAW_PATH_FLAG_28  = 0x01 << 28,      //!< 描画パス28
    EFT_DRAW_PATH_FLAG_29  = 0x01 << 29,      //!< 描画パス29
    EFT_DRAW_PATH_FLAG_30  = 0x01 << 30,      //!< 描画パス30
    EFT_DRAW_PATH_FLAG_31  = 0x01 << 31,      //!< 描画パス31
};


//---------------------------------------------------------------------------
//! @brief        描画ビューフラグ
//---------------------------------------------------------------------------
enum DrawViewFlag
{
    EFT_DRAW_VIEW_FLAG_0          = ( 0x01 << 0  ),                //!< ビュー 0
    EFT_DRAW_VIEW_FLAG_1          = ( 0x01 << 1  ),                //!< ビュー 1
    EFT_DRAW_VIEW_FLAG_2          = ( 0x01 << 2  ),                //!< ビュー 2
    EFT_DRAW_VIEW_FLAG_3          = ( 0x01 << 3  ),                //!< ビュー 3
    EFT_DRAW_VIEW_FLAG_4          = ( 0x01 << 4  ),                //!< ビュー 4
    EFT_DRAW_VIEW_FLAG_5          = ( 0x01 << 5  ),                //!< ビュー 5
    EFT_DRAW_VIEW_FLAG_6          = ( 0x01 << 6  ),                //!< ビュー 6
    EFT_DRAW_VIEW_FLAG_7          = ( 0x01 << 7  ),                //!< ビュー 7
    EFT_DRAW_VIEW_FLAG_8          = ( 0x01 << 8  ),                //!< ビュー 8
    EFT_DRAW_VIEW_FLAG_9          = ( 0x01 << 9  ),                //!< ビュー 9
    EFT_DRAW_VIEW_FLAG_10         = ( 0x01 << 10 ),                //!< ビュー 10
    EFT_DRAW_VIEW_FLAG_11         = ( 0x01 << 11 ),                //!< ビュー 11
    EFT_DRAW_VIEW_FLAG_12         = ( 0x01 << 12 ),                //!< ビュー 12
    EFT_DRAW_VIEW_FLAG_13         = ( 0x01 << 13 ),                //!< ビュー 13
    EFT_DRAW_VIEW_FLAG_14         = ( 0x01 << 14 ),                //!< ビュー 14
    EFT_DRAW_VIEW_FLAG_15         = ( 0x01 << 15 ),                //!< ビュー 15
    EFT_DRAW_VIEW_FLAG_16         = ( 0x01 << 16 ),                //!< ビュー 16
    EFT_DRAW_VIEW_FLAG_17         = ( 0x01 << 17 ),                //!< ビュー 17
    EFT_DRAW_VIEW_FLAG_18         = ( 0x01 << 18 ),                //!< ビュー 18
    EFT_DRAW_VIEW_FLAG_19         = ( 0x01 << 19 ),                //!< ビュー 19
    EFT_DRAW_VIEW_FLAG_20         = ( 0x01 << 20 ),                //!< ビュー 20
    EFT_DRAW_VIEW_FLAG_21         = ( 0x01 << 21 ),                //!< ビュー 21
    EFT_DRAW_VIEW_FLAG_22         = ( 0x01 << 22 ),                //!< ビュー 22
    EFT_DRAW_VIEW_FLAG_23         = ( 0x01 << 23 ),                //!< ビュー 23
    EFT_DRAW_VIEW_FLAG_24         = ( 0x01 << 24 ),                //!< ビュー 24
    EFT_DRAW_VIEW_FLAG_25         = ( 0x01 << 25 ),                //!< ビュー 25
    EFT_DRAW_VIEW_FLAG_26         = ( 0x01 << 26 ),                //!< ビュー 26
    EFT_DRAW_VIEW_FLAG_27         = ( 0x01 << 27 ),                //!< ビュー 27
    EFT_DRAW_VIEW_FLAG_28         = ( 0x01 << 28 ),                //!< ビュー 28
    EFT_DRAW_VIEW_FLAG_29         = ( 0x01 << 29 ),                //!< ビュー 29
    EFT_DRAW_VIEW_FLAG_30         = ( 0x01 << 30 ),                //!< ビュー 30
    EFT_DRAW_VIEW_FLAG_31         = ( 0x01 << 31 ),                //!< ビュー 31

    EFT_DRAW_VIEW_FLAG_ALL        = EFT_DRAW_VIEW_FLAG_0  | EFT_DRAW_VIEW_FLAG_1  | EFT_DRAW_VIEW_FLAG_2  | EFT_DRAW_VIEW_FLAG_3  |
    EFT_DRAW_VIEW_FLAG_4  | EFT_DRAW_VIEW_FLAG_5  | EFT_DRAW_VIEW_FLAG_6  | EFT_DRAW_VIEW_FLAG_7  |
    EFT_DRAW_VIEW_FLAG_8  | EFT_DRAW_VIEW_FLAG_9  | EFT_DRAW_VIEW_FLAG_10 | EFT_DRAW_VIEW_FLAG_11 |
    EFT_DRAW_VIEW_FLAG_12 | EFT_DRAW_VIEW_FLAG_13 | EFT_DRAW_VIEW_FLAG_14 | EFT_DRAW_VIEW_FLAG_15 |
    EFT_DRAW_VIEW_FLAG_16 | EFT_DRAW_VIEW_FLAG_17 | EFT_DRAW_VIEW_FLAG_18 | EFT_DRAW_VIEW_FLAG_19 |
    EFT_DRAW_VIEW_FLAG_20 | EFT_DRAW_VIEW_FLAG_21 | EFT_DRAW_VIEW_FLAG_22 | EFT_DRAW_VIEW_FLAG_23 |
    EFT_DRAW_VIEW_FLAG_24 | EFT_DRAW_VIEW_FLAG_25 | EFT_DRAW_VIEW_FLAG_26 | EFT_DRAW_VIEW_FLAG_27 |
    EFT_DRAW_VIEW_FLAG_28 | EFT_DRAW_VIEW_FLAG_29 | EFT_DRAW_VIEW_FLAG_30 | EFT_DRAW_VIEW_FLAG_31,  //!< ビュー ALL
    EFT_DRAW_VIEW_FLAG_NONE       = 0,                            //!< 描画しない

    EFT_DRAW_VIEW_FLAG_FORCE_4BYTE = 0x80000000                     //!< TBD
};

//@}



//----------------------------------------
//! @name シェーダ系
//@{

//---------------------------------------------------------------------------
//! @brief      無効なシェーダID
//!
//!             無効なシェーダIDを表す値。(Windows版で利用)
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_SHADER_ID = 0xFFFFFFFF, //!< 無効なシェーダID
};

//---------------------------------------------------------------------------
//! @brief      無効なシェーダアトリビュート
//!
//!             無効なシェーダアトリビュートを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_SHADER_ATTRIBUTE = 0xFFFFFFFF,  //!< 無効なシェーダアトリビュート
};

//---------------------------------------------------------------------------
//! @brief      無効なユニフォームロケーション
//!
//!             無効なユニフォームロケーションを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_SHADER_UNIFORM_REGISTER_LOCATION = 0xFFFFFFFF,  //!< 無効なユニフォームロケーション
};

//---------------------------------------------------------------------------
//! @brief      無効なユニフォームブロック バインドポイント
//!
//!             無効なユニフォームブロック バインドポインを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_UBO_BIND_POINT = 0xFFFFFFFF,    //!< 無効なユニフォームブロック バインドポイント
};

//---------------------------------------------------------------------------
//! @brief      無効なユニフォームブロック バインドID
//!
//!             無効なユニフォームブロック バインドIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_UBO_BUFFER_ID = 0xFFFFFFFF, //!< 無効なユニフォームブロック バインドID
};

//---------------------------------------------------------------------------
//! @brief      無効なグローバルID
//!
//!             無効なグローバルIDを表す値。
//---------------------------------------------------------------------------
enum
{
    EFT_INVALID_GLOBAL_ID = -1, //!< 無効なグローバルID
};


//---------------------------------------------------------------------------
//! @brief      シェーダタイプ
//!             基本機能として実装されているフラグメントシェーダタイプです。
//---------------------------------------------------------------------------
enum
{
    EFT_FRAGMENT_SHADER_TYPE_NORMAL         = 0,    //<! 通常
    EFT_FRAGMENT_SHADER_TYPE_REFRACT        = 1,    //<! カラーバッファ屈折
    EFT_FRAGMENT_SHADER_TYPE_DISTORTION     = 2,    //<! テクスチャ歪み
};
typedef u8 FragmentShaderType;

//---------------------------------------------------------------------------
//! @brief        シェーダタイプ
//---------------------------------------------------------------------------
enum ShaderType
{
    EFT_SHADER_TYPE_NORMAL      = 0,    //!< 通常
    EFT_SHADER_TYPE_PATH_DEF1   = 1,    //!< TBD
    EFT_SHADER_TYPE_PATH_DEF2   = 2,    //!< TBD
    EFT_SHADER_TYPE_MAX         = 3,    //!< シェーダタイプの総数
};

//@}



//----------------------------------------
//! @name カスタム シェーダ系
//@{

//------------------------------------------------------------------------------
//! @brief  ユニフォームブロック ID
//------------------------------------------------------------------------------
enum
{
    EFT_CUSTOM_SHADER_UBO_0     = 0,    //!< UBO インデックス 0
    EFT_CUSTOM_SHADER_UBO_1     = 1,    //!< UBO インデックス 1
    EFT_CUSTOM_SHADER_UBO_2     = 2,    //!< UBO インデックス 2
    EFT_CUSTOM_SHADER_UBO_3     = 3,    //!< UBO インデックス 3
    EFT_CUSTOM_SHADER_UBO_MAX   = 4,    //!< UBO インデックスの総数
};
typedef u8 CustomShaderUboID;

//@}


//----------------------------------------
//! @name 描画パスコールバック系
//@{

//---------------------------------------------------------------------------
//! @brief  描画パスコールバックID。
//---------------------------------------------------------------------------
enum DrawPathCallBackID
{
    EFT_DRAW_PATH_CALLBACK_0     = 0,        //!< CallbackID 0
    EFT_DRAW_PATH_CALLBACK_1     = 1,        //!< CallbackID 1
    EFT_DRAW_PATH_CALLBACK_2     = 2,        //!< CallbackID 2
    EFT_DRAW_PATH_CALLBACK_3     = 3,        //!< CallbackID 3
    EFT_DRAW_PATH_CALLBACK_4     = 4,        //!< CallbackID 4
    EFT_DRAW_PATH_CALLBACK_5     = 5,        //!< CallbackID 5
    EFT_DRAW_PATH_CALLBACK_6     = 6,        //!< CallbackID 6
    EFT_DRAW_PATH_CALLBACK_7     = 7,        //!< CallbackID 7
    EFT_DRAW_PATH_CALLBACK_MAX   = 8,        //!< CallbackID MAX

    EFT_DRAW_PATH_CALLBACK_FORCE_4BYTE = 0x80000000 //!< TBD
};

//@}



//----------------------------------------
//! @name キーフレームアニメーション系
//@{

//------------------------------------------------------------------------------
//! @brief      補間のバリエーション
//------------------------------------------------------------------------------
enum KeyFrameAnimInterpolationType
{
    EFT_KEY_FRAME_ANIM_LINEAR = 0,                  //!< 線形補間
    EFT_KEY_FRAME_ANIM_SMOOTH,                      //!< スムース補間
};

//------------------------------------------------------------------------------
//! @brief      エミッタ時間アニメ　タイプ
//------------------------------------------------------------------------------
enum EmitterAnimType
{
    EFT_EMITTER_ANIM_SCALE = 0,                //!< エミッタトランスフォーム スケール( vec3 )
    EFT_EMITTER_ANIM_ROTATE,                   //!< エミッタトランスフォーム 回転( vec3 )
    EFT_EMITTER_ANIM_TRANS,                    //!< エミッタトランスフォーム 平行移動( vec3 )
    EFT_EMITTER_ANIM_COLOR0,                   //!< エミッタカラー0( vec3 )
    EFT_EMITTER_ANIM_COLOR1,                   //!< エミッタカラー1( vec3 )
    EFT_EMITTER_ANIM_EMISSION_RATE,            //!< エミッタ放出レート( f32 )
    EFT_EMITTER_ANIM_PTCL_LIFE,                //!< 寿命( f32 )
    EFT_EMITTER_ANIM_ALPHA0,                   //!< アルファ0( f32 )
    EFT_EMITTER_ANIM_ALPHA1,                   //!< アルファ1( f32 )
    EFT_EMITTER_ANIM_ALL_DIR_VEL,              //!< 全方向速度( f32 )
    EFT_EMITTER_ANIM_DESIGNATED_DIR_SCALE,     //!< 指定方向速度( f32 )
    EFT_EMITTER_ANIM_PTCL_SCALE,               //!< パーティクルサイズの基準値 ( f32 )
    EFT_EMITTER_ANIM_VOLUME_SCALE,             //!< エミッタボリュームスケール ( f32 )
    EFT_EMITTER_ANIM_GRAVITY_SCALE,            //!< 重力 ( f32 )
    EFT_EMITTER_ANIM_MAX,                      //!< エミッタアニメ最大数
};

//@}




//----------------------------------------
//! @name コールバック系
//@{

//---------------------------------------------------------------------------
//! @brief        コールバックセットタイプ。
//---------------------------------------------------------------------------
enum CallBackSetType
{
    EFT_CALLBACK_SET_TYPE_CA     = 0,        //!< カスタムアクション
    EFT_CALLBACK_SET_TYPE_CS     = 1,        //!< カスタムシェーダ
    EFT_CALLBACK_SET_TYPE_EP     = 2,        //!< エミッタプラグイン
    EFT_CALLBACK_SET_TYPE_MAX,               //!< コールバックセットタイプの総数
};

//---------------------------------------------------------------------------
//! @brief        コールバックID。
//---------------------------------------------------------------------------
enum CallBackID
{
    EFT_CUSTOM_ACTION_CALLBACK_ID_1     = 0,        //!< アクション コールバックID 1
    EFT_CUSTOM_ACTION_CALLBACK_ID_2     = 1,        //!< アクション コールバックID 2
    EFT_CUSTOM_ACTION_CALLBACK_ID_3     = 2,        //!< アクション コールバックID 3
    EFT_CUSTOM_ACTION_CALLBACK_ID_4     = 3,        //!< アクション コールバックID 4
    EFT_CUSTOM_ACTION_CALLBACK_ID_5     = 4,        //!< アクション コールバックID 5
    EFT_CUSTOM_ACTION_CALLBACK_ID_6     = 5,        //!< アクション コールバックID 6
    EFT_CUSTOM_ACTION_CALLBACK_ID_7     = 6,        //!< アクション コールバックID 7
    EFT_CUSTOM_ACTION_CALLBACK_ID_8     = 7,        //!< アクション コールバックID 8
    EFT_CUSTOM_SHADER_CALLBACK_ID_NONE  = 8,        //!< シェーダ   コールバック設定無し
    EFT_CUSTOM_SHADER_CALLBACK_ID_1     = 9,        //!< シェーダ   コールバックID 1
    EFT_CUSTOM_SHADER_CALLBACK_ID_2     = 10,       //!< シェーダ   コールバックID 2
    EFT_CUSTOM_SHADER_CALLBACK_ID_3     = 11,       //!< シェーダ   コールバックID 3
    EFT_CUSTOM_SHADER_CALLBACK_ID_4     = 12,       //!< シェーダ   コールバックID 4
    EFT_CUSTOM_SHADER_CALLBACK_ID_5     = 13,       //!< シェーダ   コールバックID 5
    EFT_CUSTOM_SHADER_CALLBACK_ID_6     = 14,       //!< シェーダ   コールバックID 6
    EFT_CUSTOM_SHADER_CALLBACK_ID_7     = 15,       //!< シェーダ   コールバックID 7
    EFT_CUSTOM_SHADER_CALLBACK_ID_8     = 16,       //!< シェーダ   コールバックID 8

    EFT_CALLBACK_ID_MAX                 = 17,       //!< コールバックIDの総数
};

//---------------------------------------------------------------------------
//! @brief        エミッタプラグインのリソースインデックス
//---------------------------------------------------------------------------
enum EmitterPluginResourceIndex
{
    EFT_EMITTER_PLUGIN_ID_NONE          = 0,       //!< エミッタプラグイン割り当て無し
    EFT_EMITTER_PLUGIN_ID_1             = 1,       //!< 連結式ストライプ
    EFT_EMITTER_PLUGIN_ID_2             = 2,       //!< 履歴式ストライプ
    EFT_EMITTER_PLUGIN_ID_3             = 3,       //!< スーパーストライプ
    EFT_EMITTER_PLUGIN_ID_4             = 4,       //!< 範囲内ループ
    EFT_EMITTER_PLUGIN_ID_5             = 5,       //!< [未実装]EP-05
    EFT_EMITTER_PLUGIN_ID_6             = 6,       //!< [未実装]EP-06
    EFT_EMITTER_PLUGIN_ID_7             = 7,       //!< [未実装]EP-07
    EFT_EMITTER_PLUGIN_ID_8             = 8,       //!< [未実装]EP-08
    EFT_EMITTER_PLUGIN_ID_MAX           = 8,       //!< エミッタプラグインIDの総数
};

//---------------------------------------------------------------------------
//! @brief        エミッタプラグインのコールバックID。
//---------------------------------------------------------------------------
enum EmitterPluginID
{
    EFT_EMITTER_PLUGIN_CALLBACK_ID_1     = EFT_EMITTER_PLUGIN_ID_1 - 1,     //!< 連結式ストライプ
    EFT_EMITTER_PLUGIN_CALLBACK_ID_2     = EFT_EMITTER_PLUGIN_ID_2 - 1,     //!< 履歴式ストライプ
    EFT_EMITTER_PLUGIN_CALLBACK_ID_3     = EFT_EMITTER_PLUGIN_ID_3 - 1,     //!< スーパーストライプ
    EFT_EMITTER_PLUGIN_CALLBACK_ID_4     = EFT_EMITTER_PLUGIN_ID_4 - 1,     //!< 範囲内ループ
    EFT_EMITTER_PLUGIN_CALLBACK_ID_5     = EFT_EMITTER_PLUGIN_ID_5 - 1,     //!< [未実装]EP-05
    EFT_EMITTER_PLUGIN_CALLBACK_ID_6     = EFT_EMITTER_PLUGIN_ID_6 - 1,     //!< [未実装]EP-06
    EFT_EMITTER_PLUGIN_CALLBACK_ID_7     = EFT_EMITTER_PLUGIN_ID_7 - 1,     //!< [未実装]EP-07
    EFT_EMITTER_PLUGIN_CALLBACK_ID_8     = EFT_EMITTER_PLUGIN_ID_8 - 1,     //!< [未実装]EP-08
    EFT_EMITTER_PLUGIN_CALLBACK_ID_MAX   = 8,                               //!< エミッタプラグインIDの総数
};

//---------------------------------------------------------------------------
//! @brief テクスチャリング
//---------------------------------------------------------------------------
enum StripeTexturingOption
{
    EFT_STRIPE_TEXTURING_OPTION_FILL        = 0,            //!< 「全て貼る」
    EFT_STRIPE_TEXTURING_OPTION_TILE        = 1,            //!< 「描画範囲に合わせて貼る」
    EFT_STRIPE_TEXTURING_OPTION_MAX_TYPE    = 2,            //!< タイプ総数
};

//---------------------------------------------------------------------------
//! @briefprivate ランタイムエラーコードです。
//---------------------------------------------------------------------------
enum RuntimeError
{
    EFT_RUNTIME_ERROR_NONE = 0,                     //!< エラー無し
};

//---------------------------------------------------------------------------
//! @briefprivate ランタイム警告コードです。
//---------------------------------------------------------------------------
enum RuntimeWarning
{
    EFT_WARNING_NONE                             = 0,              //!< 警告無し
    EFT_WARNING_PARTICLE_EMISSION_FAILED         = ( 1 << 0  ),    //!< パーティクル放出エラー
    EFT_WARNING_PARTICLE_MAX_NUM_IS_ZERO         = ( 1 << 1  ),    //!< パーティクル最大数がゼロ
    EFT_WARNING_PARTICLE_IS_DIRTY                = ( 1 << 2  ),    //!< パーティクルが不正
    EFT_WARNING_PARTICLE_OVERWRITE               = ( 1 << 3  ),    //!< パーティクルを上書き
    EFT_WARNING_CUSTOMACTION_PARAM_IS_NOT_EXIST  = ( 1 << 4  ),    //!< カスタムアクションパラメータが無い
    EFT_WARNING_UNSAFE_PRIMITIVE_LOAD            = ( 1 << 5  ),    //!< 不正なプリミティブデータのロード
    EFT_WARNING_NOT_SET_MANUAL_EMIT_MODE         = ( 1 << 6  ),    //!< マニュアル放出エラー
    EFT_WARNING_SHADER_IS_NOT_EXIST              = ( 1 << 7  ),    //!< 描画に利用するシェーダが無い

    // 以降は内部向け警告
    EFT_WARNING_PARENT_PARTICLE_IS_NOT_EXIST     = ( 1 << 16  ),   //!< 親パーティクルが存在しない

};

//@}



//------------------------------------------------------------------------------
} // namespace eft2
} // namespace nw
