﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Emitter.h>

namespace nw   {
namespace eft2 {

class System;
struct Emitter;
struct EmitterSetInitializeArg;
struct EmitterSetFinalizeArg;
struct EmitterCalcLodArg;
struct EmitterDrawCullArg;
struct DrawEmitterProfilerArg;
typedef void (*EmitterSetInitializeCallback)( EmitterSetInitializeArg& arg );
typedef void (*EmitterSetFinalizeCallback)( EmitterSetFinalizeArg& arg );
typedef EmitterCalculationResult (*EmitterCalcLodCallback)( EmitterCalcLodArg& arg );
typedef bool (*EmitterDrawCullingCallback)( EmitterDrawCullArg& arg );
typedef void (*DrawEmitterProfilerCallback)( DrawEmitterProfilerArg& arg );

//------------------------------------------------------------------------------
//! @brief  エミッタセット
//!
//!         プログラマはエミッタセットを介してエフェクト再生操作を行います。
//------------------------------------------------------------------------------
struct EmitterSet
{

public:
    //----------------------------------------
    //! @name マトリクス操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを設定します。
    //!
    //!               設定するマトリクス情報は、計算処理内でエミッタに対して伝播されます。
    //!               旧ランタイム互換用です。SetMatrix()をご利用ください。
    //! @param[in]    matrixSRT 設定するマトリクス
    //---------------------------------------------------------------------------
    void SetMtx( const nw::math::MTX34& matrixSRT ){ return SetMatrix( matrixSRT ); }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを設定します。
    //!
    //!               設定するマトリクス情報は、計算処理内でエミッタに対して伝播されます。
    //! @param[in]    matrixSRT 設定するマトリクス
    //---------------------------------------------------------------------------
    void SetMatrix( const nw::math::MTX34& matrixSRT );

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               スケール込みのマトリクスを渡した場合表示が壊れます。
    //!               スケール込のマトリクスを扱うときは SetMatrix を利用してください。
    //!               旧ランタイム互換用です。SetMatrixFast()をご利用ください。
    //! @param[in]    matrixRT     エミッタ行列。
    //---------------------------------------------------------------------------
    void SetMtxFast( const nw::math::MTX34& matrixRT ) { return SetMatrixFast( matrixRT ); }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               スケール込みのマトリクスを渡した場合表示が壊れます。
    //!               スケール込のマトリクスを扱うときは SetMatrix を利用してください。
    //! @param[in]    matrixRT     エミッタ行列
    //---------------------------------------------------------------------------
    void SetMatrixFast( const nw::math::MTX34& matrixRT )
    {
        m_MatrixSRT = m_MatrixRT = matrixRT;
        m_AutoCalcScale.Set(1.f, 1.f, 1.f);
        updateParticleScale_();
        m_IsEmitterSRTDirty = 1;
    }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               この関数はスケール抜きのRT行列と、それとは別にスケールを指定できます。
    //!               スケール込みの行列を渡した場合表示が壊れます。
    //!               旧ランタイム互換用です。SetMatrixFast()をご利用ください。
    //! @param[in]    matrixRT RTマトリクス
    //! @param[in]    scale    スケール
    //---------------------------------------------------------------------------
    void SetMtxFast( const nw::math::MTX34 &matrixRT, f32 scale ){ return SetMatrixFast( matrixRT, scale ); }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               この関数はスケール抜きのRT行列と、それとは別にスケールを指定できます。
    //!               スケール込みの行列を渡した場合表示が壊れます。
    //! @param[in]    matrixRT RTマトリクス
    //! @param[in]    scale    スケール
    //---------------------------------------------------------------------------
    void SetMatrixFast( const nw::math::MTX34 &matrixRT, f32 scale )
    {
        m_MatrixSRT.m[0][0] = matrixRT.m[0][0] * scale;
        m_MatrixSRT.m[1][0] = matrixRT.m[1][0] * scale;
        m_MatrixSRT.m[2][0] = matrixRT.m[2][0] * scale;

        m_MatrixSRT.m[0][1] = matrixRT.m[0][1] * scale;
        m_MatrixSRT.m[1][1] = matrixRT.m[1][1] * scale;
        m_MatrixSRT.m[2][1] = matrixRT.m[2][1] * scale;

        m_MatrixSRT.m[0][2] = matrixRT.m[0][2] * scale;
        m_MatrixSRT.m[1][2] = matrixRT.m[1][2] * scale;
        m_MatrixSRT.m[2][2] = matrixRT.m[2][2] * scale;

        m_MatrixSRT.m[0][3] = matrixRT.m[0][3];
        m_MatrixSRT.m[1][3] = matrixRT.m[1][3];
        m_MatrixSRT.m[2][3] = matrixRT.m[2][3];

        m_MatrixRT  = matrixRT;
        m_AutoCalcScale.x = scale;
        m_AutoCalcScale.y = scale;
        m_AutoCalcScale.z = scale;

        updateParticleScale_();
        m_IsEmitterSRTDirty = 1;
    }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               この関数はスケール抜きのRT行列と、それとは別にスケールを指定できます。
    //!               スケール込みの行列を渡した場合表示が壊れます。
    //!               旧ランタイム互換用です。SetMatrixFast()をご利用ください。
    //! @param[in]    matrixRT RTマトリクス
    //! @param[in]    scale    スケール
    //---------------------------------------------------------------------------
    void SetMtxFast( const nw::math::MTX34 &matrixRT, const nw::math::VEC3 &scale ) { return SetMatrixFast( matrixRT, scale ); }

    //---------------------------------------------------------------------------
    //! @brief        マトリクスを高速に設定します。
    //!
    //!               この関数はスケール抜きのRT行列と、それとは別にスケールを指定できます。
    //!               スケール込みの行列を渡した場合表示が壊れます。
    //! @param[in]    matrixRT RTマトリクス
    //! @param[in]    scale    スケール
    //---------------------------------------------------------------------------
    void SetMatrixFast( const nw::math::MTX34 &matrixRT, const nw::math::VEC3 &scale )
    {
        m_MatrixSRT.m[0][0] = matrixRT.m[0][0] * scale.x;
        m_MatrixSRT.m[1][0] = matrixRT.m[1][0] * scale.x;
        m_MatrixSRT.m[2][0] = matrixRT.m[2][0] * scale.x;

        m_MatrixSRT.m[0][1] = matrixRT.m[0][1] * scale.y;
        m_MatrixSRT.m[1][1] = matrixRT.m[1][1] * scale.y;
        m_MatrixSRT.m[2][1] = matrixRT.m[2][1] * scale.y;

        m_MatrixSRT.m[0][2] = matrixRT.m[0][2] * scale.z;
        m_MatrixSRT.m[1][2] = matrixRT.m[1][2] * scale.z;
        m_MatrixSRT.m[2][2] = matrixRT.m[2][2] * scale.z;

        m_MatrixSRT.m[0][3] = matrixRT.m[0][3];
        m_MatrixSRT.m[1][3] = matrixRT.m[1][3];
        m_MatrixSRT.m[2][3] = matrixRT.m[2][3];

        m_MatrixRT  = matrixRT;
        m_AutoCalcScale = scale;

        updateParticleScale_();
        m_IsEmitterSRTDirty = 1;
    }

    //---------------------------------------------------------------------------
    //! @brief        位置をを指定してマトリクスを設定します。
    //!
    //! @param[in] pos 座標
    //---------------------------------------------------------------------------
    void SetPos( const nw::math::VEC3& pos );

    //---------------------------------------------------------------------------
    //! @brief        位置とスケールを指定してマトリクスを設定します。
    //!
    //! @param[in]    pos     座標
    //! @param[in]    scale   スケール
    //---------------------------------------------------------------------------
    void SetPos( const nw::math::VEC3& pos , f32 scale )
    {
        m_MatrixRT.SetIdentity();
        m_MatrixRT.SetTranslate( pos );
        m_MatrixSRT = m_MatrixRT;
        m_MatrixSRT.m[0][0] *= scale;
        m_MatrixSRT.m[1][1] *= scale;
        m_MatrixSRT.m[2][2] *= scale;

        m_AutoCalcScale.x = scale;
        m_AutoCalcScale.y = scale;
        m_AutoCalcScale.z = scale;
        updateParticleScale_();
        m_IsEmitterSRTDirty = 1;
    }

    //---------------------------------------------------------------------------
    //! @brief        位置とスケールを指定してマトリクスを設定します。
    //!
    //! @param[in]    pos     座標
    //! @param[in]    scale   スケール
    //---------------------------------------------------------------------------
    void SetPos( const nw::math::VEC3& pos , const nw::math::VEC3& scale )
    {
        m_MatrixRT.SetIdentity();
        m_MatrixRT.SetTranslate( pos );
        m_MatrixSRT = m_MatrixRT;
        m_MatrixSRT.m[0][0] *= scale.x;
        m_MatrixSRT.m[1][1] *= scale.y;
        m_MatrixSRT.m[2][2] *= scale.z;

        m_AutoCalcScale = scale;
        updateParticleScale_();
        m_IsEmitterSRTDirty = 1;
    }

    //---------------------------------------------------------------------------
    //! @brief        SRTマトリクスを取得します。
    //!
    //! @return       SRTエミッタセットマトリクス
    //---------------------------------------------------------------------------
    const nw::math::MTX34& GetSRTMatrix() const { return m_MatrixSRT; }

    //---------------------------------------------------------------------------
    //! @brief        RTマトリクスを取得します。
    //!
    //! @return       RTエミッタセットマトリクス
    //---------------------------------------------------------------------------
    const nw::math::MTX34& GetRTMatrix() const { return m_MatrixRT; }

    //---------------------------------------------------------------------------
    //! @brief        位置を取得します。
    //!
    //! @param[in] pos 座標
    //---------------------------------------------------------------------------
    void GetPos( nw::math::VEC3& pos ) const
    {
        pos.x = m_MatrixSRT.m[0][3];
        pos.y = m_MatrixSRT.m[1][3];
        pos.z = m_MatrixSRT.m[2][3];
    }

    inline const nw::math::MTX34& GetMatrixRt() const
    {
        return m_MatrixRT;
    }

    inline const nw::math::MTX34& GetMatrixSrt() const
    {
        return m_MatrixSRT;
    }

    void GetPos( nw::math::VEC3* pos ) const
    {
        GetPos( *pos );
    }

    u32 GetUserData1() const
    {
        return GetUserDataNum1();
    }

    u32 GetUserData2() const
    {
        return GetUserDataNum2();
    }

    //@}

    //----------------------------------------
    //! @name エミッタ形状操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エミッタ形状スケール値を設定します。
    //!
    //!               旧ランタイム互換用です。SetEmitterVolumeScale()をご利用ください。
    //! @param[in] scale スケール
    //---------------------------------------------------------------------------
    void SetEmitterScale( const nw::math::VEC3 &scale ){ return SetEmitterVolumeScale( scale ); }

    //---------------------------------------------------------------------------
    //! @brief        エミッタ形状スケール値を設定します。
    //!
    //! @param[in] scale スケール
    //---------------------------------------------------------------------------
    void SetEmitterVolumeScale( const nw::math::VEC3 &scale ) { m_EmitterVolumeScale = scale; }

    //---------------------------------------------------------------------------
    //! @brief        エミッタ形状スケール値を取得します。
    //!
    //! @return       スケール
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetEmitterScale() const{ return GetEmitterVolumeScale(); }

    //---------------------------------------------------------------------------
    //! @brief        エミッタ形状スケール値を取得します。
    //!
    //! @return       スケール
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetEmitterVolumeScale() const{ return m_EmitterVolumeScale; }

    //@}


    //----------------------------------------
    //! @name パーティクル放出量制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出レートのスケール値を設定します。
    //!
    //! @param[in]    ratio     スケール値。1.0以上の値が指定できません。
    //---------------------------------------------------------------------------
    void SetEmissionRatioScale( f32 ratio );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出レートのスケール値を取得します。
    //!
    //! @return       放出間隔のスケール値
    //---------------------------------------------------------------------------
    f32 GetEmissionRatioScale() const { return m_EmissionRatioScale; }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出間隔のスケール値を設定します。
    //!
    //! @param[in]    ratio     スケール値。1.0以下の値が指定できません。
    //---------------------------------------------------------------------------
    void SetEmissionIntervalScale( f32 ratio );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出間隔のスケール値を取得します。
    //!
    //! @return       放出間隔のスケール値
    //---------------------------------------------------------------------------
    f32 GetEmissionIntervalScale() const { return m_EmissionIntervalScale; }

    //@}


    //----------------------------------------
    //! @name パーティクル寿命制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        パーティクル寿命のスケール値を設定します。
    //!
    //! @param[in]    ratio スケール値。1.0以上の値が指定できません。
    //---------------------------------------------------------------------------
    void SetParticleLifeScale( f32 ratio );

    //---------------------------------------------------------------------------
    //! @brief        パーティクル寿命のスケール値を取得します。
    //!
    //! @return       寿命のスケール値。
    //---------------------------------------------------------------------------
    f32 GetParticleLifeScale() const { return m_ParticleLifeScale; }

    //@}


    //----------------------------------------
    //! @name パーティクルスケール制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        放出時のパーティクルの大きさを設定します。
    //!
    //!               旧ランタイム互換用です。SetEmissionParticleScale()をご利用ください。
    //! @param[in]    scale     スケール値。
    //---------------------------------------------------------------------------
    void SetEmissionScale( f32 scale ) { return SetEmissionParticleScale( scale ); }

    //---------------------------------------------------------------------------
    //! @brief        放出時のパーティクルの大きさを設定します。
    //!
    //! @param[in]    scale     スケール値。
    //---------------------------------------------------------------------------
    void SetEmissionParticleScale( f32 scale )
    {
        m_ParticleEmissionScale.x = scale;
        m_ParticleEmissionScale.y = scale;
    }

    //---------------------------------------------------------------------------
    //! @brief        放出時のパーティクルの大きさを設定します。
    //! @param[in]    scaleX     スケール値。
    //! @param[in]    scaleY     スケール値。
    //---------------------------------------------------------------------------
    void SetEmissionParticleScale( f32 scaleX , f32 scaleY ) { m_ParticleEmissionScale.x = scaleX; m_ParticleEmissionScale.y = scaleY; }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出時のスケール値を設定します。
    //!
    //!               Zスケール値は設定されません。
    //! @param[in]    ratio     XYスケール値。
    //---------------------------------------------------------------------------
    void SetEmissionParticleScale( nw::math::VEC2& ratio )
    {
        m_ParticleEmissionScale.x = ratio.x;
        m_ParticleEmissionScale.y = ratio.y;
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出時のスケール値を設定します。
    //!
    //! @param[in]    scale     XYZスケール値。
    //---------------------------------------------------------------------------
    void SetEmissionParticleScale( nw::math::VEC3& scale ) { m_ParticleEmissionScale = scale; }

    //---------------------------------------------------------------------------
    //! @brief        放出時のパーティクルの大きさを設定します。
    //!
    //! @param[in]    scale     スケール値。
    //---------------------------------------------------------------------------
    void SetEmissionParticleScale( const nw::math::VEC3& scale ){ m_ParticleEmissionScale = scale; }

    //---------------------------------------------------------------------------
    //! @brief        放出時のパーティクルの大きさを取得します。
    //!
    //! @return       放出時のパーティクルのスケール値
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetEmissionParticleScale() const { return m_ParticleEmissionScale; }


    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを設定します。
    //!
    //! @param[in] scale XYZスケール値
    //---------------------------------------------------------------------------
    void SetParticleScale( f32 scale )
    {
        m_ParticleScale.x = m_ParticleScale.y = m_ParticleScale.z = scale;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを設定します。
    //!
    //!               Zスケール値は設定されません。
    //! @param[in] scaleX Xスケール値
    //! @param[in] scaleY Yスケール値
    //---------------------------------------------------------------------------
    void SetParticleScale( f32 scaleX , f32 scaleY )
    {
        m_ParticleScale.x = scaleX;
        m_ParticleScale.y = scaleY;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを設定します。
    //!
    //!               Zスケール値は設定されません。
    //! @param[in] scale XYスケール値。
    //---------------------------------------------------------------------------
    void SetParticleScale( const nw::math::VEC2& scale )
    {
        m_ParticleScale.x = scale.x;
        m_ParticleScale.y = scale.y;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルのスケール値を設定します。
    //!
    //!               Zスケール値は設定されません。
    //! @param[in]    scale     XYスケール値。
    //---------------------------------------------------------------------------
    void SetParticleScale( nw::math::VEC2& scale )
    {
        m_ParticleScale.x = scale.x;
        m_ParticleScale.y = scale.y;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを設定します。
    //!
    //! @param[in] scaleX Xスケール値
    //! @param[in] scaleY Yスケール値
    //! @param[in] scaleZ Zスケール値
    //---------------------------------------------------------------------------
    void SetParticleScale( f32 scaleX, f32 scaleY, f32 scaleZ )
    {
        m_ParticleScale.x = scaleX;
        m_ParticleScale.y = scaleY;
        m_ParticleScale.z = scaleZ;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを設定します。
    //!
    //! @param[in]    scale     XYZスケール値。
    //---------------------------------------------------------------------------
    void SetParticleScale( const nw::math::VEC3& scale )
    {
        m_ParticleScale = scale;
        updateParticleScale_();
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを取得します。
    //!
    //! @return パーティクルスケール値
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetParticleScale() const{ return m_ParticleScale; }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルの大きさを取得します。
    //!
    //! @return パーティクルスケール値
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetParticleScaleForCalc() const { return m_ParticleScaleForCalc; }
    const nw::math::VEC3& GetParticleScaleForCalculation() const { return m_ParticleScaleForCalc; }

    //@}


    //----------------------------------------
    //! @name パーティクル初速制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        全方向速度のスケール値を設定します。
    //!
    //! @param[in] velScale 全方向速度スケール値
    //---------------------------------------------------------------------------
    void SetAllDirectionalVel( f32 velScale ){ m_FigureVel = velScale; }

    //---------------------------------------------------------------------------
    //! @brief        全方向速度のスケール値を取得します。
    //!
    //! @return 速度スケール値
    //---------------------------------------------------------------------------
    f32 GetAllDirectionalVel() const{ return m_FigureVel; }

    //---------------------------------------------------------------------------
    //! @brief        初期速度ランダムのスケール値を設定します。
    //!
    //! @param[in] velScale 初期速度ランダムスケール値
    //---------------------------------------------------------------------------
    void SetRandomVel( f32 velScale ){ m_RandomVel = velScale; }

    //---------------------------------------------------------------------------
    //! @brief        初期速度ランダムのスケール値を取得します。
    //!
    //! @return 初期速度ランダムスケール値
    //---------------------------------------------------------------------------
    f32 GetRandomVel() const{ return m_RandomVel; }

    //---------------------------------------------------------------------------
    //! @brief        最終的な速度に加算する値（ワールド座標系）を設定します。
    //!
    //! @param[in] velAdd 加算する速度
    //---------------------------------------------------------------------------
    void SetAddVel( const nw::math::VEC3 &velAdd ){ m_VelAdd = velAdd; }

    //---------------------------------------------------------------------------
    //! @brief        最終的な速度に加算する値（ワールド座標系）を取得します。
    //!
    //! @return     加算する速度
    //---------------------------------------------------------------------------
    const nw::math::VEC3 &GetVelAdd() const{ return m_VelAdd; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向速度のスケール値を設定します。
    //!
    //! @param[in] velScale 指定方向速度のスケール値
    //---------------------------------------------------------------------------
    void SetDirectionalVel( f32 velScale ){ m_DirectionalVel = velScale; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向速度のスケール値を取得します。
    //!
    //! @return 指定方向速度のスケール値
    //---------------------------------------------------------------------------
    f32 GetDirectionalVel() const{ return m_DirectionalVel; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向の方向を設定します。
    //!
    //! @param[in] dir 指定方向の方向
    //---------------------------------------------------------------------------
    void SetDirectional( const nw::math::VEC3 &dir ){ m_Directional = dir ; m_IsSetDirectional = true; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向の方向を取得します。
    //!
    //! @return 指定方向の方向
    //---------------------------------------------------------------------------
    const nw::math::VEC3 &GetDirectional() const{ return m_Directional; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向の方向設定を解除します。
    //---------------------------------------------------------------------------
    void DisableDirectional(){ m_IsSetDirectional = false; }

    //---------------------------------------------------------------------------
    //! @brief        指定方向の方向を設定しているか？
    //! @return       true で設定済み、false で未設定
    //---------------------------------------------------------------------------
    bool IsSetDirectional() const{ return ( m_IsSetDirectional != 0 ); }

    //@}


    //----------------------------------------
    //! @name フレーム操作
    //@{

    //---------------------------------------------------------------------------
    //! @briefprivate   呼び出し開始フレームを設定します。
    //!                 まったく同じ内容のループエミッタセットを放出する時は、
    //!
    //!                 この関数でエミッタ放出開始をずらすと、ワーストケースが減る可能性があります。
    //! @param[in] frame フレーム数
    //---------------------------------------------------------------------------
    void SetStartFrame( s32 frame ){ m_StartFrame = frame; }

    //---------------------------------------------------------------------------
    //! @briefprivate   呼び出し開始フレーム値を取得します。
    //!
    //! @return         呼び出し開始フレーム値
    //---------------------------------------------------------------------------
    s32 GetStartFrame() const{ return m_StartFrame; }

    //@}


    //----------------------------------------
    //! @name パーティクル マニュアル放出制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        パーティクルを手動で放出するモードに設定します。
    //---------------------------------------------------------------------------
    void SetManualParticleEmission()
    {
        m_IsManualEmission = true;
    }

    //---------------------------------------------------------------------------
    //! @brief        個数を指定してパーティクルを手動で放出するモードに設定します。
    //!
    //! @param[in]    particleCount      予約するパーティクルの個数。
    //---------------------------------------------------------------------------
    void SetManualParticleEmissionWithParticleCount( u32 particleCount );

    //---------------------------------------------------------------------------
    //! @brief        パーティクルを手動で放出するモードに設定します。
    //!
    //! @param[in] emissionEnable trueで切り替える
    //---------------------------------------------------------------------------
    void SetManualParticleEmission( bool emissionEnable )
    {
        m_IsManualEmission = emissionEnable;
    }

    //---------------------------------------------------------------------------
    //! @brief      パーティクル手動放出モードか取得します。
    //!
    //! @return     trueで手動放出モード、falseで手動放出モード未設定
    //---------------------------------------------------------------------------
    bool IsManualParticleEmission() const
    {
        return m_IsManualEmission ? true : false;
    }

    //---------------------------------------------------------------------------
    //! @brief      常駐型のマニュアルエミッタセットかどうかを取得します。
    //!
    //! @return     常駐型の場合は true 非常駐型の場合は false
    //---------------------------------------------------------------------------
    bool IsResidentManualEmitterSet() const
    {
        return ( m_IsManualEmission ) && ( m_ResidentEmitterTime < 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief      マニュアルエミッタセットの待機エミッタ時間を取得します。
    //!
    //! @return     粒の数が0でも消滅せずに待機する待ち時間。常駐型・マニュアルエミッタセットでないの場合は -1 が返ります。
    //---------------------------------------------------------------------------
    int GetResidentEmitterTime() const
    {
        if( !IsManualParticleEmission() || IsResidentManualEmitterSet() )
        {
            return -1;
        }
        return m_ResidentEmitterTime;
    }

    //---------------------------------------------------------------------------
    //! @brief  マニュアルエミッタセットが消えても大丈夫な状態かを取得します。
    //! @return 待機時間を過ぎている、もしくは常駐型のエミッタで、処理したパーティクルが無ければ true それ以外は false
    //---------------------------------------------------------------------------
    bool IsManualEmitterSetReadyToExit() const;

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出ポイントリストを設定します。
    //!
    //! @param[in]    numPoint      ポイントの個数。
    //! @param[in]    points        ポイントの座標配列（この関数の後にも参照される為、スタック上のアドレスは禁止です）。
    //---------------------------------------------------------------------------
    void SetParticleEmissionPoints( s32 numPoint, nw::math::VEC3* points );

    //---------------------------------------------------------------------------
    //! @brief          放出位置を指定してパーティクルを手動で放出します。
    //!
    //! @param[in] pos  放出位置
    //---------------------------------------------------------------------------
    void EmitParticle( const nw::math::VEC3 &pos );

    //---------------------------------------------------------------------------
    //! briefprivate
    //! @brief        パーティクルを手動で放出します。
    //! @param[in] pos          放出位置
    //! @param[in] color        放出時カラー
    //! @param[in] scale        放出時スケール
    //! @param[in] matrixSRT    放出時エミッタセットマトリクス
    //---------------------------------------------------------------------------
    void EmitParticle( const nw::math::VEC3 &pos, const nw::math::VEC4 &color, const nw::math::VEC2 &scale = nw::math::VEC2( 1.f, 1.f ), const nw::math::MTX34 *matrixSRT = NULL );

    //@}


    //----------------------------------------
    //! @name パーティクル回転制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        初期角度の設定をします。
    //!
    //! @param[in] rot 回転角度( ラジアン指定 )
    //---------------------------------------------------------------------------
    void SetInitRotate( const nw::math::VEC3 &rot ){ m_InitialRoate = rot ; }

    //@}


    //----------------------------------------
    //! @name エミッタセットカラー制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief        カラー A値(乗算値)を設定します。
    //!
    //! @param[in] alpha A値
    //---------------------------------------------------------------------------
    void SetAlpha( f32 alpha ){ m_Color.w = alpha; }

    //---------------------------------------------------------------------------
    //! @brief        カラー RGB値(乗算値)を設定します。
    //!
    //! @param[in] r R値
    //! @param[in] g G値
    //! @param[in] b B値
    //---------------------------------------------------------------------------
    void SetColor( f32 r , f32 g , f32 b ){ m_Color.x = r; m_Color.y = g; m_Color.z = b; }

    //---------------------------------------------------------------------------
    //! @brief        カラー RGBA値(乗算値)を設定します。
    //!
    //! @param[in] color RGBAカラー値
    //---------------------------------------------------------------------------
    void SetColor( const nw::math::VEC4 &color ){ m_Color = color; }

    //---------------------------------------------------------------------------
    //! @brief      カラー RGBA値(乗算値)を取得します。
    //! @return     RGBAカラー値
    //---------------------------------------------------------------------------
    const nw::math::VEC4& GetColor() const{ return m_Color; }

    //@}


    //----------------------------------------
    //! @name エミッタカラー制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief     カラー RGBA値(乗算値)を設定します。
    //! @param[in] color0 カラー0値
    //! @param[in] color1 カラー1値
    //---------------------------------------------------------------------------
    void SetEmitterColor( const nw::math::VEC4 &color0, const nw::math::VEC4 &color1 );

    //---------------------------------------------------------------------------
    //! @brief     カラー0 RGBA値(乗算値)を設定します。
    //! @param[in] color0 カラー0値
    //---------------------------------------------------------------------------
    void SetEmitterColor0( const nw::math::VEC4 &color0 );

    //---------------------------------------------------------------------------
    //! @brief     カラー1 RGBA値(乗算値)を設定します。
    //! @param[in] color1 カラー1値
    //---------------------------------------------------------------------------
    void SetEmitterColor1( const nw::math::VEC4 &color1 );

    //@}


    //----------------------------------------
    //! @name 表示 / 非表示 操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット描画処理の有効/無効を設定します。
    //!
    //! @param[in]    flag       有効/無効。
    //---------------------------------------------------------------------------
    void SetVisible( bool flag ){ m_IsDraw = flag; }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットの描画処理を停止します。
    //!
    //! @param[in]    isStopDraw       描画処理を止めるか？
    //---------------------------------------------------------------------------
    void SetStopDraw( bool isStopDraw ) { return SetVisible( !isStopDraw ); }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット描画処理の有効/無効を取得します。
    //!
    //! @return 有効/無効フラグ
    //---------------------------------------------------------------------------
    bool IsVisible() const{ return ( m_IsDraw != 0 ); }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット描画処理が停止状態かどうかチェックします。
    //! @return     停止状態であればtrue、そうでなければfalse
    //---------------------------------------------------------------------------
    bool IsStopDraw() const{ return !IsVisible(); }

    //---------------------------------------------------------------------------
    //! @brief      エミッタ描画処理の有効/無効を設定します。
    //!
    //! @param[in]  emitterName エミッタセット名
    //! @param[in]  flag        有効/無効。
    //---------------------------------------------------------------------------
    void SetEmitterVisible( const char* emitterName, bool flag );

    //@}


    //----------------------------------------
    //! @name 情報の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エフェクトシステムの取得を行います。
    //!
    //! @return     システムクラスのインスタンス
    //---------------------------------------------------------------------------
    inline System* GetSystem(){ return m_System; }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセットIDを取得します。
    //!
    //! @return     エミッタセットID
    //---------------------------------------------------------------------------
    inline s32 GetEmitterSetID() const { return m_EmitterSetID; };
    inline s32 GetEmitterSetId() const { return m_EmitterSetID; };

    //---------------------------------------------------------------------------
    //! @brief      生成IDを取得します。
    //!
    //! @return     生成ID
    //---------------------------------------------------------------------------
    inline u32 GetCreateID() const { return m_EmitterSetCreateID; }
    inline u32 GetCreateId() const { return m_EmitterSetCreateID; }

    //---------------------------------------------------------------------------
    //! @brief      所属するグループIDを取得します。
    //!
    //! @return     所属するグループID
    //---------------------------------------------------------------------------
    inline u8 GetGroupID() const { return m_GroupID; }
    inline u8 GetGroupId() const { return m_GroupID; }

    //---------------------------------------------------------------------------
    //! @brief      リソースIDを取得します。
    //!
    //! @return     リソースID
    //---------------------------------------------------------------------------
    inline u32 GetResourceID() const { return m_ResourceID; }
    inline u32 GetResourceId() const { return m_ResourceID; }

    //---------------------------------------------------------------------------
    //! @brief      データ上の親エミッタ数を取得します。
    //!
    //! @return     データ上の親エミッタ数。子エミッタの数を含みません。
    //---------------------------------------------------------------------------
    inline s32 GetEmitterNum() const{ return m_EmitterSetRes->emitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      データ上の全エミッタ数を取得します。
    //!
    //! @return     データ上の全エミッタ数。子エミッタの数も含みます。
    //---------------------------------------------------------------------------
    inline s32 GetEmitterAllNum() const{ return m_EmitterSetRes->emitterAllNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタ数を取得します。
    //!
    //!             旧ランタイム互換用です。GetProcessingEmitterCount()をご利用ください。
    //! @return     再生中のエミッタ数。0であれば、再生を終了しています。
    //---------------------------------------------------------------------------
    inline s32 GetNumEmitter() const{ return GetProcessingEmitterCount(); }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタ数を取得します。
    //!
    //!             旧ランタイム互換用です。GetProcessingEmitterCount()をご利用ください。
    //! @return     再生中のエミッタ数。0であれば、再生を終了しています。
    //---------------------------------------------------------------------------
    inline s32 GetNumAliveEmitter() const { return GetProcessingEmitterCount(); }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタ数を取得します。
    //!
    //!             旧ランタイム互換用です。GetProcessingEmitterCount()をご利用ください。
    //! @return     再生中のエミッタ数。0であれば、再生を終了しています。
    //---------------------------------------------------------------------------
    inline s32 GetEmitterAliveNum() const{ return GetProcessingEmitterCount(); }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタ数を取得します。
    //!
    //!             旧ランタイム互換用です。GetProcessingEmitterCount()をご利用ください。
    //! @return     再生中のエミッタ数。0であれば、再生を終了しています。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingEmitterCount() const{ return m_ProcessingEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生をスキップしたエミッタ数を取得します。
    //!
    //! @return     スキップしたエミッタ数。
    //---------------------------------------------------------------------------
    inline s32 GetCalcSkipEmitterCount() const{ return m_CalcSkipEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタアニメ数を取得します。
    //!
    //! @return     再生中のエミッタアニメ数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingEmitterAnimNum() const{ return m_ProcessingEmitterAnimNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のCpuエミッタ数を取得します。
    //!
    //! @return     再生中のCpuエミッタ数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingCpuEmitterCount() const{ return m_ProcessingCpuEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のGpuエミッタ数を取得します。
    //!
    //! @return     再生中のGpuエミッタ数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingGpuEmitterCount() const{ return m_ProcessingGpuEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のGpuSoエミッタ数を取得します。
    //!
    //! @return     再生中のGpuSoエミッタ数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingGpuSoEmitterCount() const{ return m_ProcessingGpuSoEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のStripe本数を取得します。
    //!
    //! @return     再生中のStripe本数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingStripeCount() const{ return m_ProcessingStripeNum; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のSuperStripe本数を取得します。
    //!
    //! @return     再生中のSuperStripe本数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingSuperStripeCount() const{ return m_ProcessingSuperStripeNum; }

    //---------------------------------------------------------------------------
    //! @brief        所属エミッタを取得します。
    //!
    //! @param[in]    idx     取得するエミッタインデックス。
    //! @return       エミッタインスタンス。
    //---------------------------------------------------------------------------
    const Emitter* GetEmitter( u32 idx ) { return GetAliveEmitter( idx ); }

    //---------------------------------------------------------------------------
    //! @brief        所属エミッタを取得します。
    //!
    //! @param[in]    idx     取得するエミッタインデックス。
    //! @return       エミッタインスタンス。
    //---------------------------------------------------------------------------
    const Emitter* GetAliveEmitter( u32 idx );

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット生成時の所属エミッタ数取得。
    //!
    //!             旧ランタイム互換用です。GetCreatedEmitterNum()をご利用ください。
    //! @return     エミッタ数。
    //---------------------------------------------------------------------------
    s32 GetNumCreatedEmitter() const{ return GetCreatedEmitterNum(); }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット生成時の所属エミッタ数取得。
    //! @return       エミッタ数。
    //---------------------------------------------------------------------------
    s32 GetCreatedEmitterNum() const{ return m_EmitterFirstNum; }

    //---------------------------------------------------------------------------
    //! @brief        再生中のCpuパーティクル数を取得します。
    //!
    //! @return       再生中のCpuパーティクル数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingCpuParticleCount() const{ return m_ProcessingCpuParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief        再生中のGpuパーティクル数を取得します。
    //!
    //! @return       再生中のGpuパーティクル数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingGpuParticleCount() const{ return m_ProcessingGpuParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief        再生中のGpuSoパーティクル数を取得します。
    //!
    //! @return       再生中のGpuSoパーティクル数。
    //---------------------------------------------------------------------------
    inline s32 GetProcessingGpuSoParticleCount() const{ return m_ProcessingGpuSoParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief      動的ヒープから確保したメモリサイズを取得します。
    //!
    //! @return     確保されたメモリサイズ。
    //---------------------------------------------------------------------------
    inline s32 GetAllocedFromDynamicHeapSize() const{ return m_AllocedFromDynamicHeapSize; }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット生死判定を取得します。
    //!
    //! @return     再生死判定。
    //---------------------------------------------------------------------------
    inline bool IsAlive() const
    {
        return ( m_ProcessingEmitterNum > 0 && m_IsUsage );
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセットがループエフェクトかどうか取得します。
    //!
    //! @return     true でループエフェクト
    //---------------------------------------------------------------------------
    bool IsLoopEffect() const { return ( m_IsLoopEffect != 0 ); }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセットが保持するランダムの種を取得します。
    //!
    //! @return     ランダムシード
    //---------------------------------------------------------------------------
    u32 GetEmitterSetRandomSeed() const { return m_RandomSeed; }

    //---------------------------------------------------------------------------
    //! @brief      再生中のエミッタの描画パス論理和を取得します。
    //!
    //! @return     描画パス論理輪
    //---------------------------------------------------------------------------
    u32 GetDrawPath(){ return m_DrawPath; }

    //---------------------------------------------------------------------------
    //! @brief      保持するエミッタ内でフレームバッファを要求する描画パスを論理和を取得します。
    //!
    //! @return     フレームバッファを要求する描画パスの論理和
    //---------------------------------------------------------------------------
    u32 GetRequestFrameBufferTextureDrawPath(){ return m_ReqFrameBufferTexturePath; }

    //---------------------------------------------------------------------------
    //! @brief      保持するエミッタ内でデプスバッファを要求する描画パスを論理和を取得します。
    //!
    //! @return     デプスバッファを要求する描画パスの論理和
    //---------------------------------------------------------------------------
    u32 GetRequestDepthBufferTextureDrawPath(){ return m_ReqDepthBufferTexturePath; }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット名を取得します。
    //!
    //!             旧ランタイム互換用です。GetName()をご利用ください。
    //! @return     エミッタセット名
    //---------------------------------------------------------------------------
    const char* GetEmitterSetName() const { return GetName(); }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット名を取得します。
    //!
    //! @return     エミッタセット名
    //---------------------------------------------------------------------------
    const char* GetName() const
    {
        return m_EmitterSetRes->emitterSetRes->name;
    }

    //---------------------------------------------------------------------------
    //! @brief      無限寿命のエミッタ( パーティクル )を含むかどうかチェックします。
    //! @return     trueで無限寿命エミッタ( パーティクル )を含む
    //---------------------------------------------------------------------------
    bool IsHaveInfinityEmitter() const;

    //@}


    //----------------------------------------
    //! @name コールバック
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット初期化時に呼び出されるコールバック関数を設定します。
    //!
    //! @param[in]  callback 登録するコールバック関数
    //---------------------------------------------------------------------------
    void SetInitializeCallback( EmitterSetInitializeCallback callback )
    {
        m_InitializeCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット初期化時に呼び出されるコールバック関数を取得します。
    //!
    //! @return     登録済みコールバック関数
    //---------------------------------------------------------------------------
    EmitterSetInitializeCallback GetInitializeCallback() const
    {
        return m_InitializeCallback;
    }

    //---------------------------------------------------------------------------
    //! @brief      Finalize()時のコールバックを設定します
    //!
    //! @param[in]  callback 登録するコールバック関数
    //---------------------------------------------------------------------------
    void SetFinalizeCallback( EmitterSetFinalizeCallback callback ){ m_FinalizeCallback = callback; }

    //---------------------------------------------------------------------------
    //! @brief      Finalize()時のコールバックを取得します
    //!
    //! @return     登録済みコールバック関数
    //---------------------------------------------------------------------------
    EmitterSetFinalizeCallback GetFinalizeCallback() const { return m_FinalizeCallback; }

    //@}


    //----------------------------------------
    //! @name フェード処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタの放出を停止します。
    //!
    //!             データ上で指定されたフェード処理を行い、エミッタセットの再生を終了します。
    //---------------------------------------------------------------------------
    void Fade();

    //---------------------------------------------------------------------------
    //! @brief      フェード処理が設定されたか？
    //!
    //! @return     設定されていればtrue、されていなければfalseが返ります。
    //---------------------------------------------------------------------------
    bool IsFadeRequest() const{ return ( m_IsFade != 0 ); }

    //---------------------------------------------------------------------------
    //! @brief      配下のエミッタが一つでもフェード中かどうかを取得します。
    //!
    //! @return     配下のエミッタが一つでもフェード中の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsAnyEmitterFadeRequest() const
    {
        return m_IsAnyEmitterFade;
    }

    //---------------------------------------------------------------------------
    //! @brief        フェード処理をキャンセルします。
    //---------------------------------------------------------------------------
    void CancelFade(){ m_IsFade = false; }

    //@}


    //----------------------------------------
    //! @name 削除
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットを削除します。
    //!
    //!               瞬時に保持しているエミッタを殺します。
    //!               引数の immediate を下げることで、削除タイミングを BeginFrame時に移動できます。
    //!               瞬時に消すのではなく、エミッタの放出を止めるだけなら Fade を使用してください。
    //!
    //! @param[in] immediate 即時に削除するかどうか。
    //---------------------------------------------------------------------------
    void Kill( bool immediate = true );

    //---------------------------------------------------------------------------
    //! @brief        無限寿命のエミッタを削除します。
    //---------------------------------------------------------------------------
    void KillInfinityEmitter();

    //@}


    //----------------------------------------
    //! @name 描画プライオリティ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      描画プライオリティを設定します。
    //!
    //!             システムのエミッタセット描画ソート機能を利用する場合、
    //!             同距離に描画するエミッタセットの描画順を制御できます。
    //! @param[in] priority プライオリティ値(デフォルト値は128)を設定します。
    //---------------------------------------------------------------------------
    void SetDrawPriority( u8 priority ) { m_DrawPriority = priority; }

    //---------------------------------------------------------------------------
    //! @brief      描画プライオリティを取得します。(デフォルト値は128)
    //!
    //! @return     プライオリティ値
    //---------------------------------------------------------------------------
    u8 GetDrawPriority() const { return m_DrawPriority; }

    //@}


    //----------------------------------------
    //! @name 描画ビュー操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタの描画ビューを指定します。
    //!
    //! @param[in]  flag       ビューフラグ。
    //---------------------------------------------------------------------------
    void SetViewFlag( DrawViewFlag flag ){ m_DrawViewFlag = flag; }

    //---------------------------------------------------------------------------
    //! @brief      エミッタの描画処ビューフラグを取得します。
    //!
    //! @return     設定済みビューフラグ
    //---------------------------------------------------------------------------
    DrawViewFlag GetDrawViewFlag() const
    {
        return m_DrawViewFlag;
    }

    //@}


    //----------------------------------------
    //! @name リスト操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief  リスト構造上の次のEmitterSetを取得します。
    //! @return 次のエミッタセットへのポインタ
    //---------------------------------------------------------------------------
    EmitterSet* GetNext() const { return m_Next; }

    //---------------------------------------------------------------------------
    //! @brief  リスト構造上の前のEmitterSetを取得します。
    //! @return 前のエミッタセットへのポインタ
    //---------------------------------------------------------------------------
    EmitterSet* GetPrev() const { return m_Next; }

    //@}


    //----------------------------------------
    //! @name 定期処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタセットの計算処理を行います。
    //! @param[in]  frameRate フレームレート( 1.0で標準 )
    //! @param[in]  swapBuffer バッファをスワップするか
    //! @param[in]  emitterCalcLod LOD処理用コールバック
    //---------------------------------------------------------------------------
    void Calc( f32 frameRate, bool swapBuffer, EmitterCalcLodCallback emitterCalcLod );

    //---------------------------------------------------------------------------
    //! @brief      エミッタセットの計算処理を行います。
    //! @param[in]  shaderType シェーダタイプ
    //! @param[in]  drawPath 描画パス
    //! @param[in]  drawViewFlag ビューフラグ
    //! @param[in]  calcStreamOut ストリームアウト処理を行うかどうか
    //! @param[in]  userParam コールバックに渡すユーザーパラメータ
    //! @param[in]  emitterDrawCull 描画カリングコールバック
    //! @param[in]  profiler GPU描画処理コスト計測用コールバック
    //---------------------------------------------------------------------------
    void Draw( ShaderType shaderType, u32 drawPath, u32 drawViewFlag, bool calcStreamOut, void* userParam, EmitterDrawCullingCallback emitterDrawCull, DrawEmitterProfilerCallback profiler );

    //---------------------------------------------------------------------------
    //! @brief      計算処理を指定回数回す。
    //!             再生開始フレームを指定した場合に利用します。
    //!
    //! @param[in]  numLoop 回す回数。
    //---------------------------------------------------------------------------
    void ForceCalc( u32 numLoop );

    //---------------------------------------------------------------------------
    //! @brief      描画処理の有効/無効を設定します。
    //!
    //! @param[in]  flag 有効/無効。
    //---------------------------------------------------------------------------
    void SetCalcEnable( bool flag ) { m_IsCalc = flag; }

    //---------------------------------------------------------------------------
    //! @brief      描画処理の有効/無効を取得します。
    //!
    //! @return     trueで描画処理有効、falseで無効
    //---------------------------------------------------------------------------
    bool IsCalc() const { return ( m_IsCalc == 1 ); }

    //---------------------------------------------------------------------------
    //! @brief      計算処理と描画処理を同時に停止します。
    //!
    //!             止めると計算が回らないので処理が早くはなりますが、
    //!             内部ではエミッタを確保したままなので、エミッタ数は減りません。
    //! @param[in]  isStopCalcAndDraw       true で計算/描画処理を停止します。
    //---------------------------------------------------------------------------
    void SetStopCalcAndDraw( bool isStopCalcAndDraw )
    {
        m_IsCalc = !isStopCalcAndDraw;
        m_IsDraw = !isStopCalcAndDraw;
    }

    //---------------------------------------------------------------------------
    //! @brief      計算処理と描画処理が停止状態かどうかチェックします。
    //!
    //! @return     trueで描画処理有効、falseで無効
    //---------------------------------------------------------------------------
    bool IsStopCalcAndDraw() const{ return ( !m_IsCalc && !m_IsDraw ); }

    //---------------------------------------------------------------------------
    //! @brief      再生中エミッタの計算処理を停止します。
    //!
    //! @param[in]  isStopCalc       計算処理を止めるか？
    //---------------------------------------------------------------------------
    void SetStopCalc( bool isStopCalc ) { SetCalcEnable( !isStopCalc ); }

    //---------------------------------------------------------------------------
    //! @brief      計算処理が停止状態かどうかチェックします。
    //!
    //! @return     trueで計算処理停止中、falseで再生中
    //---------------------------------------------------------------------------
    bool IsStopCalc() const { return ( m_IsCalc == 0 ); }

    //@}


    //----------------------------------------
    //! @name ユーザーデータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ユーザーデータを取得します。
    //!
    //!             現状動作していません。
    //! @return     ユーザーデータ
    //---------------------------------------------------------------------------
    u32 GetUserData() const{ return 0; }

    //---------------------------------------------------------------------------
    //! @brief      8bit ユーザーデータを取得します。
    //!
    //! @return     8bit ユーザーデータ
    //---------------------------------------------------------------------------
    u32  GetUserDataNum1() const{ return m_EmitterSetRes->emitterSetRes->UserDataValue1; }

    //---------------------------------------------------------------------------
    //! @brief      8bit ユーザーデータを取得します。
    //!
    //! @return     8bit ユーザーデータ
    //---------------------------------------------------------------------------
    u32  GetUserDataNum2() const{ return m_EmitterSetRes->emitterSetRes->UserDataValue2; }

    //---------------------------------------------------------------------------
    //! @brief      16bit ユーザーデータを取得します。
    //!
    //! @return     16bit ユーザーデータ
    //---------------------------------------------------------------------------
    u16 GetUserDataBit() const{ return m_EmitterSetRes->emitterSetRes->UserDataBit; }

    //@}

    //----------------------------------------
    //! @name ランタイムユーザーデータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーポインタを取得します。
    //! @return         ランタイムユーザーポインタ
    //---------------------------------------------------------------------------
    void* GetRuntimeUserPtr() const{ return m_RuntimeUserData; }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーデータ(リソースには含まれず、アプリケーションの都合で設定できる変数)を設定します。
    //!                 初期値は0になっています。
    //! @param[in] ptr  ランタイムユーザーデータ
    //---------------------------------------------------------------------------
    void SetRuntimeUserPtr( void* ptr ) { m_RuntimeUserData = ptr; }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーポインタを取得します。
    //! @deprecated     旧型式の API 名です。nw::eft2::EmitterSet::GetRuntimeUserPtr() に乗り換えてください。
    //! @return         ランタイムユーザーポインタ
    //---------------------------------------------------------------------------
    void* GetUserPtr() const { return GetRuntimeUserPtr(); }

    //---------------------------------------------------------------------------
    //! @brief          ユーザーデータへのポインタを設定します。
    //! @deprecated     旧型式の API 名です。nw::eft2::EmitterSet::SetRuntimeUserPtr() に乗り換えてください。
    //! @param[in] ptr  ユーザーデータへのポインタ
    //---------------------------------------------------------------------------
    void SetUserPtr( void* ptr ) { SetRuntimeUserPtr( ptr ); }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーポインタ( u32 形式)を取得します。
    //! @deprecated     旧型式の API 名です。nw::eft2::EmitterSet::GetRuntimeUserPtr() に乗り換えてください。
    //! @return         ランタイムユーザーポインタ( u32 形式)
    //---------------------------------------------------------------------------
    u32 GetRuntimeUserData() const{ return reinterpret_cast< u32 >( GetRuntimeUserPtr() ); }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーデータ(リソースには含まれず、アプリケーションの都合で設定できる変数)を設定します。
    //!                 初期値は0になっています。
    //! @deprecated     旧型式の API 名です。nw::eft2::EmitterSet::SetRuntimeUserPtr() に乗り換えてください。
    //! @param[in] ptr  ランタイムユーザーデータ( u32 形式)
    //---------------------------------------------------------------------------
    void SetRuntimeUserData( u32 ptr ){ SetRuntimeUserPtr( reinterpret_cast< void* >( ptr ) ); }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムユーザーデータ(リソースには含まれず、アプリケーションの都合で設定できる変数)を設定します。
    //!                 初期値は0になっています。
    //! @deprecated     旧型式の API 名です。nw::eft2::EmitterSet::SetRuntimeUserPtr() に乗り換えてください。
    //! @param[in] ptr  ランタイムユーザーデータ
    //---------------------------------------------------------------------------
    void SetRuntimeUserData( void* ptr ) { SetRuntimeUserPtr( ptr ); }

    //@}

private:
    //---------------------------------------------------------------------------
    //! @brief        エミッタを生成します。
    //! @param[in] emitterResSet TBD
    //! @param[in] particleMax TBD
    //! @param[in] parent TBD
    //! @param[in] childIndex TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    Emitter* CreateEmitter( const EmitterResource* emitterResSet, u32 particleMax, Emitter* parent = NULL, s32 childIndex = -1 );

    //---------------------------------------------------------------------------
    //! @brief        エミッタを削除します。
    //! @param[in] emitter TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    void KillEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief        初期化処理を行います。
    //! @param[in] emitterSetID TBD
    //! @param[in] createID TBD
    //! @param[in] resourceID TBD
    //! @param[in] groupID TBD
    //! @param[in] particleMax TBD
    //! @param[in] heap TBD
    //! @return       成否判定。
    //---------------------------------------------------------------------------
    bool Initialize( s32 emitterSetID, u32 createID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief        終了処理を行います。
    //! @return       成否判定。
    //---------------------------------------------------------------------------
    bool Finalize();

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムの設定を行います。
    //! @param[in] sys TBD
    //---------------------------------------------------------------------------
    void SetSystem( System *sys ){ m_System = sys; }

    //---------------------------------------------------------------------------
    //! @brief        エミッタをリストへ追加します。
    //! @param[in] emitter TBD
    //! @param[in] parent TBD
    //---------------------------------------------------------------------------
    void AddEmitterToList( Emitter* emitter, Emitter* parent );

    //---------------------------------------------------------------------------
    //! @brief        エミッタをリストから削除します。
    //! @param[in] emitter TBD
    //---------------------------------------------------------------------------
    void RemoveEmitterFromList( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief        エミッタを削除します。
    //! @param[in] emitter TBD
    //---------------------------------------------------------------------------
    void _KillEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief      リソース更新に伴うアップデートを行います。
    //! @param[in]  emitterResSet TBD
    //! @return     成功した場合 true
    //---------------------------------------------------------------------------
    bool UpdateFromResource( EmitterResource* emitterResSet );

    //---------------------------------------------------------------------------
    //! @brief        リセット処理を行います。
    //---------------------------------------------------------------------------
    void Reset();

    //---------------------------------------------------------------------------
    //! @briefprivate   Initialize にも共通するリセット処理です。
    //---------------------------------------------------------------------------
    void ResetCommon();

    //---------------------------------------------------------------------------
    //! @briefprivate スケール更新
    //---------------------------------------------------------------------------
    inline void updateParticleScale_()
    {
        m_ParticleScaleForCalc.x = m_ParticleScale.x * m_AutoCalcScale.x;
        m_ParticleScaleForCalc.y = m_ParticleScale.y * m_AutoCalcScale.y;
        m_ParticleScaleForCalc.z = m_ParticleScale.z * m_AutoCalcScale.z;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate エミッタ計算処理
    //! @param[in] emitter TBD
    //! @param[in] swapBuffer TBD
    //! @param[in] emitterCalcLod TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    EmitterCalculationResult CalcEmitter( Emitter* emitter, bool swapBuffer, EmitterCalcLodCallback emitterCalcLod );

    //---------------------------------------------------------------------------
    //! @briefprivate エミッタ描画処理
    //! @param[in] emitter TBD
    //! @param[in] shaderType TBD
    //! @param[in] drawPath TBD
    //! @param[in] drawViewFlag TBD
    //! @param[in] calcStreamOut TBD
    //! @param[in] userParam TBD
    //! @param[in] emitterDrawCull TBD
    //! @param[in] profiler TBD
    //---------------------------------------------------------------------------
    void DrawEmitter( Emitter*                    emitter,
                      ShaderType                  shaderType,
                      u32                         drawPath,
                      u32                         drawViewFlag,
                      bool                        calcStreamOut,
                      void*                       userParam,
                      EmitterDrawCullingCallback  emitterDrawCull,
                      DrawEmitterProfilerCallback profiler );

    //---------------------------------------------------------------------------
    //! @briefprivate チャイルドエミッタセットの描画処理を行います。
    //! @param[in] beforeThanParent TBD
    //! @param[in] parentEmitter TBD
    //! @param[in] shaderType TBD
    //! @param[in] drawPath TBD
    //! @param[in] drawViewFlag TBD
    //! @param[in] calcStreamOut TBD
    //! @param[in] userParam TBD
    //! @param[in] emitterDrawCull TBD
    //! @param[in] profiler TBD
    //---------------------------------------------------------------------------
    void DrawChildEmitter(  bool                         beforeThanParent,
                            Emitter*                     parentEmitter,
                            ShaderType                   shaderType,
                            u32                          drawPath,
                            u32                          drawViewFlag,
                            bool                         calcStreamOut,
                            void*                        userParam,
                            EmitterDrawCullingCallback   emitterDrawCull,
                            DrawEmitterProfilerCallback  profiler );

    //---------------------------------------------------------------------------
    //! @briefprivate 情報更新
    //! @param[in] emitter TBD
    //---------------------------------------------------------------------------
    void UpdateProcessingInfo( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @briefprivate           引数で渡されるエミッタリストの終了処理を行います。
    //! @param[in] pEmitterHead 先頭エミッタへのポインタ
    //! @return                 成否判定。
    //---------------------------------------------------------------------------
    bool FinalizeEmitterList( Emitter* pEmitterHead );

private:
    u8                          m_IsLoopEffect;                     //!< ループエフェクトかどうか
    u8                          m_IsFade;                           //!< フェード中かどうか
    u8                          m_IsCalc;                           //!< 計算処理を行うかどうか
    u8                          m_IsDraw;                           //!< 描画処理を行うかどうか
    u8                          m_IsUsage;                          //!< 利用されているか
    u8                          m_IsSetDirectional;                 //!< 指定方向が設定されたか？
    u8                          m_IsManualEmission;                 //!< マニュアル放出モードか？
    u8                          m_IsDelayCreate;                    //!< 遅延生成されたか？
    u8                          m_GroupID;                          //!< グループID
    u8                          m_DrawPriority;                     //!< 描画プライオリティ
    u8                          m_IsEmitterSRTDirty;                //!< SRT行列のダーティフラグ
    u8                          dummy;                              //!< パディング
    System*                     m_System;                           //!< システムクラス
    u32                         m_EmitterSetID;                     //!< エミッタセットID
    u32                         m_EmitterSetCreateID;               //!< エミッタセット生成ID
    u32                         m_ResourceID;                       //!< リソースID
    u32                         m_RandomSeed;                       //!< ランダム種
    u32                         m_EmitterCreateID;                  //!< エミッタセット内、エミッタ生成ID
    u32                         m_ReqFrameBufferTexturePath;        //!< フレームバッファテクスチャを要求するパス
    u32                         m_ReqDepthBufferTexturePath;        //!< デプスバッファテクスチャを要求するパス
    u32                         m_ManualEmissionAssignNum;          //!< マニュアル放出でアサインされた数
    DrawViewFlag                m_DrawViewFlag;                     //!< ビューフラグ
    f32                         m_FrameRate;                        //!< フレームレート
    f32                         m_EmissionRatioScale;               //!< 放出レートスケール
    f32                         m_EmissionIntervalScale;            //!< 放出間隔スケール
    f32                         m_ParticleLifeScale;                //!< パーティクル寿命スケール
    nw::math::MTX34             m_MatrixSRT;                        //!< エミッタセットSRT行列
    nw::math::MTX34             m_MatrixRT;                         //!< エミッタセットRT行列
    nw::math::VEC3              m_EmitterVolumeScale;               //!< エミッタ形状のスケール
    nw::math::VEC3              m_AutoCalcScale;                    //!< 自動計算されるスケール
    nw::math::VEC4              m_Color;                            //!< エミッタセットカラー
    nw::math::VEC3              m_InitialRoate;                     //!< 初期角度
    nw::math::VEC3              m_ParticleScale;                    //!< パーティクルスケール
    nw::math::VEC3              m_ParticleEmissionScale;            //!< 放出時のみに効くパーティクルスケール
    nw::math::VEC3              m_ParticleScaleForCalc;             //!< 計算用のパーティクルスケール
    Emitter*                    m_EmitterHead;                      //!< エミッタツリー(子も含む)
    Emitter*                    m_EmitterTail;                      //!< エミッタツリー
    u32                         m_EmitterNum;                       //!< データ上のエミッタ数
    s32                         m_StartFrame;                       //!< 開始フレーム
    u32                         m_ProcessingEmitterNum;             //!< 処理されたエミッタ数
    u32                         m_CalcSkipEmitterNum;               //!< 計算処理をスキップされたエミッタ数
    u32                         m_ProcessingEmitterAnimNum;         //!< 処理されたエミッタアニメ数
    u32                         m_ProcessingCpuParticleNum;         //!< 処理されたCPUパーティクル数
    u32                         m_ProcessingGpuParticleNum;         //!< 処理されたGPUパーティクル数
    u32                         m_ProcessingGpuSoParticleNum;       //!< 処理されたGPUSOパーティクル数
    u32                         m_ProcessingCpuEmitterNum;          //!< 処理されたCpuエミッタ数
    u32                         m_ProcessingGpuEmitterNum;          //!< 処理されたGpuエミッタ数
    u32                         m_ProcessingGpuSoEmitterNum;        //!< 処理されたGpuSoエミッタ数
    u32                         m_ProcessingStripeNum;              //!< 処理されたストライプ本数
    u32                         m_ProcessingSuperStripeNum;         //!< 処理されたスーパーストライプ本数
    u32                         m_AllocedFromDynamicHeapSize;       //!< 動的ヒープから確保したサイズ
    void*                       m_RuntimeUserData;                  //!< ランタイムで自由に設定してよいユーザデータ
    u32                         m_EmitterFirstNum;                  //!< セット生成時に作成したエミッタ数
    EmitterSet*                 m_Next;                             //!< EmitterSet リスト
    EmitterSet*                 m_Prev;                             //!< EmitterSet リスト
    Heap*                       m_Heap;                             //!< 生成時にメモリを確保するヒープ
    f32                         m_FigureVel;                        //!< 形状で決まる速度のスケール値
    f32                         m_RandomVel;                        //!< 初期速度ランダムのスケール値
    nw::math::VEC3              m_VelAdd;                           //!< 最終的な初速に加算する値
    nw::math::VEC3              m_EmitterScale;                     //!< エミッタ形状のスケール
    f32                         m_DirectionalVel;                   //!< 指定方向速度のスケール値
    nw::math::VEC3              m_Directional;                      //!< 指定方向
    u32                         m_DrawPath;                         //!< 保持する全てのエミッタ描画パスフラグ
    EmitterSetResource*         m_EmitterSetRes;                    //!< エミッタセットリソース
    bool                        m_IsBufferFliped;                   //!< バッファをフリップしたかどうか
    EmitterSetInitializeCallback  m_InitializeCallback;             //!< Initialize時に呼ばれるコールバック
    EmitterSetFinalizeCallback  m_FinalizeCallback;                 //!< Finalize時に呼ばれるコールバック
    s32                         m_ResidentEmitterTime;              //!< マニュアル放出エミッタで最低限待機するフレーム数
    bool                        m_IsAnyEmitterFade;                 //!< 配下のエミッタが一つでもフェードしているかどうか

    friend class System;
    friend class EmitterCalc;
};

} // namespace eft2
} // namespace nw
