﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Render.h>
#include <nw/eft/eft2_VertexBuffer.h>
#include <nw/eft/eft2_TextureSampler.h>
#include <nw/eft/eft2_Data.h>

namespace nw   {
namespace eft2 {

class System;
struct Emitter;
struct EmitterResource;
struct ParticleAttribute;
struct Particle;
struct ParticleData;
struct ParentParticleData;
struct ParticleAttribute;
struct ParticlePosAttribute;
struct Shader;
struct Primitive;
struct EmitterAnimValue;

//---------------------------------------------------
//! @brief  パーティクル挙動計算関数ポインタ定義です。
//---------------------------------------------------
typedef void (*ParticleBehavior)( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec );

//---------------------------------------------------------------------------
//! @brief              パーティクルの挙動計算を行う 積算版
//! @param[in] emitter  エミッタへのポインタ
//! @param[in] index  パーティクルインデックス
//! @param[in,out] pos  位置
//! @param[in,out] vec  速度
//! @param[in] posBack  位置（バックバッファ）
//! @param[in] vecBack  速度（バックバッファ）
//! @param[in] time     パーティクル時刻
//---------------------------------------------------------------------------
void _CalcParticleBehavior_Integrating( Emitter* emitter, s32 index, nw::math::VEC4* pos, nw::math::VEC4* vec, const nw::math::VEC4* posBack, const nw::math::VEC4* vecBack, const f32 time );

//---------------------------------------------------
//! @brief エミッタ挙動計算クラス
//---------------------------------------------------
class EmitterCalc
{
public:
    //---------------------------------------------------------------------------
    //! @brief              コンストラクタ。
    //! @param[in] system   システムへのポインタ
    //---------------------------------------------------------------------------
    explicit EmitterCalc( System* system );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ。
    //---------------------------------------------------------------------------
    ~EmitterCalc();

    //---------------------------------------------------------------------------
    //! @brief                  エミッタ計算処理をします。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] frameRate    フレームレート値
    //! @param[in] swapBuffer   バッファをスワップするか
    //! @param[in] isFade       フェード中か
    //! @param[in] isEmission   TBD
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool Calc( Emitter* emitter, f32 frameRate, bool swapBuffer, bool isFade, bool isEmission );

    //---------------------------------------------------------------------------
    //! @brief              パーティクル計算処理をします。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void CalcParticle( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief                  エミッタ描画処理をします。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] shaderType   シェーダタイプ
    //! @param[in] userParam    ユーザーパラメータへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool Draw( Emitter* emitter, ShaderType shaderType, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief                  既にシェーダがバインド済みの前提で、エミッタ描画処理をします。（複数回描画処理をするため）
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] shaderType   シェーダタイプ
    //! @param[in] userParam    ユーザーパラメータへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool DrawEmitterUsingBoundShader( Emitter* emitter, ShaderType shaderType, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief                      エミッタストリームアウト計算処理をします。
    //! @param[in] emitter          エミッタへのポインタ
    //! @param[in] shader           シェーダへのポインタ
    //! @param[in] globalCounter    グローバルカウンタ
    //! @param[in] calcStreamOut    ストリームアウトの計算をするかどうか
    //! @param[in] userParam        ユーザーパラメータへのポインタ
    //! @return                     成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool CalcStreamOut( Emitter* emitter, Shader* shader, u32 globalCounter, bool calcStreamOut, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief                  放出処理を行います。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitNum      放出数
    //! @param[in] forceSearch  空きパーティクルを強制的に検索するかどうか
    //! @return                 最後に追加されたパーティクルのインデックス値
    //---------------------------------------------------------------------------
    s32 Emit( Emitter* emitter, u32 emitNum, bool forceSearch = false );

    //---------------------------------------------------------------------------
    //! @brief                      パーティクルを初期化します。
    //! @param[in] emitter          エミッタへのポインタ
    //! @param[in] attrIndex        アトリビュートインデックス
    //! @param[in] ptclData         パーティクルデータへのポインタ
    //! @param[in] parentPtclData   親パーティクル固有のデータへのポインタ
    //! @param[in] ptclAttr         パーティクルアトリビュートへのポインタ
    //! @param[in] ptclPosAttr      パーティクル位置アトリビュートへのポインタ
    //! @param[in] emitIndex        自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax          放出の最大数
    //! @param[in] randomValue      ランダム値
    //! @return                     成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool InitializeParticle( Emitter*               emitter,
                             u32                    attrIndex,
                             ParticleData*          ptclData,
                             ParentParticleData*    parentPtclData,
                             ParticleAttribute*     ptclAttr,
                             ParticlePosAttribute*  ptclPosAttr,
                             u32                    emitIndex,
                             u32                    emitMax,
                             f32                    randomValue );

/*
    //---------------------------------------------------------------------------
    //! @brief        エミッタ移動時の等距離放出を行います。
    //---------------------------------------------------------------------------
    void EmitSameDistance( const SimpleEmitterData * __restrict res, EmitterInstance * e );
*/

    //---------------------------------------------------------------------------
    //  エミッタの動的ユニフォームブロックを生成します。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] frameRate    フレームレート
    //---------------------------------------------------------------------------
    void MakeDynamicUniformBlock( Emitter* emitter, f32 frameRate );

private:

    //---------------------------------------------------------------------------
    //! @brief              パーティクル情報を計算する
    //! @param[out] dstPtcl 出力先のパーティクル情報
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] time     時刻
    //! @param[in] life     寿命
    //! @param[in] index    自身が格納されるパーティクル配列のインデックス
    //---------------------------------------------------------------------------
    void CalcParticleInfo( Particle* dstPtcl, Emitter* emitter, f32 time, f32 life, u32 index );

    //---------------------------------------------------------------------------
    //! @brief                  親パーティクルからの継承処理を行う
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] writeIndex   自身が格納されるパーティクル配列のインデックス
    //---------------------------------------------------------------------------
    void InheritParentParticleInfo( Emitter* emitter, u32 writeIndex );

    //---------------------------------------------------------------------------
    //! @brief                  検索して死亡枠があればそこに追加、死亡枠が無い場合は後続に追加。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitNum      放出数
    //! @param[in] forceSearch  空きパーティクルを強制的に検索するかどうか
    //! @return                 最後に追加されたパーティクルのインデックス値
    //---------------------------------------------------------------------------
    s32 _Emit_SearchOrder( Emitter* emitter, u32 emitNum, bool forceSearch );

    //---------------------------------------------------------------------------
    //! @brief              生成順に詰め込む。
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] emitNum  放出数
    //! @return             最後に追加されたパーティクルのインデックス値
    //---------------------------------------------------------------------------
    s32 _Emit_SortOrder( Emitter* emitter, u32 emitNum );

    //---------------------------------------------------------------------------
    //! @brief              ソートなし描画処理を行います。
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] shader   シェーダへのポインタ
    //! @param[in] prim     プリミティブへのポインタ
    //! @param[in] primType プリミティブタイプ
    //! @return             成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool EntryParticleNoSort( Emitter* emitter, Shader* shader, Primitive* prim, Render::PrimitiveType primType );

    //---------------------------------------------------------------------------
    //! @brief              ソート済みパーティクル描画処理を行います。
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] shader   シェーダへのポインタ
    //! @param[in] prim     プリミティブへのポインタ
    //! @param[in] primType プリミティブタイプ
    //! @param[in] sortType ソートタイプ
    //! @return             成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool EntrySortedParticle( Emitter* emitter, Shader* shader, Primitive* prim, Render::PrimitiveType primType, const ParticleSortType sortType );


    //---------------------------------------------------------------------------
    //! @brief                  点エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitPoint( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  円エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitCircle( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  円等分割エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitCircleEqualDivision( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief        円フィルエミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitCircleFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  球エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitSphere( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  球 32等分割エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitSphereEqualDivision32( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  球 64等分割エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitSphereEqualDivision64( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  球 フィルエミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitSphereFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  シリンダー エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitCylinder( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  シリンダーフィル エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitCylinderFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  ボックス エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitBox( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  ボックス フィルエミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitBoxFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  ライン エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitLine( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  ライン 等分割エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitLineEqualDivision( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  矩形 エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitRectangle( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------------------------------
    //! @brief                  プリミティブ エミッタ計算処理を行います。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    static bool CalcEmitPrimitive( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------
    //! @briefprivate           エミッタ形状算出 関数ポインタ定義です。
    //! @param[in,out] localPos ローカル位置
    //! @param[in,out] localVec ローカル速度
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] emitIndex    自身が格納されるパーティクル配列のインデックス
    //! @param[in] emitMax      放出される最大数
    //! @param[in] randomValue  ランダム値
    //! @param[in] emitterAnim  エミッタアニメへのポインタ
    //! @return                 成功した場合trueを返す。
    //---------------------------------------------------
    typedef bool (*EmitFunction)( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim );

    //---------------------------------------------------
    //! @brief       エミッタ形状処理 関数テーブル
    //---------------------------------------------------
    static EmitFunction m_EmitFunctions[];

    //------------------------------------------------------------------------------
    //! @brief          平方根を計算
    //! @param[in] v    入力
    //! @return         平方根の値
    //------------------------------------------------------------------------------
    static inline f32 SafeSqrt( const f32 v )
    {
        if( v <= 0.0f ){ return 0.0f; }
        return nw::math::FSqrt( v );
    }

private:
    System*             m_System;                               //!< エフェクトシステム
    VertexBuffer        m_PositionBuffer;                       //!< Position バッファ
    VertexBuffer        m_IndexBuffer;                          //!< Index バッファ
    TextureSampler      m_TextureSamplerForCurlNoise;           //!< フレームバッファ/デプス テクスチャ用のテクスチャサンプラオブジェクト
    ResTextureSampler   m_ResTextureSamplerForCurlNoise;        //!< フレームバッファ/デプス テクスチャ用のテクスチャサンプラリソース
    TextureSampler      m_TextureSamplerForColorAndDepth;       //!< フレームバッファ/デプス テクスチャ用のテクスチャサンプラオブジェクト
    ResTextureSampler   m_ResTextureSamplerForColorAndDepth;    //!< フレームバッファ/デプス テクスチャ用のテクスチャサンプラリソース
};

//---------------------------------------------------
//! @brief                  回転行列を作成する（回転の適用順: Y->Z->X）
//! @param[out] dstMatrix   出力を格納するMTX44へのポインタ
//! @param[in] rotate       入力となる回転値を収めたVEC3
//---------------------------------------------------
void MakeRotationMatrixYZX( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate );
//---------------------------------------------------
//! @brief                  回転行列を作成する（回転の適用順: X->Y->Z）
//! @param[out] dstMatrix   出力を格納するMTX44へのポインタ
//! @param[in] rotate       入力となる回転値を収めたVEC3
//---------------------------------------------------
void MakeRotationMatrixXYZ( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate );
//---------------------------------------------------
//! @brief                  回転行列を作成する（回転の適用順: Z->X->Y）
//! @param[out] dstMatrix   出力を格納するMTX44へのポインタ
//! @param[in] rotate       入力となる回転値を収めたVEC3
//---------------------------------------------------
void MakeRotationMatrixZXY( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate );

//---------------------------------------------------
//! @brief                  現在のパーティクル回転値を計算する
//! @param[out] dstRotate   出力を収めるVEC4へのポインタ
//! @param[in] emitterRes   エミッタリソースへのポインタ
//! @param[in] initRotate   初期回転値
//! @param[in] random       ランダム
//! @param[in] time         現在時刻
//---------------------------------------------------
void CalcRotationMatrix( nw::math::VEC4*       dstRotate,
                        const EmitterResource* emitterRes,
                        const nw::math::VEC4*  initRotate,
                        const nw::math::VEC4*  random,
                        f32                    time );

//---------------------------------------------------
//! @brief  現在のパーティクル回転値を計算する
//---------------------------------------------------
void CalcRotationMatrix( nw::math::VEC4*        rotateParam,
                         const EmitterResource* emitterRes,
                         const nw::math::VEC4*  random,
                         f32                    time );


//---------------------------------------------------
//! @brief                  現在のパーティクルスケールを計算する
//! @param[out] dstScale    出力を収めるVEC4へのポインタ
//! @param[in] emitterRes   エミッタリソースへのポインタ
//! @param[in] initScale    初期スケール
//! @param[in] random       ランダム値
//! @param[in] life         寿命
//! @param[in] time         現在時刻
//---------------------------------------------------
void CalcPtclScaleVecFromTime( nw::math::VEC4*         dstScale,
                               const EmitterResource*  emitterRes,
                               const nw::math::VEC4*   initScale,
                               const nw::math::VEC4*   random,
                               f32                     life,
                               f32                     time );

//---------------------------------------------------
//! @brief                  現在のパーティクルスケールを計算する
//! @param[out] dstScale    出力を収めるVEC4へのポインタ
//! @param[in] emitterRes   エミッタリソースへのポインタ
//! @param[in] initScale    初期スケール
//! @param[in] random       ランダム値
//! @param[in] life         寿命
//! @param[in] time         現在フレーム
//---------------------------------------------------
void CalcPtclScaleVecPerFrame( nw::math::VEC4*         dstScale,
                               const EmitterResource*  emitterRes,
                               const nw::math::VEC4*   initScale,
                               const nw::math::VEC4*   random,
                               f32                     life,
                               f32                     time );

//---------------------------------------------------
//! @brief                      現在のパーティクルカラー0を計算する
//! @param[out] dstColor        出力を収めるVEC4へのポインタ
//! @param[in] emitterRes       エミッタリソースへのポインタ
//! @param[in] random           ランダム値
//! @param[in] emitterColor     エミッタカラー
//! @param[in] emitterAnimColor カラーのエミッタアニメ
//! @param[in] emitterAnimAlpha アルファのエミッタアニメ
//! @param[in] life             寿命
//! @param[in] time             現在時刻
//---------------------------------------------------
void CalcPtclColor0Vec( nw::math::VEC4*         dstColor,
                        const EmitterResource*  emitterRes,
                        const nw::math::VEC4*   random,
                        const nw::math::VEC4*   emitterColor,
                        const nw::math::VEC3*   emitterAnimColor,
                        f32                     emitterAnimAlpha,
                        f32                     life,
                        f32                     time );

//---------------------------------------------------
//! @brief                      現在のパーティクルカラー1を計算する
//! @param[out] dstColor        出力を収めるVEC4へのポインタ
//! @param[in] emitterRes       エミッタリソースへのポインタ
//! @param[in] random           ランダム値
//! @param[in] emitterColor     エミッタカラー
//! @param[in] emitterAnimColor カラーのエミッタアニメ
//! @param[in] emitterAnimAlpha アルファのエミッタアニメ
//! @param[in] life             寿命
//! @param[in] time             現在時刻
//---------------------------------------------------
void CalcPtclColor1Vec( nw::math::VEC4*        dstColor,
                       const EmitterResource*  emitterRes,
                       const nw::math::VEC4*   random,
                       const nw::math::VEC4*   emitterColor,
                       const nw::math::VEC3*   emitterAnimColor,
                       f32                     emitterAnimAlpha,
                       f32                     life,
                       f32                     time );


} // namespace eft2
} // namespace nw
