﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Particle.h>
#include <nw/eft/eft2_Random.h>
#include <nw/eft/eft2_EmitterRes.h>
#include <nw/eft/eft2_StreamOutBuffer.h>
#include <nw/ut/os/ut_Time.h>

namespace nw   {
namespace eft2 {

class  System;
struct EmitterSet;
class  EmitterCalc;
struct Shader;

struct CallbackSet;
struct RenderStateSetArg;
typedef bool (*RenderStateSetCallback)( RenderStateSetArg& arg );
typedef void (*DrawPathRenderStateSetCallback)( RenderStateSetArg& arg );

//------------------------------------------------------------------------------
//! @brief  エミッタ時間アニメ 計算結果格納用構造体
//------------------------------------------------------------------------------
struct EmitterAnimValue
{
    nw::math::VEC3_     vscale;                 //!< スケール
    nw::math::VEC3_     vrotate;                //!< 回転
    nw::math::VEC3_     vtrans;                 //!< 平行移動
    nw::math::VEC3_     vcolor0;                //!< カラー0
    nw::math::VEC3_     vcolor1;                //!< カラー1
    union
    {
        nw::math::VEC3_ vemissionRate;          //!< 放出レート
        f32             emissionRate;           //!< 放出レート
    };
    union
    {
        nw::math::VEC3_ vparticleLife;          //!< パーティクル寿命
        f32             particleLife;           //!< パーティクル寿命
    };
    nw::math::VEC3_     valpha0;                //!< アルファ0
    nw::math::VEC3_     valpha1;                //!< アルファ1
    union
    {
        nw::math::VEC3_ vallVelDir;             //!< 全方向初速
        f32             allVelDir;              //!< 全方向初速
    };
    union
    {
        nw::math::VEC3_ vdesignatedDirScale;    //!< 指定方向初速
        f32             designatedDirScale;     //!< 指定方向初速
    };
    nw::math::VEC3_     vparticleScale;         //!< パーティクルスケール
    nw::math::VEC3_     vvolumeScale;           //!< エミッタ形状スケール
    union
    {
        nw::math::VEC3_ vgravityScale;          //!< 重力スケール
        f32             gravityScale;           //!< 重力スケール
    };
};


//------------------------------------------------------------------------------
//! @brief  エミッタ
//------------------------------------------------------------------------------
struct Emitter
{
    u8                              isChild;                                    //!< 子エミッタかどうか
    u8                              isEmit;                                     //!< 放出済みかどうか？
    u8                              isVisible;                                  //!< 表示/非表示
    u8                              groupID;                                    //!< グループID
    u8                              calcedEmitterAnimCount;                     //!< 計算処理されたエミッタアニメ数
    u8                              isEnableEmit;                               //!< 放出が有効かどうか
    u8                              isEmitterInitializeFailed;                  //!< eft2ランタイム側でエミッタ初期化失敗したか
    u8                              m_IsKillReservation;                                                    //!< 削除予約
    bool                            aliveChildren;                              //!< 子エミッタ生存フラグ
    u32                             emitterSetCreateID;                         //!< 親エミッタセット生成ID
    u32                             emitterCreateID;                            //!< エミッタ生成ID
    u32                             particleCreateID;                           //!< パーティクル生成ID
    s32                             childResIndex;                              //!< チャイルドリソースインデックス
    u32                             ptclMaxAssignmentNum;                       //!< アプリ側から指定されたパーティクル最大数
    u32                             ptclNum;                                    //!< 放出中のパーティクル数
    u32                             ptclProcessingNum;                          //!< 今フレーム計算処理したパーティクル数
    u32                             ptclProcessingEnd;                          //!< バッファ内で利用されている最終パーティクルインデックス
    u32                             bufferID;                                   //!< バッファID
    u32                             soCounter;                                  //!< ストリームアウトを二重実行しない為のカウンタ
    u32                             primEmitCounter;                            //!< 放出形状 プリミティブ用カウンタ
    u32                             calcSkipFlag;                               //!< 計算処理フラグ
    DrawViewFlag                    drawViewFlag;                               //!< ビューフラグ
    f32                             time;                                       //!< エミッタ時間
    f32                             frameRate;                                  //!< フレームレート
    f32                             emitCnt;                                    //!< エミッタカウンタ
    f32                             emitSaving;                                 //!< 放出貯留
    f32                             emitInterval;                               //!< 放出間隔
    f32                             emitRatio;                                  //!< 放出レート係数
    f32                             emitIntervalRatio;                          //!< 放出間隔係数
    f32                             emitLastFrame;                              //!< 最終放出フレーム
    f32                             particleLifeScale;                          //!< パーティクルライフスケール
    f32                             fadeOutAlpha;                                  //!< フェードアウトα値
    f32                             fadeInAlpha;                                //!< フェードインα値
    EmitterSet*                     emitterSet;                                 //!< 親エミッタセット
    Emitter*                        next;                                       //!< Next Emitter
    Emitter*                        prev;                                       //!< Prev Emitter
    Emitter*                        nextSoEmitter;                              //!< ストリームアウトエミッタリスト用
    Emitter*                        childHead[EFT_EMITTER_INSET_NUM];           //!< チャイルドエミッタリスト
    Emitter*                        childTail[EFT_EMITTER_INSET_NUM];           //!< チャイルドエミッタリスト
    Emitter*                        nextChild;                                  //!< チャイルドエミッタ
    EmitterCalc*                    emitterCalc;                                //!< エミッタ挙動計算クラス
    ResEmitter*                     emitterData;                                //!< エミッタリソース
    DynamicUniformBlockBuffer       dynamicUBO[EFT_BUFFER_ID_TRIPLE_MAX];       //!< 動的ユニフォームブロック
    Random                          random;                                     //!< ランダム
    ParticleData*                   particleData;                               //!< パーティクルデータ
    ParentParticleData*             parentParticleData;                         //!< 親パーティクル固有のデータ
    ParticlePosAttribute*           particlePosAttr;                            //!< パーティクル位置情報アトリビュート（表：現在使用するバッファ）
    ParticlePosAttribute*           particlePosAttrBack;                        //!< パーティクル位置情報アトリビュート（裏：前回使用したバッファ）
    ParticlePosAttribute*           particlePosAttrOrg[EFT_BUFFER_ID_TRIPLE_MAX];//!< パーティクル位置情報アトリビュート
    ParticleAttribute*              particleAttr;                               //!< パーティクル情報アトリビュート（表：現在使用するバッファ）
    ParticleAttribute*              particleAttrBack;                           //!< パーティクル情報アトリビュート（裏：前回使用したバッファ）
    ParticleAttribute*              particleAttrOrg[EFT_BUFFER_ID_TRIPLE_MAX];  //!< パーティクル情報アトリビュート
    u32                             ptclAttrFillMax;                            //!< パーティクル描画アトリビュートバッファ Fill可能数
    u32                             ptclAttrFillIndex;                          //!< パーティクル描画アトリビュートバッファ 現状のFillインデックス
    void*                           allocedFromDynamicHeap;                     //!< 動的バッファから確保したメモリ
    u32                             allocedFromDynamicHeapSize;                 //!< 動的バッファから確保したメモリサイズ
    Shader*                         shader[EFT_SHADER_TYPE_MAX];                //!< シェーダ
    const EmitterResource*          emitterRes;                                 //!< エミッタリソースセット
    const EmitterResource*          childEmitterResSet[EFT_EMITTER_INSET_NUM];  //!< 子エミッタリソースセット
    nw::math::MTX34                 resMatrixSRT;                               //!< エミッタマトリクス
    nw::math::MTX34                 resMatrixRT;                                //!< エミッタマトリクスRT
    nw::math::MTX34                 matrixSRT;                                  //!< エミッタマトリクス
    nw::math::MTX34                 matrixRT;                                   //!< エミッタマトリクスRT
    nw::math::VEC3                  emitterPrevPos;                             //!< １フレーム前のエミッタ位置
    nw::math::VEC3                  emitterLocalVec;                            //!< １フレーム前のエミッタ速度
    u32                             manualEmissionNum;                          //!< 今フレームでマニュアルエミッションされた数
    CallbackSet*                    callbackSet[EFT_CALLBACK_SET_TYPE_MAX];     //!< コールバック 0:CustomAction 1:CustomShader
    DrawPathRenderStateSetCallback  drawPathCallback;                           //!< 描画パスコールバック
    u32                             drawPath;                                   //!< 描画パス
    Heap*                           heap;                                       //!< 生成時に指定されたheap
    StreamOutAttributeBuffer        streamOutPos;                               //!< ストリームアウト管理クラス
    StreamOutAttributeBuffer        streamOutVec;                               //!< ストリームアウト管理クラス
    u32                             streamOutFlip;                              //!< ストリームアウトフリップ
    void*                           stripeDelayList;                            //!< 遅延ストライプリスト
    void*                           stripeVertexBuffer;                         //!< ストライプ頂点バッファ
    u32                             stripeVertexBufferStartIdx[EFT_BUFFER_ID_TRIPLE_MAX];   //!< ストライプ頂点バッファの開始インデックス
    u32                             stripeVertexBufferEndIdx[EFT_BUFFER_ID_TRIPLE_MAX];     //!< ストライプ頂点バッファの終了インデックス
    u32                             stripeVertexBufferCurIdx;                   //!< ストライプ頂点バッファの現インデックス
    u32                             stripeVertexBufferNum;                      //!< ストライプ頂点バッファの数
    u32                             stripeVertexBufferRenderNum;                //!< ストライプ頂点バッファの描画数

    void*                           customActionUserData;                       //!< カスタムアクションユーザーデータ
    void*                           customShaderUserData;                       //!< カスタムシェーダユーザーデータ

    Emitter*                        parentEmitter;                              //!< 親エミッタ
    u32                             parentEmitterCreateID;                      //!< 親エミッタの生成ID
    u32                             parentParticleCreateID;                     //!< 親パーティクルの生成ID
    u32                             parentParticleAttrIdx;                      //!< 親となるパーティクルのインデックス
    f32                             parentParticleLife;                         //!< 親パーティクルの寿命
    f32                             parentParticleBirthTime;                    //!< 親パーティクルの生成時間
    nw::math::VEC3                  parentParticleLocalPos;                     //!< 親となるパーティクルのローカル座標
    nw::math::VEC3                  parentParticleLocalVec;                     //!< 親となるパーティクルのローカル速度

    nw::math::VEC4                  parentParticleScale;                        //!< 親となるパーティクルのスケール値
    nw::math::VEC4                  parentParticleRotate;                       //!< 親となるパーティクルの回転値
    nw::math::VEC4                  parentParticleRandom;                       //!< 親となるパーティクルのランダム値

    f32                             parentParticleTime;                         //!< 親となるパーティクルの時間
    nw::math::VEC3                  parentParticleWorldPos;                     //!< 親となるパーティクルのワールド座標
    nw::math::VEC3                  parentParticleWorldVec;                     //!< 親となるパーティクルのワールドベクトル

    nw::math::VEC4                  color0;                                     //!< エミッタカラー0
    nw::math::VEC4                  color1;                                     //!< エミッタカラー1

    nw::math::VEC3                  manualEmitCommonOffset;                     //!< マニュアル放出時の共通オフセット
    nw::math::VEC3*                 manualEmitPoints;                           //!< マニュアル放出時のオフセット配列

    u32                             processEmitterFlag;                         //!< 処理するストリームアウトエミッタのフラグ値

    union
    {
        EmitterAnimValue            m_EmitterAnim;                              //!< エミッタ時間アニメ計算結果格納変数
        nw::math::VEC3_             m_EmitterAnimV[EFT_EMITTER_ANIM_MAX];       //!< エミッタ時間アニメ計算結果格納変数
    };
    bool                            m_EmitterAnimEnd[EFT_EMITTER_ANIM_MAX];     //!< エミッタ時間アニメの終了判定
    bool                            m_IsSoloFade;                               //!< （エミッタ単独で） フェード中かどうか
    nw::ut::Tick                    m_CpuCost;                                  //!< Calc の CPU 負荷
public:

    //---------------------------------------------------------------------------
    //! @brief  エミッタをフェードさせます。
    //---------------------------------------------------------------------------
    inline void Fade()
    {
        m_IsSoloFade = true;
    }

    //---------------------------------------------------------------------------
    //! @brief  エミッタを削除します。
    //!         削除処理は、次のエミッタ計算処理内で行われます。
    //---------------------------------------------------------------------------
    void Kill()
    {
        m_IsKillReservation = true;
    }

    //---------------------------------------------------------------------------
    //! @brief  生死判定を取得します。
    //! @return 生存中の場合trueを返す。
    //---------------------------------------------------------------------------
    inline bool IsAlive() const { return emitterCalc ? true : false; }

    //---------------------------------------------------------------------------
    //! @brief  表示/非表示フラグを取得します。
    //! @return 表示中の場合trueを返す。
    //---------------------------------------------------------------------------
    inline bool IsVisible() const { return ( isVisible != 0 ); }

    //---------------------------------------------------------------------------
    //! @brief  マニュアルエミッタが消えても大丈夫な状態かを取得します。
    //! @return 待機時間を過ぎている、もしくは常駐型のエミッタで、処理したパーティクルが無ければ true それ以外は false
    //---------------------------------------------------------------------------
    bool IsManualEmitterReadyToExit() const;

    //---------------------------------------------------------------------------
    //! @brief              放出処理の有効無効を切り替える。
    //!
    //!                     放出処理を無効に切り替え、放出済みパーティクルの再生が終えた場合、
    //!                     エミッタの再生は終了しません。
    //! @param[in] enable   有効無効の指定
    //---------------------------------------------------------------------------
    inline void SetEmissionProcessing( bool enable )
    {
        isEnableEmit = enable;
    }

    //---------------------------------------------------------------------------
    //! @brief  描画可能な状態かどうか。
    //! @return 描画可能な場合trueを返す。
    //---------------------------------------------------------------------------
    inline bool IsRenderAvailable() const
    {
        if ( dynamicUBO[bufferID].IsValidate() &&
             emitterData->emitter.isParticleDraw &&
             isVisible &&
             fadeOutAlpha > 0 &&
             ( ptclNum > 0 || ( ptclNum == 0 && ( 1 <= emitterRes->emitterPluginIndex && emitterRes->emitterPluginIndex <= 3 ) ) ) )
        {
            return true;
        }
        return false;
    }

    //------------------------------------------------------------------------------
    //! @brief              ワールド座標変化を行います。
    //!
    //!                     pSrc をエミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] pSrc     入力となるVEC3
    //------------------------------------------------------------------------------
    inline void TransformWorldVec( nw::math::VEC3* pDst, const nw::math::VEC3* pSrc ) const
    {
        VEC3Transform( pDst, &matrixSRT, pSrc );
    }

    //------------------------------------------------------------------------------
    //! @brief              ワールド法線を取得します。
    //!
    //!                     pSrc を追従設定を考慮して、エミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] pSrc     入力となるVEC3
    //------------------------------------------------------------------------------
    inline void TransformWorldVecNormal( nw::math::VEC3* pDst, const nw::math::VEC3* pSrc ) const
    {
        VEC3TransformNormal( pDst, &matrixSRT, pSrc );
    }

    //------------------------------------------------------------------------------
    //! @brief              ローカルベクトルを取得します。
    //!
    //!                     pSrc を追従設定を考慮して、エミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] pSrc     入力となるVEC3
    //------------------------------------------------------------------------------
    inline void TransformLocalVec( nw::math::VEC3* pDst, const nw::math::VEC3* pSrc ) const
    {
        nw::math::MTX34 invMat;
        MTX34Inverse( &invMat, &matrixSRT );
        VEC3Transform( pDst, &invMat, pSrc );
    }

    //------------------------------------------------------------------------------
    //! @brief              ローカル法線を取得します。
    //!
    //!                     pSrc を追従設定を考慮して、エミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] pSrc     入力となるVEC3
    //------------------------------------------------------------------------------
    inline void TransformLocalVecNormal( nw::math::VEC3* pDst, const nw::math::VEC3* pSrc ) const
    {
        nw::math::MTX34 invMat;
        MTX34Inverse( &invMat, &matrixSRT );
        VEC3TransformNormal( pDst, &invMat, pSrc );
    }

    //------------------------------------------------------------------------------
    //! @brief              方向ベクトルのローカル座標変化を行います。
    //!
    //!                     pSrc をエミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //------------------------------------------------------------------------------
    void TransformToLocalVec( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const;

    //------------------------------------------------------------------------------
    //! @brief              方向ベクトルのローカル座標変化を行います。
    //!
    //!                     pSrc をエミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //! @param[in] attr     パーティクルアトリビュートへのポインタ
    //------------------------------------------------------------------------------
    void TransformToLocalVec( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const;

    //------------------------------------------------------------------------------
    //! @brief              方向ベクトルのワールド座標変換を行います。
    //!
    //!                     pSrc をワールド空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //------------------------------------------------------------------------------
    void TransformToWorldVec( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const;

    //------------------------------------------------------------------------------
    //! @brief              方向ベクトルのワールド座標変換を行います。
    //!
    //!                     pSrc をワールド空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //! @param[in] attr     パーティクルアトリビュートへのポインタ
    //------------------------------------------------------------------------------
    void TransformToWorldVec( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const;

    //------------------------------------------------------------------------------
    //! @brief              位置ベクトルのローカル座標変換を行います。
    //!
    //!                     pSrc をエミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //------------------------------------------------------------------------------
    void TransformToLocalPos( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const;

    //------------------------------------------------------------------------------
    //! @brief              位置ベクトルのローカル座標変換を行います。
    //!
    //!                     pSrc をエミッタマトリクス空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //! @param[in] attr     パーティクルアトリビュートへのポインタ
    //------------------------------------------------------------------------------
    void TransformToLocalPos( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const;

    //------------------------------------------------------------------------------
    //! @brief              位置ベクトルのワールド座標変換を行います。
    //!
    //!                     pSrc をワールド空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //------------------------------------------------------------------------------
    void TransformToWorldPos( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const;

    //------------------------------------------------------------------------------
    //! @brief              位置ベクトルのワールド座標変換を行います。
    //!
    //!                     pSrc をワールド空間へ変換し pDst に出力します。
    //! @param[out] pDst    変換後のVEC3へのポインタ
    //! @param[in] src      入力となるVEC3
    //! @param[in] attr     パーティクルアトリビュートへのポインタ
    //------------------------------------------------------------------------------
    void TransformToWorldPos( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const;

    //------------------------------------------------------------------------------
    //! @brief                  パーティクル位置をワールド座標系に変換するための、追従設定を加味した変換行列を取得します。
    //! @param[out] pOutMatrix  変換用の行列へのポインタ
    //! @param[in]  index       パーティクルインデックス
    //------------------------------------------------------------------------------
    void GetEmitterTransformMatrix( nw::math::MTX34* pOutMatrix, u32 index ) const;

    //---------------------------------------------------------------------------
    //! @brief  エミッタがフレームバッファテクスチャを要求するか。
    //! @return 要求する場合trueを返す。
    //---------------------------------------------------------------------------
    bool IsRequestFrameBufferTexture() const;

    //---------------------------------------------------------------------------
    //! @brief  エミッタがデプスバッファテクスチャを要求するか。
    //! @return 要求する場合trueを返す。
    //---------------------------------------------------------------------------
    bool IsRequestDepthTexture() const;

    //---------------------------------------------------------------------------
    //! @brief     カラー0 RGBA値(乗算値)を設定します。
    //! @param[in] color0 カラー0値
    //---------------------------------------------------------------------------
    void SetColor0( const nw::math::VEC4 &color0 );

    //---------------------------------------------------------------------------
    //! @brief     カラー1 RGBA値(乗算値)を設定します。
    //! @param[in] color1 カラー1値
    //---------------------------------------------------------------------------
    void SetColor1( const nw::math::VEC4 &color1 );

    //---------------------------------------------------------------------------
    //! @brief              放出レートスケール設定を行います。
    //!
    //!                     ratio に、1.0以上の値は指定できません。
    //! @param[in] ratio    放出レートスケール値（0.0～1.0）
    //---------------------------------------------------------------------------
    void SetEmissionRatio( f32 ratio )
    {
        emitRatio = ratio;
        if ( emitRatio > 1.0f ) emitRatio = 1.0f;
    }

    //---------------------------------------------------------------------------
    //! @brief              放出間隔スケール設定を行います。
    //! @param[in] ratio    放出間隔スケール値（0.0～1.0）
    //---------------------------------------------------------------------------
    void SetEmissionInterval( f32 ratio )
    {
        emitIntervalRatio = ratio;
        if ( emitIntervalRatio < 1.0f ) emitIntervalRatio = 1.0f;
    }

    //---------------------------------------------------------------------------
    //! @brief  パーティクル寿命のスケール値を設定します。
    //! @param[in] ratio 寿命スケール値（0.0～1.0）
    //---------------------------------------------------------------------------
    void SetParticleLifeScale( f32 ratio )
    {
        particleLifeScale = ratio;
        if ( particleLifeScale > 1.0f ) particleLifeScale = 1.0f;
    }

    //---------------------------------------------------------------------------
    //! @brief          描画パスを設定します。
    //! @param[in] path 描画パスの値
    //---------------------------------------------------------------------------
    void SetDrawPath( u32 path );

    //---------------------------------------------------------------------------
    //! @brief          処理するストリームアウトエミッタのフラグ値を設定します。
    //! @param[in] flag 処理するストリームアウトエミッタのフラグの値
    //---------------------------------------------------------------------------
    void SetProcessEmitterFlag( u32 flag ) { processEmitterFlag = flag; }

    //---------------------------------------------------------------------------
    //! @brief                  内部バッファをスワップします。
    //! @param[in] bufferMode   バッファモード
    //---------------------------------------------------------------------------
    void Swap( BufferMode bufferMode );

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダパラメータを取得します。
    //! @return カスタムシェーダパラメータ値
    //---------------------------------------------------------------------------
    void* GetCustomShaderParameter() const { return emitterRes->customShaderParam; }

    //---------------------------------------------------------------------------
    //! @brief  カスタムアクションパラメータを取得します。
    //! @return カスタムアクションパラメータ値
    //---------------------------------------------------------------------------
    void* GetCustomActionParameter() const { return emitterRes->customActionParam; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタ拡張パラメータを取得します。
    //! @return エミッタ拡張パラメータ値
    //---------------------------------------------------------------------------
    void* GetCustomDataParameter() const { return emitterRes->customDataParam; }

    //---------------------------------------------------------------------------
    //! @brief          カスタムシェーダパラメータをインデックスを指定して取得します。
    //! @param[in] idx  カスタムシェーダパラメータのインデックス値
    //! @return         idx で指定したカスタムシェーダパラメータ
    //---------------------------------------------------------------------------
    f32 GetCustomShaderParameter( u32 idx ) const
    {
        if ( !emitterRes->customShaderParam ) return 0.0f;
        f32* fparam = reinterpret_cast<f32*>( emitterRes->customShaderParam );
        return fparam[idx];
    }

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダフラグを取得します。
    //! @return カスタムシェーダフラグ
    //---------------------------------------------------------------------------
    u64 GetCustomShaderFlag() const { return emitterRes->emitterData->shader.customShaderFlag; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットを取得します。
    //! @return エミッタセットへのポインタ
    //---------------------------------------------------------------------------
    EmitterSet* GetEmitterSet() { return emitterSet; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセットを取得します。
    //! @return エミッタセットへのポインタ
    //---------------------------------------------------------------------------
    const EmitterSet* GetEmitterSet() const { return emitterSet; }

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダインデックスを取得します。
    //! @return カスタムシェーダインデックス
    //---------------------------------------------------------------------------
    u32 GetCustomShaderIndex() const { return emitterRes->emitterData->shader.customShaderIndex; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタ計算タイプタイプを取得します。
    //! @return エミッタ計算タイプ
    //---------------------------------------------------------------------------
    EmitterCalcType GetCalcType() const { return emitterData->emitter.calcType; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタ追従タイプを取得します。
    //! @return エミッタ追従タイプ
    //---------------------------------------------------------------------------
    EmitterFollowType GetFollowType() const { return emitterData->emitter.followType; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタの名前を取得します。
    //! @return エミッタの名前
    //---------------------------------------------------------------------------
    const char* GetName() const { return emitterData->name; }

    //---------------------------------------------------------------------------
    //! @brief  処理するストリームアウトエミッタのフラグ値を取得します。
    //! @return 処理するストリームアウトエミッタのフラグの値
    //---------------------------------------------------------------------------
    u32 GetProcessEmitterFlag() const { return processEmitterFlag; }

    //---------------------------------------------------
    //! @brief  エミッタアニメ計算結果を取得する
    //! @return エミッタアニメ計算結果
    //---------------------------------------------------
    const EmitterAnimValue& GetEmitterAnimValue() const { return m_EmitterAnim; }

    //---------------------------------------------------------------------------
    //! @brief  Particle Attribute がダブルバッファかどうかを取得する
    //! @return Particle Attribute がダブルバッファかどうか
    //---------------------------------------------------------------------------
    bool IsParticleAttributeDoubleBuffer() const { return m_IsParticleAttributeDoubleBuffer; }

    //---------------------------------------------------------------------------
    //! @brief  CPU負荷の取得
    //! @return calcにかかったCPU時間(ミリ秒)
    //---------------------------------------------------------------------------
    f32 GetCpuCost() const { return m_CpuCost.ToTimeSpan().GetNanoSeconds() / 1000.0f / 1000.0f; }

    //---------------------------------------------------------------------------
    // vfxとの互換のために追加したインターフェース
    //---------------------------------------------------------------------------
    inline const ResEmitter* GetResEmitter() const
    {
        return emitterData;
    }

    inline const EmitterResource* GetEmitterResource() const
    {
        return emitterRes;
    }

    inline float GetFrameRate() const
    {
        return frameRate;
    }

    inline float GetTime() const
    {
        return time;
    }

    inline int GetGroupID() const
    {
        return groupID;
    }

    inline int GetGroupId() const
    {
        return GetGroupID();
    }

    inline const nw::math::MTX34& GetMatrixRt() const
    {
        return matrixRT;
    }

    inline const nw::math::MTX34& GetMatrixSrt() const
    {
        return matrixSRT;
    }

    inline void SetMatrixSrt( const nw::math::MTX34& srtMatrix )
    {
        matrixSRT = srtMatrix;
    }

    inline void SetMatrixRt( const nw::math::MTX34& rtMatrix )
    {
        matrixRT = rtMatrix;
    }

    Shader* GetShader( ShaderType type ) const
    {
        return shader[ type ];
    }

    inline void SetCustomShaderUserData( void* ptr )
    {
        customShaderUserData = ptr;
    }

    inline void* GetCustomShaderUserData() const
    {
        return customShaderUserData;
    }

    inline void SetCustomActionUserData( void* ptr )
    {
        customActionUserData = ptr;
    }

    inline void* GetCustomActionUserData() const
    {
        return customActionUserData;
    }

    DrawViewFlag GetDrawViewFlag() const
    {
        return drawViewFlag;
    }

    void SetDrawViewFlag( DrawViewFlag viewFlag )
    {
        drawViewFlag = viewFlag;
    }

    bool GetSkipCalculationFlag() const
    {
        return ( calcSkipFlag != 0 );
    }

    void SetSkipCalculationFlag( bool flag )
    {
        calcSkipFlag = ( flag ) ? 1 : 0;
    }

    float GetCustomShaderParameter( int index ) const
    {
        if( !emitterRes->customShaderParam )
        {
            return 0.0f;
        }
        float* fparam = reinterpret_cast< float* >( emitterRes->customShaderParam );
        return fparam[ index ];
    }

    //---------------------------------------------------------------------------
    //! @brief  αフェードインするかどうかを取得します。
    //! @return αフェードインするかどうか
    //---------------------------------------------------------------------------
    inline bool IsAlphaFadeIn() const
    {
        return ( emitterRes->m_pResEmitter->emitter.isAlphaFadeIn != 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief  αフェードアウトするかどうかを取得します。
    //! @return αフェードアウトするかどうか
    //---------------------------------------------------------------------------
    inline bool IsAlphaFadeOut() const
    {
        return ( emitterRes->m_pResEmitter->emitter.isFadeAlphaFade != 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief  スケールフェードインするかどうかを取得します。
    //! @return スケールフェードインするかどうか
    //---------------------------------------------------------------------------
    inline bool IsScaleFadeIn() const
    {
        return ( emitterRes->m_pResEmitter->emitter.isScaleFadeIn != 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief  スケールフェードアウトするかどうかを取得します。
    //! @return スケールフェードアウトするかどうか
    //---------------------------------------------------------------------------
    inline bool IsScaleFadeOut() const
    {
        return ( emitterRes->m_pResEmitter->emitter.isScaleFade != 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief  αフェードアウト値を取得します。
    //! @return αフェードアウト値
    //---------------------------------------------------------------------------
    inline f32 GetFadeOutAlpha() const
    {
        return fadeOutAlpha;
    }

    //---------------------------------------------------------------------------
    //! @brief  αフェードイン値を取得します。
    //! @return αフェードイン値
    //---------------------------------------------------------------------------
    inline f32 GetFadeInAlpha() const
    {
        return fadeInAlpha;
    }

    //---------------------------------------------------------------------------
    //! @brief  フェードイン・アウトを考慮した現在のスケールフェード値を取得します。
    //! @return フェードイン・アウトを考慮した現在のスケールフェード値
    //---------------------------------------------------------------------------
    inline f32 CalculateCurrentScaleFadeValue() const
    {
        f32 fadeValue = 1.0f;
        if( IsScaleFadeIn() )
        {
            fadeValue *= GetFadeInAlpha();
        }
        if( IsScaleFadeOut() )
        {
            fadeValue *= GetFadeOutAlpha();
        }
        return fadeValue;
    }

    //---------------------------------------------------------------------------
    //! @brief  フェードイン・アウトを考慮した現在のアルファフェード値を取得します。
    //! @return フェードイン・アウトを考慮した現在のアルファフェード値
    //---------------------------------------------------------------------------
    inline f32 CalculateCurrentAlphaFadeValue() const
    {
        f32 fadeValue = 1.0f;
        if( IsAlphaFadeIn() )
        {
            fadeValue *= GetFadeInAlpha();
        }
        if( IsAlphaFadeOut() )
        {
            fadeValue *= GetFadeOutAlpha();
        }
        return fadeValue;
    }

private:
    //---------------------------------------------------------------------------
    //! @brief                      エミッタ初期化を行います。
    //! @param[in] emitterResource  エミッタリソースへのポインタ
    //! @param[in] particleMax      パーティクル最大数
    //! @param[in] heap             ヒープへのポインタ
    //! @return                     成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool Initialize( const EmitterResource* emitterResource, u32 particleMax, Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief  リソース更新時処理を行います。
    //! @return 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool ResourceUpdate();

    //---------------------------------------------------------------------------
    //! @brief  確保しているバッファの解放処理を行います。
    //---------------------------------------------------------------------------
    void ReleaseBuffer();

    //---------------------------------------------------------------------------
    //! @brief  エミッタ終了処理を行います。
    //! @param[in] invokeCallback   コールバック関数を呼び出すか。コールバック関数を呼び出さずにFinalizeしたい場合に利用します。
    //---------------------------------------------------------------------------
    void Finalize( bool invokeCallback = true );

    //---------------------------------------------------------------------------
    //! @brief  放出毎に更新が必要な情報を更新します。
    //---------------------------------------------------------------------------
    void UpdateByEmit();

    //---------------------------------------------------------------------------
    //! @brief  マトリクス情報を作成します。
    //---------------------------------------------------------------------------
    void CreateResMatrix();

    //---------------------------------------------------------------------------
    //! @brief  リソース変更時に更新が必要な情報を更新します。
    //---------------------------------------------------------------------------
    void UpdateByChangeRes();

    //---------------------------------------------------------------------------
    //! @brief              子エミッタを追加します。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void AddChildEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief              子エミッタを削除します。
    //! @param[in] emitter  エミッタへのポインタ
    //---------------------------------------------------------------------------
    void RemoveChildEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @briefprivate                   パーティクルの位置と速度を指定した時間分強制的に進めます（誤差あり）
    //! @param[in, out] pOutPos         パーティクル位置へのポインタ
    //! @param[in, out] pOutVec         パーティクル速度へのポインタ
    //! @param[in]      deltaTime       経過時間
    //! @param[in]      pParticleAttr   パーティクルアトリビュート
    //! @param[in]      pParticleData   パーティクル情報
    //---------------------------------------------------------------------------
    void ForceUpdateParticlePosAttribute( nw::math::VEC4* pOutPos, nw::math::VEC4* pOutVec, float deltaTime, const ParticleAttribute* pParticleAttr, const ParticleData* pParticleData );

    //---------------------------------------------------------------------------
    //! @briefprivate  StreamOut バッファを強制的に更新します。
    //---------------------------------------------------------------------------
    void ForceUpdateForStreamOutBuffer();

    bool m_IsParticleAttributeDoubleBuffer; //!<　Particle Attributeがダブルバッファかどうか

    friend class  EmitterCalc;
    friend struct EmitterSet;
};

//---------------------------------------------------------------------------
//! @brief              ワールド座標系への変換
//! @param[out] pDst    出力となるVEC3へのポインタ
//! @param[in] src      入力となるVEC3へのポインタ
//! @param[in] attr     パーティクルアトリビュートへのポインタ
//---------------------------------------------------------------------------
inline void Emitter::TransformToWorldPos( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const
{
    if ( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        pDst->x = matrixSRT.f._00 * src.x + matrixSRT.f._01 * src.y + matrixSRT.f._02 * src.z + matrixSRT.f._03;
        pDst->y = matrixSRT.f._10 * src.x + matrixSRT.f._11 * src.y + matrixSRT.f._12 * src.z + matrixSRT.f._13;
        pDst->z = matrixSRT.f._20 * src.x + matrixSRT.f._21 * src.y + matrixSRT.f._22 * src.z + matrixSRT.f._23;
        return;
    }
    else if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        pDst->x = attr->emitterMat0.x * src.x + attr->emitterMat0.y * src.y + attr->emitterMat0.z * src.z + matrixSRT.f._03;
        pDst->y = attr->emitterMat1.x * src.x + attr->emitterMat1.y * src.y + attr->emitterMat1.z * src.z + matrixSRT.f._13;
        pDst->z = attr->emitterMat2.x * src.x + attr->emitterMat2.y * src.y + attr->emitterMat2.z * src.z + matrixSRT.f._23;
        return;
    }
    else
    {
        // 追従なし
        pDst->x = attr->emitterMat0.x * src.x + attr->emitterMat0.y * src.y + attr->emitterMat0.z * src.z + attr->emitterMat0.w;
        pDst->y = attr->emitterMat1.x * src.x + attr->emitterMat1.y * src.y + attr->emitterMat1.z * src.z + attr->emitterMat1.w;
        pDst->z = attr->emitterMat2.x * src.x + attr->emitterMat2.y * src.y + attr->emitterMat2.z * src.z + attr->emitterMat2.w;
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief              ローカル座標系への変換
//! @param[out] pDst    出力となるVEC3へのポインタ
//! @param[in] src      入力となるVEC3へのポインタ
//! @param[in] attr     パーティクルアトリビュートへのポインタ
//---------------------------------------------------------------------------
inline void Emitter::TransformToLocalPos( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const
{
    // ローカル位置に戻して保存
    if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, matrixSRT );

        pDst->x = mtxInv.f._00 * src.x + mtxInv.f._01 * src.y + mtxInv.f._02 * src.z + mtxInv.f._03;
        pDst->y = mtxInv.f._10 * src.x + mtxInv.f._11 * src.y + mtxInv.f._12 * src.z + mtxInv.f._13;
        pDst->z = mtxInv.f._20 * src.x + mtxInv.f._21 * src.y + mtxInv.f._22 * src.z + mtxInv.f._23;
        return;
    }
    else if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        const nw::math::MTX34 mtx(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, matrixSRT.f._03,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, matrixSRT.f._13,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, matrixSRT.f._23
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        pDst->x = mtxInv.f._00 * src.x + mtxInv.f._01 * src.y + mtxInv.f._02 * src.z + mtxInv.f._03;
        pDst->y = mtxInv.f._10 * src.x + mtxInv.f._11 * src.y + mtxInv.f._12 * src.z + mtxInv.f._13;
        pDst->z = mtxInv.f._20 * src.x + mtxInv.f._21 * src.y + mtxInv.f._22 * src.z + mtxInv.f._23;
        return;
    }
    else
    {
        // 追従なし
        const nw::math::MTX34 mtx(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, attr->emitterMat0.w,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, attr->emitterMat1.w,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, attr->emitterMat2.w
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        pDst->x = mtxInv.f._00 * src.x + mtxInv.f._01 * src.y + mtxInv.f._02 * src.z + mtxInv.f._03;
        pDst->y = mtxInv.f._10 * src.x + mtxInv.f._11 * src.y + mtxInv.f._12 * src.z + mtxInv.f._13;
        pDst->z = mtxInv.f._20 * src.x + mtxInv.f._21 * src.y + mtxInv.f._22 * src.z + mtxInv.f._23;
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief              ローカル座標系への変換
//! @param[out] pDst    出力となるVEC3へのポインタ
//! @param[in] src      入力となるVEC3へのポインタ
//! @param[in] attr     パーティクルアトリビュートへのポインタ
//---------------------------------------------------------------------------
inline void Emitter::TransformToLocalVec( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const
{
    if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        nw::math::MTX34 mtxSRT( matrixSRT );
        mtxSRT.m[ 0 ][ 3 ] = 0;
        mtxSRT.m[ 1 ][ 3 ] = 0;
        mtxSRT.m[ 2 ][ 3 ] = 0;

        nw::math::MTX34 mtxInvSRT;
        nw::math::MTX34Inverse( &mtxInvSRT, mtxSRT );
        nw::math::VEC3Transform( pDst, mtxInvSRT, src );
        return;
    }
    else if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        const nw::math::MTX34 mtx(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, 0,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, 0,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, 0
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );
        nw::math::VEC3Transform( pDst, mtxInv, src );
        return;
    }
    else
    {
        // 追従なし
        const nw::math::MTX34 mtx(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, 0,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, 0,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, 0
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );
        nw::math::VEC3Transform( pDst, mtxInv, src );
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief              ローカル座標系への変換
//! @param[out] pDst    出力となるVEC3へのポインタ
//! @param[in] src      入力となるVEC3へのポインタ
//! @param[in] attr     パーティクルアトリビュートへのポインタ
//---------------------------------------------------------------------------
inline void Emitter::TransformToWorldVec( nw::math::VEC3* pDst, const nw::math::VEC3& src, const ParticleAttribute* attr ) const
{
    const nw::math::VEC3& localVec = src;
    if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        pDst->x = matrixSRT.f._00 * localVec.x + matrixSRT.f._01 * localVec.y + matrixSRT.f._02 * localVec.z;
        pDst->y = matrixSRT.f._10 * localVec.x + matrixSRT.f._11 * localVec.y + matrixSRT.f._12 * localVec.z;
        pDst->z = matrixSRT.f._20 * localVec.x + matrixSRT.f._21 * localVec.y + matrixSRT.f._22 * localVec.z;
        return;
    }
    else if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        pDst->x = attr->emitterMat0.x * localVec.x + attr->emitterMat0.y * localVec.y + attr->emitterMat0.z * localVec.z;
        pDst->y = attr->emitterMat1.x * localVec.x + attr->emitterMat1.y * localVec.y + attr->emitterMat1.z * localVec.z;
        pDst->z = attr->emitterMat2.x * localVec.x + attr->emitterMat2.y * localVec.y + attr->emitterMat2.z * localVec.z;
        return;
    }
    else
    {
        // 追従なし
        pDst->x = attr->emitterMat0.x * localVec.x + attr->emitterMat0.y * localVec.y + attr->emitterMat0.z * localVec.z;
        pDst->y = attr->emitterMat1.x * localVec.x + attr->emitterMat1.y * localVec.y + attr->emitterMat1.z * localVec.z;
        pDst->z = attr->emitterMat2.x * localVec.x + attr->emitterMat2.y * localVec.y + attr->emitterMat2.z * localVec.z;
        return;
    }
}

//------------------------------------------------------------------------------
//  パーティクル位置をワールド座標系に変換するための、追従設定を加味した変換行列を取得します。
//------------------------------------------------------------------------------
inline void Emitter::GetEmitterTransformMatrix( nw::math::MTX34* pOutMatrix, u32 index ) const
{
    NW_NULL_ASSERT( pOutMatrix );
    NW_ASSERT( index < ptclAttrFillMax );

    const ParticleAttribute* attr = &particleAttr[ index ];

    // 追従設定を見て、適切に行列を作成して返す。
    if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        *pOutMatrix = matrixSRT;
    }
    else if( emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        *pOutMatrix = nw::math::MTX34(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, matrixSRT.f._03,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, matrixSRT.f._13,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, matrixSRT.f._23 );
    }
    else
    {
        // 追従なし
        *pOutMatrix = nw::math::MTX34(
            attr->emitterMat0.x, attr->emitterMat0.y, attr->emitterMat0.z, attr->emitterMat0.w,
            attr->emitterMat1.x, attr->emitterMat1.y, attr->emitterMat1.z, attr->emitterMat1.w,
            attr->emitterMat2.x, attr->emitterMat2.y, attr->emitterMat2.z, attr->emitterMat2.w );
    }
}

} // namespace eft2
} // namespace nw
