﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <nw/eft/eft2_Macro.h>
#include <nw/eft/eft2_EndianUtil.h>


namespace nw   {
namespace eft2 {

// バイナリデータ
#define BIN_DATA_NOTHING    (0xFFFFFFFF)

//------------------------------------------------------------------------------
//! @brief      バイナリデータフリップフラグ
//------------------------------------------------------------------------------
enum
{
    EFT_BINARY_DATA_UNFLIPPED = 0,      //!< 未フリップ
    EFT_BINARY_DATA_FLIPPED   = 1,      //!< フリップ済
};

//------------------------------------------------------------------------------
//! @brief      バイナリデータ
//------------------------------------------------------------------------------
struct BinaryData
{
    u8              tag[4];             //!< バイナリタグ
    u32             binSize;            //!< バイナリサイズ
    u32             child;              //!< 子方向のバイナリオフセット
    u32             next;               //!< 右方向のバイナリオフセット
    u32             sub;                //!< サブバイナリデータ
    u32             offset;             //!< バイナリイメージへのオフセット
    u32             guid;               //!< バイナリツリー内におけるユニークID
    u16             childNum;           //!< 子要素の個数
    u8              flipped;            //!< フリップが必要な場合に、このノードが既にフリップされたかどうか
    u8              endian;             //!< このノードのエンディアン

    //------------------------------------------------------------------------------
    //! @brief          パラメータ初期化
    //------------------------------------------------------------------------------
    void Init()
    {
        binSize  = 0;
        child    = BIN_DATA_NOTHING;
        next     = BIN_DATA_NOTHING;
        sub      = BIN_DATA_NOTHING;
        offset   = sizeof(BinaryData);
        guid     = 0;
        childNum = 0;
        flipped  = endian = 0;
    }

    //------------------------------------------------------------------------------
    //! @brief          バイナリタグ設定
    //! @param[in] a    タグ文字1
    //! @param[in] b    タグ文字2
    //! @param[in] c    タグ文字3
    //! @param[in] d    タグ文字4
    //------------------------------------------------------------------------------
    void SetBinaryTag( u8 a, u8 b, u8 c, u8 d )
    {
        tag[0]  = a;
        tag[1]  = b;
        tag[2]  = c;
        tag[3]  = d;
    }

    //------------------------------------------------------------------------------
    //! @brief          バイナリタグ取得
    //! @return         バイナリタグを返します。
    //------------------------------------------------------------------------------
    u32 GetBinaryTag()
    {
        return EFT_MAKE_TAG( tag[0], tag[1], tag[2], tag[3] );
    }

    //------------------------------------------------------------------------------
    //! @brief          バイナリ取得
    //! @return         バイナリへのポインタを返します。
    //------------------------------------------------------------------------------
    void* GetBinaryData()
    {
        u32 offset_size = offset;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &offset_size );
        }
#endif//EFT_BI_ENDIAN

        return reinterpret_cast<void*>( static_cast<u8*>( tag ) + offset_size );
    }

    //------------------------------------------------------------------------------
    //! @brief          フリップを伴うバイナリ取得
    //! @tparam  T      バイナリの型名
    //! @return         バイナリへのポインタを返します。
    //------------------------------------------------------------------------------
    template<typename T>
    T* GetBinaryDataWithFlip()
    {
        u32 offset_size = offset;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &offset_size );
        }
#endif
        T* binary = reinterpret_cast<T*>( static_cast<u8*>( tag ) + offset_size );
#ifdef  EFT_BI_ENDIAN
        if ( IsNeedFlip() )
        {
            binary->FlipEndian();
            flipped = EFT_BINARY_DATA_FLIPPED;
        }
#endif
        return binary;
    }

    //------------------------------------------------------------------------------
    //! @brief          バイナリサイズ取得
    //! @return         バイナリサイズを返します。
    //------------------------------------------------------------------------------
    u32 GetBinarySize()
    {
        u32 binary_size = binSize;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &binary_size );
        }
#endif//EFT_BI_ENDIAN

        return binary_size;
    }

    //------------------------------------------------------------------------------
    //! @brief          次要素へのバイナリオフセット取得
    //! @return         次要素へのバイナリオフセットを返します。
    //------------------------------------------------------------------------------
    u32 GetNextBinaryOffset()
    {
        u32 binary_size = next;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &binary_size );
        }
#endif//EFT_BI_ENDIAN

        return binary_size;
    }

    //------------------------------------------------------------------------------
    //! @brief          サブバイナリ取得
    //! @return         サブバイナリへのポインタを返します。
    //------------------------------------------------------------------------------
    BinaryData* GetSubData()
    {
        u32 binary_sub = sub;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &binary_sub );
        }
#endif//EFT_BI_ENDIAN

        if ( binary_sub == BIN_DATA_NOTHING ) return NULL;
        return reinterpret_cast<BinaryData*>( (u8*)&tag + binary_sub );
    }

    //------------------------------------------------------------------------------
    //! @brief          子バイナリ取得
    //! @return         子バイナリへのポインタを返します。
    //------------------------------------------------------------------------------
    BinaryData* GetChildData()
    {
        u32 child_offset = child;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &child_offset );
        }
#endif//EFT_BI_ENDIAN

        if ( child_offset == BIN_DATA_NOTHING ) return NULL;
        return reinterpret_cast<BinaryData*>( (u8*)&tag + child_offset );
    }

    //------------------------------------------------------------------------------
    //! @brief          Nextバイナリ取得
    //! @return         次データへのポインタを返します。
    //------------------------------------------------------------------------------
    BinaryData* GetNextData()
    {
        u32 next_offset = next;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &next_offset );
        }
#endif//EFT_BI_ENDIAN

        if ( next_offset == BIN_DATA_NOTHING ) return NULL;
        return reinterpret_cast<BinaryData*>( (u8*)&tag + next_offset );
    }

    //------------------------------------------------------------------------------
    //! @brief               タグを指定して子バイナリ取得
    //! @param[in] tagValue  タグ
    //! @return              指定したタグが存在すればそのポインタを返し、無ければNULLを返します。
    //------------------------------------------------------------------------------
    BinaryData* GetDirectChildData( u32 tagValue )
    {
        BinaryData* pChild = GetChildData();
        if ( !pChild) return NULL;

        while(pChild)
        {
            if (pChild->GetBinaryTag() == tagValue) return pChild;
            pChild = pChild->GetNextData();
        }
        return NULL;
    }

    //------------------------------------------------------------------------------
    //! @brief          子バイナリ数取得
    //! @return         子バイナリの数を返します。
    //------------------------------------------------------------------------------
    u32 GetDirectChildNum()
    {
        u32 num = 0;
        BinaryData* pChild = GetChildData();
        while(pChild)
        {
            num++;
            pChild = pChild->GetNextData();
        }
        return num;
    }

    //------------------------------------------------------------------------------
    //! @brief               タグを指定して子バイナリ数取得
    //! @param[in] tagValue  子バイナリのタグ
    //! @return              指定したタグが存在すればその子バイナリの数を返します。無い場合は0を返します。
    //------------------------------------------------------------------------------
    u32 GetDirectChildNum( u32 tagValue )
    {
        u32 num = 0;
        BinaryData* pChild = GetChildData();
        while(pChild)
        {
            if (pChild->GetBinaryTag() == tagValue) num++;
            pChild = pChild->GetNextData();
        }
        return num;
    }

    //------------------------------------------------------------------------------
    //! @brief               タグを指定してNextバイナリ取得
    //! @param[in] tagValue  タグ
    //! @return              指定したタグが存在すればそのポインタを返し、無い場合はNULLを返します。
    //------------------------------------------------------------------------------
    BinaryData* GetNextData( u32 tagValue )
    {
        BinaryData* pNext = GetNextData();
        if ( !pNext) return NULL;

        while(pNext)
        {
            if (pNext->GetBinaryTag() == tagValue) return pNext;
            pNext = pNext->GetNextData();
        }
        return NULL;
    }

    //------------------------------------------------------------------------------
    //! @brief          GUID取得
    //! @return         GUIDを返します。
    //------------------------------------------------------------------------------
    u32 GetGuid()
    {
        u32 temp = guid;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &temp );
        }
#endif//EFT_BI_ENDIAN
        return temp;
    }

    //------------------------------------------------------------------------------
    //! @brief          ヘッダに記載された子バイナリ数取得
    //! @return         ヘッダに記載された子バイナリ数を返します。
    //------------------------------------------------------------------------------
    u16 GetChildNum()
    {
        u16 temp = childNum;
#ifdef  EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            EndianUtil::Flip( &temp );
        }
#endif//EFT_BI_ENDIAN
        return temp;
    }

    //------------------------------------------------------------------------------
    //! @brief          フリップが必要な状態かを取得
    //! @return         フリップが必要な状態ならtrue,不要ならfalseを返します。
    //------------------------------------------------------------------------------
    bool IsNeedFlip()
    {
#ifndef  EFT_BI_ENDIAN
        return false;
#else
        return endian != EFT_ENDIAN_TYPE && static_cast<u32>(flipped) == EFT_BINARY_DATA_UNFLIPPED;
#endif
    }
};

} // namespace eft2
} // namespace nw
