﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut/ut_Color.h>
#include <nw/dev/dev_Pad.h>
#include <nw/dw/system/dw_Inputs.h>
#include <nw/dw/system/dw_Types.h>
#include <nw/dw/system/dw_Thickness.h>
#include <nw/dw/system/dw_UIRenderer.h>
#include <nw/dw/control/dw_UIElementTreeContext.h>
#include <nw/dw/control/dw_UIElementList.h>
#include <nw/dw/control/dw_UIElementRenderArgs.h>
#include <nw/dw/control/dw_Style.h>

namespace nw {
namespace internal {
namespace dw {

class UIElementListEventArgs;

//---------------------------------------------------------------------------
//! @brief UI要素の基底クラスです。
//---------------------------------------------------------------------------
class UIElement
{
public:
    static const f32 InvalidWidth;
    static const f32 InvalidHeight;

private:
    static const s32 DEFAULT_WHEEL_SCROLL_DELTA = -8;

public:
    UIElement();
    virtual ~UIElement() { }

public:
    UIElement* GetParent() const;

    const nw::math::Vector2 GetTopLeft() const;
    UIElement& SetTopLeft(f32 x, f32 y);
    UIElement& SetTopLeft(const nw::math::Vector2& value);
    UIElement& SetLeft(f32 value);
    UIElement& SetTop(f32 value);

    const nw::math::Vector2 GetSize() const;
    const f32 GetWidth() const;
    const f32 GetHeight() const;
    UIElement& SetSize(f32 width, f32 height);
    UIElement& SetSize(const nw::math::Vector2& value);
    UIElement& SetWidth(f32 value);
    UIElement& SetHeight(f32 value);

    const nw::math::Vector2& GetMeasuredSize() const;
    const f32 GetMeasuredWidth() const;
    const f32 GetMeasuredHeight() const;

    const nw::math::Vector2& GetFixedSize() const;
    const f32 GetFixedWidth() const;
    const f32 GetFixedHeight() const;

    const nw::math::Vector2 GetMinimumSize() const;
    const f32 GetMinimumWidth() const;
    const f32 GetMinimumHeight() const;
    UIElement& SetMinimumSize(f32 width, f32 height);
    UIElement& SetMinimumSize(const nw::math::Vector2& value);
    UIElement& SetMinimumWidth(f32 value);
    UIElement& SetMinimumHeight(f32 value);

    const nw::math::Vector2 GetMaximumSize() const;
    const f32 GetMaximumWidth() const;
    const f32 GetMaximumHeight() const;
    UIElement& SetMaximumSize(f32 width, f32 height);
    UIElement& SetMaximumSize(const nw::math::Vector2& value);
    UIElement& SetMaximumWidth(f32 value);
    UIElement& SetMaximumHeight(f32 value);

    const Thickness GetMargin() const;
    UIElement& SetMargin(const Thickness& value);

    const Thickness GetPadding() const;
    UIElement& SetPadding(const Thickness& value);

    const nw::math::Vector2 GetContentAreaTopLeft() const;
    const nw::math::Vector2 GetContentAreaSize() const;

    Visibility GetVisibility() const;
    UIElement& SetVisibility(Visibility value);

    //---------------------------------------------------------------------------
    //! @brief   フォーカスを設定できる要素かどうかを示します。
    //!
    //! @details フォーカスを設定できる要素かどうかを示します。
    //!
    //! @return  フォーカスを設定できる要素なら true、それ以外は false を返します。
    //---------------------------------------------------------------------------
    bool GetIsFocusable() const;

    //---------------------------------------------------------------------------
    //! @brief   フォーカスを設定できる要素かどうかを設定します。
    //!
    //! @details フォーカスを設定できる要素かどうかを設定します。
    //!
    //! @param[in]  value  フォーカスを設定できるようにする場合は true、設定できないようにする場合は false を指定します。
    //!
    //! @return   自身の参照を返します。
    //---------------------------------------------------------------------------
    UIElement& SetIsFocusable(bool value);

    bool GetIsFocused() const;
    bool GetIsContainsFocus() const;
    UIElement* GetLastFocusEnterContent() const;

    bool GetIsPointerOver() const;
    bool GetIsContainsPointerOver() const;
    UIElement* GetPointerOverContent() const;

    Measurement GetMeasurement() const;
    UIElement& SetMeasurement(Measurement value);

    HorizontalAlignment GetHorizontalAlignment() const;
    UIElement& SetHorizontalAlignment(HorizontalAlignment value);

    VerticalAlignment GetVerticalAlignment() const;
    UIElement& SetVerticalAlignment(VerticalAlignment value);

    Dock GetDock() const;
    UIElement& SetDock(Dock value);

    NW_DEPRECATED_FUNCTION(const nw::ut::Color4f GetBackgroundColor() const);
    UIElement& SetBackgroundColor(const nw::ut::Color4f& value);
    UIElement& SetBackgroundColor(const nw::ut::Color4f& value1, const nw::ut::Color4f& value2);
    const nw::ut::Color4f GetBackgroundColor0() const;
    UIElement& SetBackgroundColor0(const nw::ut::Color4f& value);
    const nw::ut::Color4f GetBackgroundColor1() const;
    UIElement& SetBackgroundColor1(const nw::ut::Color4f& value);

    const nw::ut::Color4f GetBorderColor() const;
    UIElement& SetBorderColor(const nw::ut::Color4f& value);

    bool GetIsBordered() const;
    UIElement& SetIsBordered(bool value);

    UIElementList& GetContents();
    UIElementList& GetContents() const;
    UIElement& SetContents(UIElementList& value);

    bool SetFocus();

    void EnsureVisible();

    //---------------------------------------------------------------------------
    //! @brief       UI要素のコンテントサイズを計算します。
    //!
    //! @details     UI要素のコンテントサイズを計算します。
    //!              このサイズにはコンテントのマージンも含まれます。
    //!              ２つ以上のコンテントを持つ要素は、この関数をオーバーライドして適切なコンテントサイズを返す必要があります。
    //!
    //! @param[in]   renderer  UIRenderer を指定します。
    //!
    //! @return      UI要素のコンテントサイズを返します。
    //---------------------------------------------------------------------------
    const nw::math::Vector2& Measure(UIRenderer& renderer);

    //---------------------------------------------------------------------------
    //! @brief       入力を更新します。
    //!
    //! @details     入力を更新します。
    //!
    //! @param[in]   inputs   nw::internal::dw::Inputs を指定します。
    //---------------------------------------------------------------------------
    void UpdateInputs(const Inputs& inputs);

    //---------------------------------------------------------------------------
    //! @brief       UI要素を更新します。
    //!
    //! @details     UI要素を更新します。
    //!
    //! @param[in]   context        UIElementTreeContext を指定します。
    //! @param[in]   renderer       UIRenderer を指定します。
    //! @param[in]   isMeasurement  サイズ計算の有無を指定します。
    //---------------------------------------------------------------------------
    void Update(UIElementTreeContext& context, UIRenderer& renderer, bool isMeasurement=true);

    //---------------------------------------------------------------------------
    //! @brief       UI要素を描画します。
    //!
    //! @details     UI要素を描画します。
    //!
    //! @param[in]   context   UIRenderContext を指定します。
    //! @param[in]   renderer  UIRenderer を指定します。
    //---------------------------------------------------------------------------
    void Render(UIElementTreeContext& context, UIRenderer& renderer);

protected:
    // 派生クラスから MeasuredSize を変更するためのメソッドです。
    static void SetMeasuredSize(UIElement& element, f32 width, f32 height);
    static void SetMeasuredSize(UIElement& element, const nw::math::Vector2 value);

    const nw::math::Vector2& GetScrollOffset() const;
    void SetScrollOffset(nw::math::Vector2 value);

    virtual s32 GetWheelScrollDelta(s32 lines) const
    {
        return DEFAULT_WHEEL_SCROLL_DELTA * lines;
    }

    //---------------------------------------------------------------------------
    //! @brief       指定子要素が表示されるようにスクロールします。
    //!
    //! @param[in]   element  表示する要素への参照を指定します。
    //---------------------------------------------------------------------------
    virtual void OnEnsureVisible(UIElement& element) { (void)element; }

    //---------------------------------------------------------------------------
    //! @brief       自身または子要素がフォーカスを取得する際に実行されます。
    //!
    //! @details     自身がフォーカスを取得する際に実行されます。
    //!              イベントは LostFocus, FocusLeave, FocusEnter, GotFocus の順に処理されます。
    //!
    //! @param[in]   pElement  新しくフォーカスを持つ要素へのポインタを指定します。
    //---------------------------------------------------------------------------
    virtual void OnFocusEnter(UIElement* pElement) { (void)pElement; }

    //---------------------------------------------------------------------------
    //! @brief       自身または子要素がフォーカスを失う際に実行されます。
    //!
    //! @details     自身がフォーカスを失う際に実行されます。
    //!              イベントは LostFocus, FocusLeave, FocusEnter, GotFocus の順に処理されます。
    //---------------------------------------------------------------------------
    virtual void OnFocusLeave() { }

    //---------------------------------------------------------------------------
    //! @brief       自身がフォーカスを取得した際に実行されます。
    //!
    //! @details     自身がフォーカスを取得した際に実行されます。
    //!              イベントは LostFocus, FocusLeave, FocusEnter, GotFocus の順に処理されます。
    //---------------------------------------------------------------------------
    virtual void OnGotFocus() { }

    //---------------------------------------------------------------------------
    //! @brief       自身がフォーカスを失った際に実行されます。
    //!
    //! @details     自身がフォーカスを失った際に実行されます。
    //!              イベントは LostFocus, FocusLeave, FocusEnter, GotFocus の順に処理されます。
    //---------------------------------------------------------------------------
    virtual void OnLostFocus() { }

    virtual void OnPointerEnter(const nw::internal::dw::Inputs& inputs) { (void)inputs; }
    virtual void OnPointerLeave() { }

    //---------------------------------------------------------------------------
    //! @brief       ポインターオーバーを通知します。
    //!
    //! @param[in]   inputs  nw::internal::dw::Inputs を指定します。
    //---------------------------------------------------------------------------
    virtual void OnPointerOver(const nw::internal::dw::Inputs& inputs) { (void)inputs; }

    //---------------------------------------------------------------------------
    //! @brief       ポインター入力を更新します。
    //!
    //! @param[in]   inputs  nw::internal::dw::Inputs を指定します。
    //!
    //! @return      入力を処理した場合は true、処理しなかった場合は false を返します。
    //!              false を返すと、親UI要素に処理を委譲します。
    //---------------------------------------------------------------------------
    virtual bool OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs) { (void)inputs; return false; }

    virtual bool OnPreviewUpdateFocusedInput(const nw::internal::dw::Inputs& inputs) { (void)inputs; return false; }

    //---------------------------------------------------------------------------
    //! @brief       フォーカス入力を更新します。
    //!
    //! @param[in]   inputs  nw::internal::dw::Inputs を指定します。
    //!
    //! @return      入力を処理した場合は true、処理しなかった場合は false を返します。
    //!              false を返すと、親UI要素に処理を委譲します。
    //---------------------------------------------------------------------------
    virtual bool OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs) { (void)inputs; return false; }

    //---------------------------------------------------------------------------
    //! @brief       UI要素のコンテントサイズを計算します。
    //!
    //! @details     UI要素のコンテントサイズを計算します。
    //!              このサイズにはコンテントのマージンも含まれます。
    //!              ２つ以上のコンテントを持つ要素は、この関数をオーバーライドして適切なコンテントサイズを返す必要があります。
    //!
    //! @param[in]   renderer  UIRenderer を指定します。
    //!
    //! @return      UI要素のコンテントサイズを返します。
    //---------------------------------------------------------------------------
    virtual const nw::math::Vector2 OnMeasure(UIRenderer& renderer) const;

    //---------------------------------------------------------------------------
    //! @brief       UI要素を更新します。
    //!
    //! @details     UI要素を更新します。
    //!              特別な更新処理を行う場合は、この関数をオーバーライドしてください。
    //!
    //! @param[in]   context  UIElementTreeContext を指定します。
    //---------------------------------------------------------------------------
    virtual void OnUpdate(const UIElementTreeContext& context) { (void)context; }

    //---------------------------------------------------------------------------
    //! @brief       UI要素サイズ計算の準備を行います。
    //!
    //! @details     UI要素サイズ計算の準備を行います。
    //!              OnMeasure, OnUpdate 前にUI要素を更新するには、この関数をオーバーライドしてください。
    //---------------------------------------------------------------------------
    virtual void OnPrepareMeasure() { }

    //---------------------------------------------------------------------------
    //! @brief       UI要素描画の準備を行います。
    //!
    //! @details     UI要素描画の準備を行います。
    //!              特別な更新処理を行う場合は、この関数をオーバーライドしてください。
    //!
    //! @param[in]   context  UIElementTreeContext を指定します。
    //!
    //! @return      描画に使用するパラメータへの参照を返します。
    //---------------------------------------------------------------------------
    virtual UIElementRenderArgs& OnPrepareRender(const UIElementTreeContext& context);

    virtual void OnRender(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
    {
        (void)context; (void)renderer; (void)args;
    }
    virtual void PreRenderContents(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
    {
        (void)context; (void)renderer; (void)args;
    }
    virtual void PostRenderContents(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
    {
        (void)context; (void)renderer; (void)args;
    }

    virtual bool CanAlignHorizontal(const UIElement& element) const { (void)element; return true; }
    virtual bool CanAlignVertical(const UIElement& element) const { (void)element; return true; }

    //---------------------------------------------------------------------------
    //! @brief       コンテンツリストの内容が変更されると発生します。
    //!
    //! @details     コンテンツリストの内容が変更されると発生します。
    //!              コンテンツリスト変更イベントを処理したい場合にオーバーライドしてください。
    //!
    //! @param[in]   args UI要素リストの変更パラメータを指定します。
    //---------------------------------------------------------------------------
    virtual void OnContentsChanged(UIElementListEventArgs& args) { (void)args; }

private:
    UIElement& SetParent(UIElement* value);

    //---------------------------------------------------------------------------
    //! @brief       コンテンツリストの変更に同期します。
    //!
    //! @details     コンテンツリストの変更から親要素やフォーカス要素の設定を同期します。
    //!
    //! @param[in]   args UI要素リストの変更パラメータを指定します。
    //---------------------------------------------------------------------------
    void SynchronizeContentsChanged(UIElementListEventArgs& args);

    //---------------------------------------------------------------------------
    //! @brief       スクロールオフセットを考慮したUI要素の表示領域を行います。
    //!
    //! @return      UI要素の表示領域を示す矩形を返します。
    //---------------------------------------------------------------------------
    nw::ut::Rect GetVisibleRectangle() const;

    //---------------------------------------------------------------------------
    //! @brief       指定子要素が表示されるようにスクロールします。
    //!
    //! @param[in]   element  表示する要素への参照を指定します。
    //---------------------------------------------------------------------------
    void EnsureVisibleInternal(UIElement& element);

    void SetFocusInternal();
    void KillFocusInternal();

    //---------------------------------------------------------------------------
    //! @brief       フォーカスを取得したことを通知します。
    //!
    //! @details     UI要素ツリーの親に対してフォーカスを取得したことを通知します。
    //!
    //! @param[in]   pElement  新しくフォーカスを持つ要素へのポインタを指定します。
    //---------------------------------------------------------------------------
    void FocusEnter(UIElement* pElement);

    //---------------------------------------------------------------------------
    //! @brief       フォーカスを失ったことを通知します。
    //!
    //! @details     UI要素ツリーの親に対してフォーカスを失ったことを通知します。
    //---------------------------------------------------------------------------
    void FocusLeave();

    //---------------------------------------------------------------------------
    //! @brief       ポインターを更新します。
    //!
    //! @details     ポインターを更新します。
    //!
    //! @param[in]   inputs   nw::internal::dw::Inputs を指定します。
    //---------------------------------------------------------------------------
    void UpdatePointer(const Inputs& inputs);

    void UpdatePointerImpl(const nw::internal::dw::Inputs& inputs);
    void UpdatePointerInput(const nw::internal::dw::Inputs& inputs);

    //---------------------------------------------------------------------------
    //! @brief       ポインターオーバー状態の入力を更新します。
    //!
    //! @param[in]   inputs  nw::internal::dw::Inputs を指定します。
    //---------------------------------------------------------------------------
    void UpdatePointerInputImpl(const nw::internal::dw::Inputs& inputs);

    void UpdateFocusedInput(const nw::internal::dw::Inputs& inputs);

    bool PreviewUpdateFocusedInputImpl(const nw::internal::dw::Inputs& inputs);

    //---------------------------------------------------------------------------
    //! @brief       フォーカスを持った状態のパッド入力を更新します。
    //!
    //! @param[in]   inputs  nw::internal::dw::Inputs を指定します。
    //---------------------------------------------------------------------------
    void UpdateFocusedInputImpl(const nw::internal::dw::Inputs& inputs);

    UIElement* HitTestContents(const nw::math::Vector2& position) const;

    void PointerEnter(const nw::internal::dw::Inputs& inputs);
    void PointerOver(const nw::internal::dw::Inputs& inputs);
    void PointerLeave();

    //---------------------------------------------------------------------------
    //! @brief       UI要素のサイズ計算を行います。
    //!
    //! @details     UI要素の自動サイズ計算を行います。
    //!              自動サイズ調整は子要素の影響を受けるので、子→親の順で処理します。
    //!
    //! @param[in]   renderer  UIRenderer を指定します。
    //---------------------------------------------------------------------------
    void MeasureInternal(UIRenderer& renderer);

    //---------------------------------------------------------------------------
    //! @brief       UI要素サイズ計算して更新します。
    //!
    //! @details     UI要素の自動サイズ計算を行います。
    //!              自動サイズ調整は子要素の影響を受けるので、子→親の順で処理します。
    //!
    //! @param[in]   renderer  UIRenderer を指定します。
    //---------------------------------------------------------------------------
    void UpdateMeasuredSize(UIRenderer& renderer);

    //---------------------------------------------------------------------------
    //! @brief       UI要素を更新します。
    //!
    //! @details     UI要素を更新します。
    //!              ストレッチ処理は親要素の影響を受けるので親→子の順で処理します。
    //!
    //! @param[in]   context  UIElementTreeContext を指定します。
    //---------------------------------------------------------------------------
    void UpdateInternal(UIElementTreeContext& context);

    void UpdateAlignment(const UIElement* pParentElement);

    void MeasureContents(UIRenderer& renderer);
    void UpdateContents(UIElementTreeContext& context);
    void RenderContents(UIElementTreeContext& context, UIRenderer& renderer) const;

    bool ShouldRenderContent(UIElement& element) const;

    void SetCaluclatedSize(f32 width, f32 height);
    void ValidateScrollOffset();

    void OnItemsChanged(UIElementListEventArgs& args)
    {
        SynchronizeContentsChanged(args);
    }

private:
    UIElement* m_pParent;

    nw::math::Vector2  m_TopLeft;
    nw::math::Matrix34 m_TransformMatrix;
    nw::math::Vector2  m_Size;
    nw::math::Vector2  m_MeasuredSize;  // OnMeasure() にて計測したサイズ
    nw::math::Vector2  m_FixedSize;
    nw::math::Vector2  m_MinimumSize;
    nw::math::Vector2  m_MaximumSize;
    nw::math::Vector2  m_ScrollOffset;

    Thickness m_Margin;
    Thickness m_Padding;

    Visibility m_Visibility;
    bool       m_IsFocasable;

    Measurement         m_Measurement;
    HorizontalAlignment m_HorizontalAlignment;
    VerticalAlignment   m_VerticalAlignment;

    Dock m_Dock;

    nw::ut::Color4f m_BackgroundColor0;
    nw::ut::Color4f m_BackgroundColor1;
    nw::ut::Color4f m_BorderColor;
    bool m_IsBordered;

    bool       m_IsFocused;
    bool       m_IsContainsFocus;
    UIElement* m_pLastFocusEnterContent;

    bool m_IsContainsPointerOver;
    UIElement* m_pPointerOverContent;

    UIElementRenderArgs m_RenderArgs;

    // TODO : StyleList クラスに変更する
    nw::ut::LinkList<Style::Reference, offsetof(Style::Reference, node)> m_Styles;

    UIElementList* m_pContents;
    UIElementList::ItemsChangedEventHandler m_ContentsChangedObserver;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)

public:
    virtual const char* ToString() const
    {
        return "UIElement";
    }

#endif
};

} // dw
} // internal
} // nw
