﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dw/control/dw_UIElementList.h>

namespace nw {
namespace internal {
namespace dw {


template<s32 TItemCount=0>
class FixedUIElementList : public UIElementList
{
public:
    FixedUIElementList() : m_ValidItemCount(0)
    {
        NW_ASSERT(TItemCount >= 0);
        memset(&m_pItems, 0, sizeof(m_pItems));
    }

    FixedUIElementList(const UIElement** ppItems, s32 count) : m_ValidItemCount(0)
    {
        NW_ASSERT(TItemCount >= 0);
        memset(&m_pItems, 0, sizeof(m_pItems));

        AddItems(ppItems, count);
    }

    virtual ~FixedUIElementList() { }

public:
    virtual s32 GetCount() const { return m_ValidItemCount; }

    virtual UIElement* GetItem(s32 index) const
    {
        return index >= m_ValidItemCount ? NULL : m_pItems[index];
    }

    virtual bool AddItem(UIElement* pItem)
    {
        NW_ASSERT(pItem != NULL);

        if(m_ValidItemCount >= TItemCount)
        {
            NW_ASSERTMSG(false, "failed to AddItem.(over limit)");
            return false;
        }

        s32 index = m_ValidItemCount;

        m_ValidItemCount++;
        SetItemInternal(index, pItem);
        return true;
    }

    virtual bool InsertItem(s32 index, UIElement* pItem)
    {
        NW_ASSERT(pItem != NULL);

        if(index < 0 || index > m_ValidItemCount + 1)
        {
            NW_ASSERTMSG(false, "failed to InsertItem.(invalid index)");
            return false;
        }

        if(m_ValidItemCount >= TItemCount)
        {
            NW_ASSERTMSG(false, "failed to InsertItem.(over limit)");
            return false;
        }

        // 挿入インデックスより後ろのアイテムをシフトします。
        s32 shiftItemCount = m_ValidItemCount - index + 1;
        if(shiftItemCount > 0)
        {
            memcpy(m_pItems + index + 1, m_pItems + index, sizeof(UIElement*) * shiftItemCount);
        }

        m_ValidItemCount++;
        SetItemInternal(index, pItem);

        return true;
    }

    virtual bool SetItem(s32 index, UIElement* pItem)
    {
        if(index < 0 || index >= m_ValidItemCount)
        {
            NW_ASSERTMSG(false, "failed to SetItem.(invalid index)");
            return false;
        }

        SetItemInternal(index, pItem);
        return true;
    }

    virtual bool AddItems(UIElement* const* ppItems, s32 count)
    {
        if(count < 0)
        {
            NW_ASSERTMSG(false, "failed to AddItems.(invalid parameter)");
            return false;
        }

        if(m_ValidItemCount + count >= TItemCount)
        {
            NW_ASSERTMSG(false, "failed to AddItems.(over limit)");
            return false;
        }

        for(s32 i=0; i<count; ++i)
        {
            AddItem(ppItems[i]);
        }

        return true;
    }

    virtual bool SetItems(s32 index, UIElement* const* ppItems, s32 count)
    {
        if(index < 0 || count < 0 || index + count >= m_ValidItemCount)
        {
            NW_ASSERTMSG(false, "failed to SetItems.");
            return false;
        }

        for(s32 i=0; i<count; ++i)
        {
            SetItemInternal(i + index, ppItems[i]);
        }

        return true;
    }

    virtual bool ClearItems()
    {
        for(int i = m_ValidItemCount - 1; i >= 0; --i)
        {
            SetItemInternal(i, NULL);
        }

        m_ValidItemCount = 0;

        return true;
    }

    virtual s32 IndexOf(const UIElement* pItem) const
    {
        NW_ASSERT(pItem != NULL);

        for(s32 i=0; i<m_ValidItemCount; ++i)
        {
            if(GetItem(i) == pItem)
            {
                return i;
            }
        }

        return -1;
    }

    virtual UIElement* operator [] (s32 index) const
    {
        return GetItem(index);
    }

    virtual ItemsChangedEvent& GetItemsChangedEvent()
    {
        return m_ItemsChangedEvent;
    }

protected:
    virtual void OnItemsChanged(UIElementListEventArgs& args)
    {
        GetItemsChangedEvent().Invoke(args);
    }

private:
    void SetItemInternal(s32 index, UIElement* pItem)
    {
        NW_ASSERT(0 <= index && index < m_ValidItemCount && index < TItemCount);

        UIElement* pOldItem = m_pItems[index];

        if(pOldItem == pItem)
        {
            return;
        }

        if(pOldItem != NULL)
        {
            UIElementListEventArgs args(UIElementListEventArgs::REMOVE_ITEM, index, pOldItem);
            OnItemsChanged(args);
        }

        m_pItems[index] = pItem;

        if(pItem != NULL)
        {
            UIElementListEventArgs args(UIElementListEventArgs::INSERT_ITEM, index, pItem);
            OnItemsChanged(args);
        }
    }

private:
    s32 m_ValidItemCount;
    UIElement* m_pItems[TItemCount];
    ItemsChangedEvent m_ItemsChangedEvent;
};

template<>
class FixedUIElementList<0> : public UIElementList
{
public:
    static FixedUIElementList<0>& Empty();

private:
    FixedUIElementList() { }

public:
    virtual s32 GetCount() const { return 0; }
    virtual UIElement* GetItem(s32 index) const { (void)index; return NULL; }
    virtual bool AddItem(UIElement* pItem) { (void)pItem; return false; }
    virtual bool InsertItem(s32 index, UIElement* pItem) { (void)index; (void)pItem; return false; }
    virtual bool SetItem(s32 index, UIElement* pItem) { (void)index; (void)pItem; return false; }
    virtual bool AddItems(UIElement* const* ppItems, s32 count) { (void)ppItems; (void)count; return false; }
    virtual bool SetItems(s32 index, UIElement* const* ppItems, s32 count) { (void)index; (void)ppItems; (void)count; return false; }
    virtual bool ClearItems() { return false; }
    virtual s32 IndexOf(const UIElement* pItem) const { (void)pItem; return INVALID_INDEX; }

    virtual UIElement* operator [] (s32 index) const { (void)index; return NULL; }

    virtual ItemsChangedEvent& GetItemsChangedEvent() { return m_ItemsChangedEvent; }

private:
    ItemsChangedEvent m_ItemsChangedEvent;
};

#define UIELEMENTLIST_EMPTY FixedUIElementList<0>::Empty()

} // dw
} // internal
} // nw
