﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_DRAW_METER_H_
#define NW_DEV_DRAW_METER_H_

#include <nw/dev/dev_LoadMeter.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/dev/dev_DevTextWriter.h>
#include <nw/ut/ut_Memory.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        負荷メーターを描画するクラスです。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
class MeterDrawer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    frameMeter フレーム全体時間を計測する負荷メーターです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit MeterDrawer( LoadMeterBase* frameMeter )
      : m_TextWriter( NULL ),
        m_ProjMtx( NULL ),
        m_ViewMtx( NULL ),
        m_ModelMtx( nw::math::MTX34::Identity() ),
        m_Position( 10, 10 ),
        m_Width( 1260.f ),
        m_Height( 48.f ),
        m_BarHeight( 16.f ),
        m_FontSize( 14.f ),
        m_SectionNum( 0 ),
        m_CurrentSectionNum( 0 ),
        m_AdjustTime( 60 ),
        m_AdjustCounter( m_AdjustTime ),
        m_MinSectionNum( 1 ),
        m_MaxSectionSeparatorNum( 4 ),
        m_Visible( true ),
        m_TextVisible( true ),
        m_BorderColor( nw::ut::Color4u8::WHITE ),
        m_TextColor( nw::ut::Color4u8::WHITE ),
        m_FrameMeter( frameMeter )
    {
        NW_ASSERT_NOT_NULL( frameMeter );
        SetFrameRate( 60.f );
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~MeterDrawer() {}


    //---------------------------------------------------------------------------
    //! @brief        描画する負荷メーターを登録します。
    //!
    //! @param[in]    meter     登録する負荷メーターです。
    //---------------------------------------------------------------------------
    void AttachLoadMeter( LoadMeterBase* meter );

    //---------------------------------------------------------------------------
    //! @brief        登録されている負荷メーターを解除します。
    //!
    //! @param[in]    meter     登録を解除する負荷メーターです。
    //---------------------------------------------------------------------------
    void DetachLoadMeter( LoadMeterBase* meter );

    //---------------------------------------------------------------------------
    //! @brief    フレーム全体時間を計測する負荷メーターを設定します。
    //!
    //! @param[in]    meter     フレーム全体時間を計測する負荷メーターです。
    //---------------------------------------------------------------------------
    void SetFrameMeter( LoadMeterBase* meter ) { m_FrameMeter = meter; }

    //---------------------------------------------------------------------------
    //! @brief    フレーム終了時の処理です。
    //!
    //!           ここでは、登録されている負荷メーターの OnEndFrame を呼びます。
    //---------------------------------------------------------------------------
    void EndFrame();


    //---------------------------------------------------------------------------
    //! @brief    負荷メーター全体の表示/非表示を設定します。
    //!
    //! @param[in]    visible   true を指定した場合、負荷メーターを表示します。
    //---------------------------------------------------------------------------
    void SetVisible( bool visible ){ m_Visible = visible; }

    //---------------------------------------------------------------------------
    //! @brief    負荷メーターの表示/非表示状態を取得します。
    //!
    //! @return   表示状態の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsVisible() const { return m_Visible; }

    //---------------------------------------------------------------------------
    //! @brief    各負荷メーターの名前や処理率の表示/非表示を設定します。
    //!
    //! @param[in]    visible   true を指定した場合、メーター名や処理率を表示します。
    //---------------------------------------------------------------------------
    void SetTextVisible( bool visible ){ m_TextVisible = visible; }

    //---------------------------------------------------------------------------
    //! @brief    各負荷メーターの名前や処理率の表示/非表示状態を取得します。
    //!
    //! @return   表示状態の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsTextVisible() const { return m_TextVisible; }


    //---------------------------------------------------------------------------
    //! @brief        文字描画に用いる DevTextWriter を設定します。
    //!
    //! @param[in]    textWriter 設定する DevTextWriter です。
    //---------------------------------------------------------------------------
    void SetTextWriter( nw::dev::DevTextWriter* textWriter ) { m_TextWriter = textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        設定されている DevTextWriter を取得します。
    //!
    //! @return       設定されている DevTextWriter を返します。
    //---------------------------------------------------------------------------
    nw::dev::DevTextWriter* GetTextWriter() const { return m_TextWriter; }

    //---------------------------------------------------------------------------
    //! @brief        描画用の行列を設定します。
    //!
    //! @param[in]    projMtx   プロジェクション行列です。
    //! @param[in]    viewMtx   ビュー行列です。
    //! @param[in]    modelMtx  変換行列です。
    //---------------------------------------------------------------------------
    void SetMatrix(
        const nw::math::MTX44& projMtx,
        const nw::math::MTX34& viewMtx,
        const nw::math::MTX34& modelMtx = nw::math::MTX34::Identity()
    )
    {
        m_ProjMtx = &projMtx;
        m_ViewMtx = &viewMtx;
        m_ModelMtx = modelMtx;
    }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの表示のために用いるフレームレートを設定します。
    //!
    //! @param[in]    frameRate 設定するフレームレートです。
    //---------------------------------------------------------------------------
    void SetFrameRate( f32 frameRate )
    {
        m_ReferenceFrame = nw::ut::TimeSpan::FromMicroSeconds( static_cast<s64>( 1000000.f / frameRate ) );
    }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画位置を設定します。
    //!
    //! @param[in]    pos       設定する描画位置です。
    //---------------------------------------------------------------------------
    void SetPosition( nw::math::VEC2 pos ) { m_Position = pos; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画位置を取得します。
    //!
    //! @return       描画位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetPosition() const { return m_Position; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画幅を設定します。
    //!
    //! @param[in]    width     設定する描画幅です。
    //---------------------------------------------------------------------------
    void SetWidth( f32 width ) { m_Width = width; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画幅を取得します。
    //!
    //! @return       描画幅を返します。
    //---------------------------------------------------------------------------
    f32 GetWidth() const { return m_Width; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画高さを設定します。
    //!
    //!               SetMinimizeBarSize で 0 が設定されている場合に SetHeight で設定した値が使用されます。
    //!               各負荷メーターの高さは ( SetHeight で設定した値 / 負荷メーターの数 ) となります。
    //!
    //! @param[in]    height     設定する描画高さです。
    //---------------------------------------------------------------------------
    void SetHeight( f32 height ) { m_Height = height; }

    //---------------------------------------------------------------------------
    //! @brief        設定されている負荷メーターの描画高さを取得します。
    //!
    //! @return       描画高さを返します。
    //---------------------------------------------------------------------------
    f32 GetHeight() const { return m_Height; }

    //---------------------------------------------------------------------------
    //! @brief        自動調整された負荷メーターの描画高さを取得します。
    //!
    //! @return       自動調整された描画高さを返します。
    //---------------------------------------------------------------------------
    f32 GetAdjustedHeight() const;

    //---------------------------------------------------------------------------
    //! @brief        文字列のフォントサイズを指定します。
    //!
    //! @param[in]    size      設定するフォントサイズです。
    //---------------------------------------------------------------------------
    void SetFontSize( f32 size ) { m_FontSize = size; }

    //---------------------------------------------------------------------------
    //! @brief        文字列のフォントサイズを取得します。
    //!
    //! @return       フォントサイズを返します。
    //---------------------------------------------------------------------------
    f32 GetFontSize() const { return m_FontSize; }

    //---------------------------------------------------------------------------
    //! @brief        画面に表示するフレームの目盛りの数を設定します。
    //!
    //!               sectionNum に 0 を指定した場合、すべての負荷メーターが表示できる
    //!               最低限の目盛り数に自動調整します。
    //!
    //! @param[in]    sectionNum 目盛りの数です。
    //---------------------------------------------------------------------------
    void SetSectionNum( u32 sectionNum ){ m_SectionNum = sectionNum; }

    //---------------------------------------------------------------------------
    //! @brief        設定されているフレームの目盛りの数を取得します。
    //!
    //! @return       目盛りの数を返します。
    //---------------------------------------------------------------------------
    u32 GetSectionNum() const { return m_SectionNum; }

    //---------------------------------------------------------------------------
    //! @brief        フレームを区切る線の最大本数を設定します。
    //!
    //!               極端に処理落ちした際に、区切り線が大量に表示されて処理落ちが続く問題を回避します。
    //!               0 を指定すると最大数の制限を行いません。
    //!
    //! @param[in]    maxNum    最大本数です。
    //---------------------------------------------------------------------------
    void SetMaxSectionSeparatorNum( u32 maxNum ) { m_MaxSectionSeparatorNum = maxNum; }

    //---------------------------------------------------------------------------
    //! @brief        フレームを区切る線の最大本数を取得します。
    //!
    //! @return       最大本数を返します。
    //---------------------------------------------------------------------------
    u32 GetMaxSectionSeparatorNum() const { return m_MaxSectionSeparatorNum; }

    //---------------------------------------------------------------------------
    //! @brief        目盛りの数が自動調整の場合、自動調整するまでのフレーム数を指定します。
    //!
    //!               自動調整するまでのフレーム数を指定することにより、
    //!               フレームの目盛り表示がチャタリングするのを防ぎます。
    //!
    //! @param[in]    adjustMargin 自動調整するまでのフレーム数です。
    //---------------------------------------------------------------------------
    void SetSectionNumAdjustMargin( s32 adjustMargin ) { m_AdjustTime = adjustMargin; }

    //---------------------------------------------------------------------------
    //! @brief        目盛りの数を自動調整するまでのフレーム数を取得します。
    //!
    //! @return       フレーム数を返します。
    //---------------------------------------------------------------------------
    s32 GetSectionNumAdjustMargin() const { return m_AdjustTime; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メーター1つあたりの高さを設定します。
    //!
    //! @param[in]    barSize   負荷メーター1つあたりの高さです。
    //---------------------------------------------------------------------------
    void SetMinimizeBarSize( f32 barSize ){ m_BarHeight = barSize; };

    //---------------------------------------------------------------------------
    //! @brief        負荷メーター1つあたりの高さを取得します。
    //!
    //! @return       負荷メーター1つあたりの高さを返します。
    //---------------------------------------------------------------------------
    f32 GetMinimizeBarSize() const { return m_BarHeight; };

    //---------------------------------------------------------------------------
    //! @brief        枠線の色を設定します。
    //!
    //! @param[in]    color     枠線の色です。
    //---------------------------------------------------------------------------
    void SetBorderColor( const nw::ut::Color4u8& color ){ m_BorderColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        枠線の色を取得します。
    //!
    //! @return       枠線の色を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8& GetBorderColor() const { return m_BorderColor; }

    //---------------------------------------------------------------------------
    //! @brief        文字列描画の色を設定します。
    //!
    //! @param[in]    color     文字列描画の色です。
    //---------------------------------------------------------------------------
    void SetTextColor( const nw::ut::Color4u8& color ){ m_TextColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        文字列描画の色を取得します。
    //!
    //! @return       文字列描画の色を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8& GetTextColor() const { return m_TextColor; }


    //---------------------------------------------------------------------------
    //! @brief        負荷メーターの描画を行います。
    //---------------------------------------------------------------------------
    void Draw();

    //---------------------------------------------------------------------------
    //! @brief        計測結果をログに出力します。
    //---------------------------------------------------------------------------
    void Dump();


private:
    typedef nw::ut::LinkList<LoadMeterBase, offsetof(LoadMeterBase, m_linkNode)> LoadMeterList;

    //! @brief ある時刻tがバー上の座標でどこに来るか計算します。
    f32 CalcTimeLinePos( f32 maxWidth, u32 sectionNum, u64 time );
    //! @brief ある時間tのバー上での幅を計算します。
    f32 CalcTimeLineWidth( f32 maxWidth, u32 sectionNum, u64 time );
    //! @brief 表示する必要がある目盛りの数を計算します。
    u32 CalcMaxSectionNum();

    nw::dev::DevTextWriter* m_TextWriter;       //!< 文字列描画に用いる DevTextWriter です。
    const nw::math::MTX44* m_ProjMtx;           //!< 描画用プロジェクション行列です。
    const nw::math::MTX34* m_ViewMtx;           //!< 描画用ビュー行列です。
    nw::math::MTX34        m_ModelMtx;          //!< 描画用変換行列です。

    nw::math::VEC2   m_Position;                //!< 負荷メーターの位置です。
    f32              m_Width;                   //!< 負荷メーターの幅です。
    f32              m_Height;                  //!< 負荷メーターの高さです。
    f32              m_BarHeight;               //!< 各負荷メーターの高さです。
    f32              m_FontSize;                //!< 文字列のフォントサイズです。

    u32              m_SectionNum;              //!< 何区切り分を表示するかです。 0 の場合は、自動で表示範囲を変えます。
    u32              m_CurrentSectionNum;       //!< 自動区切りとき、現在の区切り数です。
    s32              m_AdjustTime;              //!< 自動区切りの区切り数をアジャストするまでの時間です。
    s32              m_AdjustCounter;           //!< あと何フレームでアジャストするかのカウンターです。
    u32              m_MinSectionNum;           //!< 最低でもこの数だけ目盛りを表示します。
    u32              m_MaxSectionSeparatorNum;  //!< 区切り線の最大本数です。

    nw::ut::TimeSpan m_ReferenceFrame;          //!< 1 フレームのチックです。

    bool             m_Visible;                 //!< 負荷メーター全体を表示するかです。
    bool             m_TextVisible;             //!< テキストを表示するかです。

    nw::ut::Color4u8 m_BorderColor;             //!< 枠線の色です。
    nw::ut::Color4u8 m_TextColor;               //!< 文字列色です。

    LoadMeterList    m_LoadMeterList;           //!< 描画する負荷メーターのリストです。
    LoadMeterBase*   m_FrameMeter;              //!< フレーム全体時間を計測する負荷メーターです。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_DRAW_METER_H_
