﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief YouTubeVideo を開くのに利用される引数の宣言
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/web/web_Types.h>

namespace nn { namespace web {

/**
 * @brief YouTubeVideo を開くための情報を構築するクラスです。
 */
class ShowYouTubeVideoArg
{
public:
    /**
     * @brief NULL 終端を含むリクエストできる URL の最大バイト数です。
     */
    static const size_t RequestUrlMaxLength = 3072;

    /**
     * @brief YouTubeVideo を開くための最低限の情報を持ったクラスを作成します。
     *
     * @param[in]   pRequestUrl    開く Video の URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          RequestUrlMaxLength -1 文字以下にする必要があります。
     */
    explicit ShowYouTubeVideoArg(const char* pRequestUrl) NN_NOEXCEPT;

    /**
     * @brief 起動時の画面の種類を指定します。
     *
     * @param[in]   bootDisplayKind  起動時画面の種類
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetBootDisplayKind(const WebBootDisplayKind bootDisplayKind) NN_NOEXCEPT;

    /**
     * @brief 動画が最後まで再生された際に、動画プレイヤーを自動的に終了するかどうかを設定します。
     *
     * @param[in] autoCloseEnabled 自動終了を有効にする場合は true を指定してください。
     *
     * @details デフォルト値は false です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetMediaPlayerAutoCloseEnabled(const bool autoCloseEnabled) NN_NOEXCEPT;

    /**
     * @brief メディアプレイヤーで再生速度を変更できるようにするかどうかを設定します。
     *
     * @param[in]   mediaPlayerSpeedControlEnabled  メディアプレイヤーで再生速度を変更できるようにする場合は true を指定してください。
     *
     * @details デフォルト値は false です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetMediaPlayerSpeedControlEnabled(const bool mediaPlayerSpeedControlEnabled) NN_NOEXCEPT;

    /**
     * @brief ウェブページとして表示するかを指定します。
     *
     * @param[in]   bootAsWebPage ウェブページとして表示する場合は true を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetBootAsWebPage(const bool bootAsWebPage) NN_NOEXCEPT;

    //! @brief 内部用機能のため使用禁止です。
    const uint8_t* Data() const NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];

private:
    /**
     * @internal
     */
    ShowYouTubeVideoArg() NN_NOEXCEPT;
};

}} // namespace nn::web
