﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "repair_CryptUtility.h"
#include "repair_FileSystem.h"
#include "repair_IFile.h"
#include "repair_ProtectedFileEncryptor.h"
#include <nn/repair/repair_Authentication.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/crypto.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <string>
#include <algorithm>
#include <memory>
#include <random>

namespace nn
{
    namespace repair
    {
        struct ProtectedFileHeader
        {
            static const size_t SIGNATURE_SIZE = 16;
            static const uint64_t CURRENT_VERSION = 1;
            static const char SIGNATURE[SIGNATURE_SIZE];

            char Signature[SIGNATURE_SIZE];
            Cmac HeaderCmac;
            AuthenticationArchiveContent AuthticationContent;

            uint64_t Version;
            uint64_t DataSize;
            uint64_t BlockSize;
            uint64_t NumberOfBlocks;

            static ProtectedFileHeader Make(Cmac headerCmac, const AuthenticationArchiveContent &content, uint64_t version, uint64_t dataSize, uint64_t blockSize, uint64_t numBlocks);
        };

        class ProtectedFile : public IFile
        {
        public:
            ProtectedFile();
            virtual ~ProtectedFile() { Close(); }

            nn::Result OpenWrite(std::shared_ptr<FileSystem> fs, const char* path, std::shared_ptr<IProtectedFileEncryptor> pEncryptor, const Key128 &key, const AuthenticationArchiveContent &content);
            nn::Result OpenRead(std::shared_ptr<FileSystem> fs, const char* path, std::shared_ptr<IProtectedFileEncryptor> pEncryptor, const Key128 &key);
            virtual void Close() override;
            virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, bool flush) override;
            virtual nn::Result Read(size_t* pOut, int64_t offset, void* buffer, size_t size) override;
            virtual nn::Result GetSize(int64_t* pOut) override;
            virtual nn::Result Flush() override;

            int64_t GetDataSize();
            int64_t GetBlockCount();

            nn::Result Finish();
            static const uint64_t GetBlockSize(){ return BlockSize; };

        private:
            nn::Result LoadHeader(const Key128 &encryptedKey);
            nn::Result WriteHeader();
            nn::Result ReadBlock(void* pOut, int64_t blockIndex, size_t size);
            nn::Result WriteBlock(int64_t blockIndex, int64_t offset, const void* buffer, size_t size, bool flush);
            void UpdateMetaData(int64_t blockIndex, size_t writeSize);
            bool IsValidBlockIndex(int64_t blockIndex);

            bool m_IsFinished;
            int m_Mode;
            int64_t m_NumBlocks;
            int64_t m_DataSize;
            static const uint64_t BlockSize = 1 * 1024 * 1024;
            Key128 m_EncryptedKey;
            AuthenticationArchiveContent m_AuthContent;
            std::shared_ptr<IFile> m_RawFile;
            std::shared_ptr<IProtectedFileEncryptor> m_pEncryptor;
        };

        nn::Result CreateProtectedFile(std::shared_ptr<ProtectedFile> *pOut, std::shared_ptr<FileSystem> fs, const char * path, std::shared_ptr<nn::repair::IProtectedFileEncryptor> pEncryptor, const Key128 &key, const AuthenticationArchiveContent &content);
        nn::Result OpenProtectedFile(std::shared_ptr<ProtectedFile> *pOut, std::shared_ptr<FileSystem> fs, const char* path, std::shared_ptr<nn::repair::IProtectedFileEncryptor> pEncryptor, const Key128 &key);
        nn::Result LoadEncryptedKey(AuthenticationArchiveContent *pOut, std::string protectedFilePath, std::shared_ptr<FileSystem> fs);
    }
}
