﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace nd { namespace detail {

struct InitializationManager
{
    nn::os::SdkMutexType _mutex;
    uint32_t _counter;

    NN_EXPLICIT_OPERATOR bool() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(_mutex);
        return _counter > 0;
    }

    template <typename Initializer>
    void Initialize(Initializer initializer) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(_mutex);
        NN_ABORT_UNLESS(_counter < std::numeric_limits<uint32_t>::max(), "Too many Initilize() call.\n");
        if( _counter == 0 )
        {
            initializer();
        }
        _counter++;
    }

    template <typename Finalizer>
    void Finalize(Finalizer finalizer) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(_mutex);
        NN_ABORT_UNLESS(_counter > 0, "Too many Finalize() call.\n");
        _counter--;
        if( _counter == 0 )
        {
            finalizer();
        }
    }
};

#define NN_ND_DETAIL_INITIALIZATION_MANAGER_INITIALIZER {NN_OS_SDK_MUTEX_INITIALIZER(), 0u}

}}} // ~namespace nn::nd::detail
