﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_AbstractStorage.h>
#include <nn/migration/user/migration_UserMigrationInternalTypes.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>

namespace nn { namespace migration { namespace user {

class StateManager;

}}} // ~namespace nn::migration::user


namespace nn { namespace migration { namespace user {

class StateManager
{
    NN_DISALLOW_COPY(StateManager);
    NN_DISALLOW_MOVE(StateManager);

private:
    class ServerStateController
        : public StateController
    {
        NN_DISALLOW_COPY(ServerStateController);
        NN_DISALLOW_MOVE(ServerStateController);
    private:
        StateManager& m_Manager;
    public:
        explicit ServerStateController(StateManager& manager) NN_NOEXCEPT
            : m_Manager(manager)
        {
        }
        virtual Result SetStateInInitialization() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateServerState(MigrationState_InInitialization);
        }
        virtual Result SetStateInTransfer() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateServerState(MigrationState_InTransfer);
        }
        virtual Result SetStateInFinalization() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateServerState(MigrationState_InFinalization);
        }
        virtual Result SetStateFinalized() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateServerState(MigrationState_Finalized);
        }
        virtual void Reset() NN_NOEXCEPT final NN_OVERRIDE
        {
            m_Manager.ResetServer();
        }
        virtual bool IsResumable() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadServerState(&state);
            return r &&
                (false
                    || state == MigrationState_InInitialization
                    || state == MigrationState_InTransfer
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsInitialized() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadServerState(&state);
            return r &&
                (false
                    || state == MigrationState_InTransfer
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsTransferDone() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadServerState(&state);
            return r &&
                (false
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsFinalized() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadServerState(&state);
            return r && state == MigrationState_Finalized;
        }
    };

    class ClientStateController
        : public StateController
    {
        NN_DISALLOW_COPY(ClientStateController);
        NN_DISALLOW_MOVE(ClientStateController);
    private:
        StateManager& m_Manager;
    public:
        explicit ClientStateController(StateManager& manager) NN_NOEXCEPT
            : m_Manager(manager)
        {
        }
        virtual Result SetStateInInitialization() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateClientState(MigrationState_InInitialization);
        }
        virtual Result SetStateInTransfer() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateClientState(MigrationState_InTransfer);
        }
        virtual Result SetStateInFinalization() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateClientState(MigrationState_InFinalization);
        }
        virtual Result SetStateFinalized() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.UpdateClientState(MigrationState_Finalized);
        }
        virtual void Reset() NN_NOEXCEPT final NN_OVERRIDE
        {
            return m_Manager.ResetClient();
        }
        virtual bool IsResumable() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadClientState(&state);
            return r &&
                (false
                    || state == MigrationState_InInitialization
                    || state == MigrationState_InTransfer
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsInitialized() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadClientState(&state);
            return r &&
                (false
                    || state == MigrationState_InTransfer
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsTransferDone() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadClientState(&state);
            return r &&
                (false
                    || state == MigrationState_InFinalization
                    || state == MigrationState_Finalized);
        }
        virtual bool IsFinalized() const NN_NOEXCEPT final NN_OVERRIDE
        {
            MigrationState state;
            auto r = m_Manager.TryLoadClientState(&state);
            return r && state == MigrationState_Finalized;
        }
    };

private:
    const detail::AbstractStorage& m_Storage;
    ServerStateController m_ServerController;
    ClientStateController m_ClientController;

    Result TryLoadLastMigrationServerState(bool* pOutExists, MigrationState* pOut) const NN_NOEXCEPT;
    Result TryLoadLastMigrationClientState(bool* pOutExists, MigrationState* pOut) const NN_NOEXCEPT;

    bool TryLoadServerState(MigrationState* pOut) const NN_NOEXCEPT;
    bool TryLoadClientState(MigrationState* pOut) const NN_NOEXCEPT;
    void StoreServerState(MigrationState state) const NN_NOEXCEPT;
    void StoreClientState(MigrationState state) const NN_NOEXCEPT;

    Result UpdateServerState(MigrationState state) const NN_NOEXCEPT;
    Result UpdateClientState(MigrationState state) const NN_NOEXCEPT;
    void ResetServer() const NN_NOEXCEPT;
    void ResetClient() const NN_NOEXCEPT;

public:
    explicit StateManager(const detail::AbstractStorage& storage) NN_NOEXCEPT
        : m_Storage(storage)
        , m_ServerController(*this)
        , m_ClientController(*this)
    {
    }
    Result TryGetLastMigrationInfo(bool* pOut, LastMigrationInfo* pOutInfo) const NN_NOEXCEPT;

    Result StartServer(StateController** pOut) NN_NOEXCEPT;
    Result ResumeServer(StateController** pOut) NN_NOEXCEPT;

    Result StartClient(StateController** pOut) NN_NOEXCEPT;
    Result ResumeClient(StateController** pOut) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::user
