﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_NetworkConnection.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationIdcServer.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace migration { namespace user {

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer>
class Server
{
private:
    static const int ResetDelayMilliSeconds = 3 * 1000;

    // アカウントへのログインセッション
    detail::LocalUserLoginSession m_LoginSession;

    // サブシステムの初期化
    SubsystemInitializer m_SubsystemInitializer;

    // ローカル通信の接続管理
    os::Mutex m_ConnectionLock;
    std::atomic<bool> m_IsConnected;
    typename ConnectionPolicy::Connection m_Connection;
    typename ConnectionPolicy::ConnectionCreator m_ConnectionCreator;

    // 対向デバイスとの通信モジュール
    IdcServer<typename ConnectionPolicy::Connection, EncryptionPolicy> m_Idc;

    // 移行の状態管理
    typename ContextStoragePolicy::Storage m_ContextStorage;
    ServerContext m_Context;
    StateController& m_StateController;

    // 動作に必要なメモリ
    struct Resource
    {
        union
        {
            std::aligned_storage<1024 * 1024>::type contextWorkBuffer; // 1MiB
        } u;
    } m_Resource;

private:
    void Connect(typename ConnectionPolicy::Connection&& connection) NN_NOEXCEPT;
    void Disconnect(TimeSpan resetDelay) NN_NOEXCEPT;
    void CleanupContext() NN_NOEXCEPT;

public:
    Server(StateController& stateController, detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT;

    Result Initialize(const account::Uid& uid, const UserMigrationServerProfile& profile) NN_NOEXCEPT;
    Result Resume() NN_NOEXCEPT;

    account::Uid GetUid() const NN_NOEXCEPT;
    void GetServerProfile(UserMigrationServerProfile* pOut) const NN_NOEXCEPT;

    Result Prepare(const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    bool IsConnectionRequired() const NN_NOEXCEPT;
    Result WaitConnection(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    void GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT;
    Result AcceptConnection(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result DeclineConnection(const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result ProcessTransfer(const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result Complete(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Abort() NN_NOEXCEPT;
};

}}} // ~namesapce nn::migration::user

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace user {

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer>
inline Server<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy, SubsystemInitializer>::Server(StateController& stateController, detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
    : m_ConnectionLock(false)
    , m_IsConnected(false)
    , m_Idc(std::move(idcThreadResource))
    , m_Context(m_ContextStorage)
    , m_StateController(stateController)
{
}

#define NN_MIGRATION_USER_DEFINE_SERVER_METHOD(rtype, name) \
template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer> \
inline rtype Server<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy, SubsystemInitializer>::name NN_NOEXCEPT

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    Initialize(const account::Uid& uid, const UserMigrationServerProfile& profile))
{
    NN_SDK_REQUIRES(uid);
    NN_RESULT_THROW_UNLESS(!m_StateController.IsResumable(), detail::ResultResumableMigrationServerExists());

    m_ContextStorage.Cleanup();
    m_Context.CreateMigrationInfo(uid, profile);
    m_LoginSession.Initialize(m_Context.GetUid());

    NN_MIGRATION_DETAIL_TRACE("[Server] Initialized for user %016llx_%016llx\n", uid._data[0], uid._data[1]);
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    Resume())
{
    if (!m_Context.TryResume())
    {
        // コンテキストの破損やシステムバージョンの不一致を検知した場合
        NN_MIGRATION_DETAIL_THROW(detail::ResultUnresumableMigrationServer());
    }
    NN_SDK_ASSERT(m_StateController.IsResumable());
    m_LoginSession.Initialize(m_Context.GetUid());

    NN_MIGRATION_DETAIL_TRACE("[Server] Resumed for user %016llx_%016llx\n", m_Context.GetUid()._data[0], m_Context.GetUid()._data[1]);
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    void,
    Connect(typename ConnectionPolicy::Connection&& connection))
{
    std::lock_guard<os::Mutex> lock(m_ConnectionLock);
    NN_SDK_ASSERT(!m_IsConnected);

    m_Connection = std::move(connection);
    m_IsConnected = true;
    NN_MIGRATION_DETAIL_TRACE("[Server] Connected\n");
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    void,
    Disconnect(TimeSpan resetDelay))
{
    std::lock_guard<os::Mutex> lock(m_ConnectionLock);
    NN_SDK_ASSERT(m_IsConnected);

    m_Connection = decltype(m_Connection)();
    os::SleepThread(resetDelay);

    m_ConnectionCreator.Reset();
    m_IsConnected = false;
    NN_MIGRATION_DETAIL_TRACE("[Server] Disconnected\n");
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    void,
    CleanupContext())
{
    m_StateController.Reset();
    m_ContextStorage.Cleanup();

    NN_MIGRATION_DETAIL_TRACE("[Server] Context removed\n");
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    account::Uid,
    GetUid() const)
{
    return m_Context.GetUid();
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    void,
    GetServerProfile(UserMigrationServerProfile* pOut) const)
{
    m_Context.GetServerProfile(pOut);
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    Prepare(const detail::Cancellable* pCancellable))
{
    if (!m_StateController.IsResumable())
    {
        NN_MIGRATION_DETAIL_DO(m_Context.CreateMigrationList(
            &m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer),
            pCancellable));
        NN_MIGRATION_DETAIL_TRACE("[Server] Migration list size: %zu\n", m_Context.GetMigrationListSize());
    }

    account::NetworkServiceAccountId nsaId;
    NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));
    NN_MIGRATION_DETAIL_TRACE("[Server] NSA ID: %016llx\n", nsaId.id);

    NN_MIGRATION_DETAIL_DO(m_Idc.Initialize(m_LoginSession, pCancellable));
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    bool,
    IsConnectionRequired() const)
{
    return !m_StateController.IsFinalized();
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    WaitConnection(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Waiting for connection...\n");
    account::NetworkServiceAccountId nsaId;
    NN_MIGRATION_DETAIL_DO(m_LoginSession.GetNetworkServiceAccountId(&nsaId));

    UserMigrationServerProfile profile;
    m_Context.GetServerProfile(&profile);

    typename ConnectionPolicy::Connection connection;
    NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.template CreateConnection<EncryptionPolicy>(&connection, nsaId, profile, m_Idc, pCancellable));
    Connect(std::move(connection));

    if (m_StateController.IsResumable())
    {
        NN_MIGRATION_DETAIL_DO(m_Idc.MatchUserMigrationClientInfo(m_Context, m_Connection, pCancellable));
        NN_RESULT_SUCCESS;
    }

    NN_MIGRATION_DETAIL_DO(m_Idc.ImportUserMigrationClientInfo(m_Context, m_Connection, pCancellable));
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    void,
    GetClientProfile(UserMigrationClientProfile* pOut) const)
{
    m_Context.GetClientProfile(pOut);
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    AcceptConnection(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());

    NN_MIGRATION_DETAIL_DO(m_Idc.AcceptUserMigrationClient(m_Connection, pCancellable));

    NN_MIGRATION_DETAIL_DO(m_Idc.ExportUserMigrationServerInfo(m_Context, m_Connection, pCancellable));

    if (m_StateController.IsResumable())
    {
        NN_RESULT_SUCCESS;
    }

    NN_MIGRATION_DETAIL_DO(m_Idc.ExportMigrationList(m_Context, m_Connection, pCancellable));
    NN_MIGRATION_DETAIL_DO(m_Idc.WaitStateSychronizationInInitialization(m_StateController, m_Connection, pCancellable));
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    DeclineConnection(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());

    NN_MIGRATION_DETAIL_DO(m_Idc.DeclineUserMigrationClient(m_Connection, pCancellable));
    Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    ProcessTransfer(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());

    if (!m_StateController.IsInitialized())
    {
        NN_MIGRATION_DETAIL_TRACE("[Server] State <- InTransfer\n");
        NN_MIGRATION_DETAIL_DO(m_StateController.SetStateInTransfer());
    }
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_StateController.IsInitialized(), detail::ResultInvalidProtocol());

    do
    {
        // 再開時の状態同期のため、TransferDone の状態でも 1 度は要求を処理する必要がある。
        NN_MIGRATION_DETAIL_DO(m_Idc.ExportSaveData(m_StateController, m_Context, m_Connection, pCancellable));
    } while (!m_StateController.IsTransferDone());
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    Complete(const detail::Cancellable* pCancellable))
{
    NN_SDK_REQUIRES_NOT_NULL(pCancellable);

    NN_MIGRATION_DETAIL_THROW_UNLESS(m_StateController.IsTransferDone(), detail::ResultInvalidProtocol());
    if (m_IsConnected)
    {
        NN_MIGRATION_DETAIL_DO(m_Idc.WaitStateSychronizationFinalized(m_StateController, m_Connection, pCancellable));
        Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    }
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_StateController.IsFinalized(), detail::ResultInvalidProtocol());

    NN_MIGRATION_DETAIL_TRACE("[Server] Connecting to the internet\n");
    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    NN_MIGRATION_DETAIL_DO(detail::ConnectToInternet(connection, nifm::RequirementPreset_InternetBestEffort, pCancellable));

    NN_MIGRATION_DETAIL_TRACE("[Server] Cleaning up user data\n");
    NN_MIGRATION_DETAIL_DO(m_LoginSession.DeleteUser(pCancellable));

    CleanupContext();
    NN_MIGRATION_DETAIL_TRACE("[Server] Migration done\n");
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_SERVER_METHOD(
    Result,
    Abort())
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Aborting\n");
    if (m_IsConnected)
    {
        Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    }

    if (!m_StateController.IsTransferDone())
    {
        // 終了処理を開始していないのでコンテキストの削除のみ
        CleanupContext();
        NN_MIGRATION_DETAIL_TRACE("[Server] Aborted with keeping user account\n");
        NN_RESULT_SUCCESS;
    }

    NN_MIGRATION_DETAIL_TRACE("[Server] Connecting to the internet\n");
    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    NN_MIGRATION_DETAIL_DO(detail::ConnectToInternet(connection, nifm::RequirementPreset_InternetBestEffort, nullptr));

    NN_MIGRATION_DETAIL_TRACE("[Server] Cleaning up user data\n");
    NN_MIGRATION_DETAIL_DO(m_LoginSession.DeleteUser(nullptr));

    CleanupContext();
    NN_MIGRATION_DETAIL_TRACE("[Server] Aborted with user account deleted\n");
    NN_RESULT_SUCCESS;
}

#undef NN_MIGRATION_USER_DEFINE_SERVER_METHOD

}}} // ~namesapce nn::migration::user
