﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <new>
#include <type_traits>

#include <nn/migration/migration_UserMigrationServiceTypes.h>
#include <nn/migration/detail/migration_AsyncContextImpl.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/detail/migration_SharedResource.h>
#include <nn/migration/detail/migration_Interface.sfdl.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace migration { namespace user {
class StateController;
}}} // ~namespace nn::migration::user

namespace nn { namespace migration { namespace user {

template <typename ObjectAllocator, typename Impl>
class ProxyBase
    : public sf::IServiceObject
{
public:
    typedef sf::ObjectFactory<typename ObjectAllocator::Policy> FactoryType;
    typedef sf::SharedPointer<sf::IServiceObject> ParentPtr;

private:
    // 固定リソース
    ParentPtr m_pParent;
    detail::SharedResource<ObjectAllocator> m_ObjectAllocator;

    // 実装
    Impl* m_pImpl;

    // ランタイムのリソース
    os::TransferMemory m_Memory;
    size_t m_MemorySize;

    struct Resource
    {
        typename std::aligned_storage<sizeof(Impl), std::alignment_of<Impl>::value>::type implStorage;
    }* m_pResource;
    NN_STATIC_ASSERT(std::alignment_of<Resource>::value <= os::MemoryPageSize);
    NN_STATIC_ASSERT(sizeof(Resource) <= RequiredWorkBufferSizeForUserMigrationOperation);

    detail::AsyncExecutionResource m_AsyncExecResource;

public:
    static const size_t RequiredWorkMemorySize = sizeof(Resource);

protected:
    template <typename... Args>
    ProxyBase(
        ParentPtr&& pParent, detail::SharedResource<ObjectAllocator>&& objectAllocator, detail::ThreadResource&& threadResource,
        sf::NativeHandle&& transferMemoryHandle, size_t transferMemorySize,
        Args&&... args) NN_NOEXCEPT
        : m_pParent(std::move(pParent))
        , m_ObjectAllocator(std::move(objectAllocator))
        , m_pImpl(nullptr)
        , m_MemorySize(transferMemorySize)
        , m_AsyncExecResource(std::move(threadResource))
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(transferMemorySize, sizeof(Resource));
        m_Memory.Attach(m_MemorySize, transferMemoryHandle.GetOsHandle(), transferMemoryHandle.IsManaged());
        transferMemoryHandle.Detach();

        void* address;
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Memory.Map(&address, os::MemoryPermission_None));
        m_pResource = reinterpret_cast<decltype(m_pResource)>(address);

        m_pImpl = new(&m_pResource->implStorage) Impl(std::forward<Args>(args)...);
    }

    ~ProxyBase() NN_NOEXCEPT
    {
        m_pImpl->~Impl();
        std::memset(m_pResource, 0xDA, m_MemorySize);
        m_Memory.Unmap();
    }

    Impl& GetImplRef() NN_NOEXCEPT
    {
        return *m_pImpl;
    }
    const Impl& GetImplRef() const NN_NOEXCEPT
    {
        return *m_pImpl;
    }

    template <typename Task, typename... Arg>
    sf::EmplacedRef<detail::IAsyncContext, detail::AsyncContextImpl<Task>> CreateAsyncContext(Arg&&... args) NN_NOEXCEPT
    {
        return sf::ObjectFactory<typename ObjectAllocator::Policy>::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncContextImpl<Task>>(
            &m_ObjectAllocator.Get(), m_AsyncExecResource, std::forward<Arg>(args)...);
    }
};

template <typename ObjectAllocator, typename Impl>
const size_t ProxyBase<ObjectAllocator, Impl>::RequiredWorkMemorySize;

}}} // ~namesapce nn::migration::user
