﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace detail {

void ReportError(Result result, bool isAborting) NN_NOEXCEPT;

enum DiagnosisKind
{
    DiagnosisKind_Info,
    DiagnosisKind_Trace,
    DiagnosisKind_Warn,
    DiagnosisKind_Error,
};
void PrintWithTimeStamp(DiagnosisKind kind, const char* pFormat, ...) NN_NOEXCEPT;

}}} // ~nn::migration::detail

#define NN_MIGRATION_DETAIL_INFO(...) ::nn::migration::detail::PrintWithTimeStamp(::nn::migration::detail::DiagnosisKind_Info, __VA_ARGS__)
#define NN_MIGRATION_DETAIL_TRACE(...) ::nn::migration::detail::PrintWithTimeStamp(::nn::migration::detail::DiagnosisKind_Trace, __VA_ARGS__)
#define NN_MIGRATION_DETAIL_WARN(...) ::nn::migration::detail::PrintWithTimeStamp(::nn::migration::detail::DiagnosisKind_Warn, __VA_ARGS__)
#define NN_MIGRATION_DETAIL_ERROR(...) ::nn::migration::detail::PrintWithTimeStamp(::nn::migration::detail::DiagnosisKind_Error, __VA_ARGS__)

#define NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(expression) \
    do { \
        ::nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_MIGRATION_DETAIL_ERROR("<!> Aborting by %s failed with %03d-%04d (%08lx)\n", #expression, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ::nn::migration::detail::ReportError(_r, true); \
            NN_ABORT_UNLESS_RESULT_SUCCESS(_r); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NN_MIGRATION_DETAIL_DO(expression) \
    do { \
        ::nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_MIGRATION_DETAIL_ERROR( \
                "<!> " #expression " failed with %03d-%04d (%08lx)\n", \
                _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ::nn::migration::detail::ReportError(_r, false); \
            NN_RESULT_THROW(_r); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NN_MIGRATION_DETAIL_THROW_UNLESS(expression, result) \
    do { \
        bool _b = static_cast<bool>(expression); \
        if (!_b) \
        { \
            ::nn::Result _r = (result);\
            NN_MIGRATION_DETAIL_WARN( \
                "<!> " #expression " failed, then throwing %03d-%04d (%08lx)\n", \
                _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ::nn::migration::detail::ReportError(_r, false); \
            NN_RESULT_THROW(_r); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NN_MIGRATION_DETAIL_THROW(result) \
    do { \
        ::nn::Result _r = (result);\
        NN_MIGRATION_DETAIL_ERROR( \
            "<!> Throwing %03d-%04d (%08lx)\n", \
            _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
        ::nn::migration::detail::ReportError(_r, false); \
        NN_RESULT_THROW(_r); \
    } while (NN_STATIC_CONDITION(false))

#define NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(expression) \
    do \
    { \
        ::nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_MIGRATION_DETAIL_WARN( \
                "<!> " #expression " failed with %03d-%04d (%08lx)\n", \
                _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ::nn::migration::detail::ReportError(_r, false); \
        } \
    } while (NN_STATIC_CONDITION(false))
