﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Device.h>

namespace nn { namespace manu { namespace detail {

class UsbTransferPipe
{
public:
    ::nn::Result Initialize(uint8_t interfaceNumber) NN_NOEXCEPT;
    ::nn::Result Finalize() NN_NOEXCEPT;
    ::nn::Result Cancel() NN_NOEXCEPT;
    ::nn::Result Receive(uint32_t *pOutBytesTransferred, void *buffer, const uint32_t bufferLength) NN_NOEXCEPT;
    ::nn::Result Send(uint32_t *pOutBytesTransferred, const void *buffer, const uint32_t bufferLength) NN_NOEXCEPT;

private:
    enum class EndPointIndex : uint32_t
    {
        BulkIn  = 0,
        BulkOut = 1,
        MaxEndPointNum,
    };

    ::nn::usb::UsbStringDescriptor languageStringDescriptor            = {4,   ::nn::usb::UsbDescriptorType_String, {0x0409}};
    ::nn::usb::UsbStringDescriptor manufacturerStringDescriptor        = {18,  ::nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
    ::nn::usb::UsbStringDescriptor productStringDescriptor             = {40,  ::nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'D', 'e', 'b', 'u', 'g', 'g', 'e', 'r'}};
    ::nn::usb::UsbStringDescriptor serialNumberStringDescriptor        = {0,   ::nn::usb::UsbDescriptorType_String, {}};


    ::nn::usb::UsbDeviceDescriptor fullSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0110,                                                  // bcdUSB 1.1
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x40,                                                    // bMaxPacketSize0
        0x057e,                                                  // idVendor
        0x3000,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    ::nn::usb::UsbDeviceDescriptor highSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0200,                                                  // bcdUSB 2.0
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x40,                                                    // bMaxPacketSize0
        0x057e,                                                  // idVendor
        0x3000,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    ::nn::usb::UsbDeviceDescriptor superSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0300,                                                  // bcdUSB 3.0
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x09,                                                    // bMaxPacketSize0, SS 512 or 2^9
        0x057e,                                                  // idVendor
        0x3000,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    uint8_t binaryObjectStore[22] =
    {
        0x05,                                                    // bLength
        ::nn::usb::UsbDescriptorType_Bos,                        // bDescriptorType
        0x16,0x00,                                               // Length of this descriptor and all sub descriptors
        0x02,                                                    // Number of device capability descriptors

        // USB 2.0 extension
        0x07,                                                    // bLength
        ::nn::usb::UsbDescriptorType_DeviceCapability,           // bDescriptorType
        0x02,                                                    // USB 2.0 extension capability type
        0x02,0x00,0x00,0x00,                                     // Supported device level features: LPM support

        // SuperSpeed device capability
        0x0A,                                                    // bLength
        ::nn::usb::UsbDescriptorType_DeviceCapability,           // bDescriptorType
        0x03,                                                    // SuperSpeed device capability type
        0x00,                                                    // Supported device level features
        0x0e,0x00,                                               // Speeds supported by the device : SS, HS
        0x03,                                                    // Functionality support
        0x00,                                                    // U1 Device Exit latency
        0x00,0x00                                                // U2 Device Exit latency
    };

    ::nn::usb::UsbInterfaceDescriptor    usbInterfaceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Interface  ,               // bLength
        ::nn::usb::UsbDescriptorType_Interface  ,               // bDescriptorType
        0,                                                      // bInterfaceNumber assigned at runtime
        0,                                                      // bAlternateSetting
        2,                                                      // bNumEndpoints
        0xff,                                                   // bInterfaceClass
        0xff,                                                   // bInterfaceSubClass
        0xff,                                                   // bInterfaceProtocol
        0,                                                      // iInterface
    };


    ::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsFullSpeed[2] = {
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            64,                                                  // wMaxPacketSize
            0                                                    // bInterval
        },
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            64,                                                  // wMaxPacketSize
            0                                                    // bInterval
        },
    };


    ::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsHighSpeed[2] = {
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            512,                                                 // wMaxPacketSize
            0                                                    // bInterval
        },
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            512,                                                 // wMaxPacketSize
            0                                                    // bInterval
        },
    };


    ::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsSuperSpeed[2] = {
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            1024,                                                // wMaxPacketSize
            0                                                    // bInterval
        },
        {
            ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
            ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
            0,                                                   // bEndpointAddress assigned at runtime
            ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
            1024,                                                // wMaxPacketSize
            0                                                    // bInterval
        },
    };


    // Use this endpoint companion descriptor for both endpoints :)
    ::nn::usb::UsbEndpointCompanionDescriptor    usbSuperSpeedUsbEndpointCompanionDescriptor =
    {
        0x06,                                                    // bLength
        ::nn::usb::UsbDescriptorType_EndpointCompanion,          // bDescriptorType
        15,                                                      // bMaxBurst
        0,                                                       // bmAttributes
        0                                                        // wBytesPerInterval
    };

    ::nn::usb::DsClient                  dsClient;
    ::nn::usb::DsInterface               dsInterface;
    ::nn::usb::DsEndpoint                dsEndpoints[static_cast<int>(EndPointIndex::MaxEndPointNum)];

    bool CanTransfer() NN_NOEXCEPT;
    bool IsUrbEmpty(::nn::usb::DsEndpoint& endPoint) NN_NOEXCEPT;
    bool WaitXferCompletion(::nn::usb::DsEndpoint& endPoint, nn::TimeSpan timespan) NN_NOEXCEPT;
    bool WaitUsbStateEqualTo(nn::usb::UsbState usbStatus, nn::TimeSpan timespan) NN_NOEXCEPT;
    ::nn::Result Xfer(::nn::usb::DsEndpoint& endPoint, uint32_t *pOutBytesTransferred, void *buffer, const uint32_t bufferLength) NN_NOEXCEPT;
};

}}}
