﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_Types.h>

namespace nn { namespace ldn
{
    /**
     * @brief ローカル通信の動作モードです。
     *
     * @details
     * 動作モードは SetOperationMode() で変更できます。
     * アクセスポイントとステーションで動作モードが一致していなければ、
     * Connect() による接続が ResultInvalidNetwork を返して失敗します。
     *
     * 特に設定が無かった場合、デフォルトでは OperationMode_Stable を使用します。
     * このモードでは、スループットよりも通信の安定性を優先します。
     * 通信対戦など、大容量の通信が必要ない用途で、
     * パケット損失や遅延を抑えたい場合にはこちらのモードを使用してください。
     *
     * OperationMode_HighSpeed は通信の安定性よりもスループットを優先します。
     * 通信のリアルタイム性が要求されず、かつ、大容量のデータを送受信する場合には、
     * OperationMode_Stable よりも OperationMode_HighSpeed が適しています。
     */
    enum OperationMode
    {
        //! 安定通信モードです。
        OperationMode_Stable,

        //! 大容量転送モードです。
        OperationMode_HighSpeed,
    };

    /**
     * @brief 無線コントローラーの接続制限です。
     *
     * @details
     * デフォルトでは LDN ライブラリによって無線コントローラーの接続が制限されています。
     * NetworkConfig の nodeCountMax に 4 以下を設定したネットワークでは 4 台、
     * 5 以上を設定したネットワークでは 2 台までしか無線コントローラーを接続できません。
     *
     * この制限を回避するには WirelessControllerRestriction_Disabled を引数として
     * SetWirelessControllerRestriction() を実行してください。
     * これによって、最大数である 8 台まで無線コントローラーを接続できるようになりますが、
     * 適切に制御しなければローカル通信が不安定になる可能性があります。
     */
    enum WirelessControllerRestriction
    {
        //! LDN ライブラリによる無線コントローラーの接続制限を無効化します。
        WirelessControllerRestriction_Disabled,

        //! LDN ライブラリによる無線コントローラーの接続制限を有効化します。
        WirelessControllerRestriction_Enabled,
    };

    //! @name 高度な API
    //! @{

   /**
    * @brief         ローカル通信の動作モードを設定します。
    *
    * @param[in]     mode                   ローカル通信の動作モードです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a mode == OperationMode_Stable || @a mode == OperationMode_HighSpeed
    *
    * @details
    * ローカル通信の動作モードを設定します。
    * 動作モードの詳細については OperationMode のリファレンスを参照してください。
    *
    * この関数は State_Initialized 状態で実行してください。
    * それ以外の状態で実行すると ResultInvalidState を返します。
    * ローカル通信の動作モードは Finalize() で LDN ライブラリの使用を終了するまで有効です。
    * ネットワークを破棄したり切断したりしても元の設定にリセットされることはありません。
    *
    * @note
    * この機能を使用する場合は事前に任天堂の許可が必要です。
    * 許可なく使用した場合はロットチェックを通過できませんので注意してください。
    */
    Result SetOperationMode(OperationMode mode) NN_NOEXCEPT;

    /**
     * @brief         無線コントローラーの接続制限を有効化あるいは無効化します。
     *
     * @param[in]      restriction            接続制限の設定です。
     *
     * @retresult
     *  @handleresult{ResultSuccess}
     *  @handleresult{ResultInvalidState}
     *  @handleresult{ResultWifiOff}
     *  @handleresult{ResultSleep}
     *  @handleresult{ResultDeviceOccupied}
     * @endretresult
     *
     * @pre
     * - GetState() != @ref State_None
     * - @a restriction == WirelessControllerRestriction_Disabled ||
     *   @a restriction == WirelessControllerRestriction_Enabled
     *
     * @details
     * LDN ライブラリによる無線コントローラーの接続制限を有効化あるいは無効化します。
     * 詳細については WirelessControllerRestriction のリファレンスを参照してください。
     *
     * この関数は State_Initialized 状態で実行してください。
     * それ以外の状態で実行すると ResultInvalidState を返します。
     * 接続制限の設定は Finalize() で LDN ライブラリの使用を終了するまで有効です。
     * ネットワークを破棄したり切断したりしても元の設定にリセットされることはありません。
     *
     * @note
     * この機能を使用する場合は事前に任天堂の許可が必要です。
     * 許可なく使用した場合はロットチェックを通過できませんので注意してください。
     */
    Result SetWirelessControllerRestriction(
        WirelessControllerRestriction restriction) NN_NOEXCEPT;

    //! @}

}} // end of namespace nn::ldn
