﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/Authentication/ldn_Authentication.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameReceiver.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameSender.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         認証クライアントのインタフェースです。
     */
    class IAuthenticationClient
    {
    public:

        /**
         * @brief       スコープ内限定でクライアントとして動作します。
         */
        class ScopedClient
        {
        public:

            ScopedClient(
                IAuthenticationClient* pClient, const Bit8 (&clientRandom)[RandomSize],
                IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT
                : m_pClient(pClient)
            {
                m_pClient->BeginClient(clientRandom, pReceiver, pSender);
            }

            ~ScopedClient() NN_NOEXCEPT
            {
                if (m_pClient != nullptr)
                {
                    m_pClient->EndClient();
                }
            }

            void Detach() NN_NOEXCEPT
            {
                m_pClient = nullptr;
            }

        private:

            IAuthenticationClient* m_pClient;
        };

        /**
         * @brief       デストラクタです。
         */
        virtual ~IAuthenticationClient() NN_NOEXCEPT
        {
        }

        /**
         * @brief       クライアントとしての動作を開始します。
         * @param[in]   clientRandom    クライアントが生成した乱数です。
         * @param[in]   pReceiver       データ受信に使用するインタフェースです。
         * @param[in]   pSender         データ送信に使用するインタフェースです
         */
        virtual void BeginClient(
            const Bit8 (&clientRandom)[RandomSize],
            IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT = 0;

        /**
         * @brief       クライアントを終了します。
         */
        virtual void EndClient() NN_NOEXCEPT = 0;

        /**
         * @brief       認証時にサーバに送信するデータを設定します。
         * @param[in]   data            認証時にサーバに送信するデータです。
         * @param[in]   dataSize        data のサイズです。
         */
        virtual void SetData(const void* data, size_t dataSize) NN_NOEXCEPT = 0;

        /**
         * @brief       認証を試行します。
         * @param[out]  buffer          サーバから受信したデータです。
         * @param[out]  pOutSize        buffer に出力したサイズです。
         * @param[in]   bufferSize      buffer のバッファサイズです。
         * @param[in]   serverAddress   サーバの MAC アドレスです。
         * @param[in]   networkId       ネットワーク識別子です。
         * @param[in]   serverRandom    サーバが生成した乱数です。
         * @return      認証の結果です。
         */
        virtual AuthenticationResult Authenticate(
            void* buffer, size_t* pOutSize, size_t bufferSize,
            MacAddress serverAddress, const NetworkId& networkId,
            const Bit8 (&serverRandom)[RandomSize]) NN_NOEXCEPT = 0;
    };

}}} // namespace nn::ldn::detail
