﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/ldn_Version.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseBuilder.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseParser.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    struct AdvertiseDistributorBuffer
    {
        char builder[1408];
        char advertise[1408];
        char body[1280];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseDistributorBuffer) == 4096U);

    struct AdvertiseScannerBuffer
    {
        char parser[4 * 1024];
        L2ScanResult l2ScanResult[ScanResultCountMax];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseScannerBuffer) == 40960U);

    struct AdvertiseMonitorBuffer
    {
        char parser[4 * 1024];
        L2ScanResult l2ScanResult;
        char _reserved[2560];
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseMonitorBuffer) == 8192U);

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         アドバータイズの配信を管理するクラスです。
     */
    class AdvertiseDistributor
    {
    public:

        /**
         * @brief           コンストラクタです。
         */
        AdvertiseDistributor() NN_NOEXCEPT;

        /**
         * @brief           デストラクタです。
         */
        ~AdvertiseDistributor() NN_NOEXCEPT;

        /**
         * @brief           初期化に必要なバッファサイズを取得します。
         * @return          初期化に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        /**
         * @brief           AdvertiseDistributor を初期化します。
         * @param[in]       buffer          アドバータイズの配信に使用するバッファです。
         * @param[in]       bufferSize      buffer のバッファサイズです。
         * @param[in]       pInterface      配信に使用するネットワーク・インタフェースです。
         */
        void Initialize(
            void* buffer, size_t bufferSize, INetworkInterface* pInterface) NN_NOEXCEPT;

        /**
         * @brief           AdvertiseDistributor を解放します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief           アドバータイズデータの配信を開始します。
         * @param[in]       version         プロトコル・バージョンです。
         * @param[in]       networkId       ネットワーク識別子です。
         * @param[in]       security        セキュリティの設定です。
         * @param[in]       ldn             現在のネットワークの状態です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result StartDistribution(
            Version version, const NetworkId& networkId,
            SecurityMode security, const LdnNetworkInfo& ldn) NN_NOEXCEPT;

        /**
         * @brief           アドバータイズで配信するデータを更新します。
         * @param[in]       ldn             現在のネットワークの状態です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result Update(const LdnNetworkInfo& ldn) NN_NOEXCEPT;

        /**
         * @brief           アドバータイズの配信を停止します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result StopDistribution() NN_NOEXCEPT;

    private:

        Result SetAdvertiseData(const LdnNetworkInfo& ldn) NN_NOEXCEPT;

        impl::AdvertiseDistributorBuffer* m_Buffer;
        IAdvertiseBuilder* m_pBuilder;
        INetworkInterface* m_pNetworkInterface;
    };

    /**
     * @brief         アドバータイズをスキャンするクラスです。
     */
    class AdvertiseScanner
    {
    public:

        /**
         * @brief           コンストラクタです。
         */
        AdvertiseScanner() NN_NOEXCEPT;

        /**
         * @brief           デストラクタです。
         */
        ~AdvertiseScanner() NN_NOEXCEPT;

        /**
         * @brief           初期化に必要なバッファサイズを取得します。
         * @param[in]       scanResultCount スキャン結果の最大数です。
         * @return          初期化に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize(int scanResultCount) NN_NOEXCEPT;

        /**
         * @brief           AdvertiseScanner を初期化します。
         * @param[in]       buffer          アドバータイズの配信に使用するバッファです。
         * @param[in]       bufferSize      buffer のバッファサイズです。
         * @param[in]       scanResultCount スキャン結果の最大数です。
         * @param[in]       pInterface      スキャンに使用するネットワーク・インタフェースです。
         */
        void Initialize(
            void* buffer, size_t bufferSize, int scanResultCount,
            INetworkInterface* pInterface) NN_NOEXCEPT;

        /**
         * @brief           AdvertiseScanner を解放します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief           周辺のネットワークをスキャンします。
         */
        Result Scan(
            NetworkInfo* pOutScanResultArray,
            int* pOutCount, int bufferCount,
            const ScanFilter& filter, int channel) NN_NOEXCEPT;

    private:

        impl::AdvertiseScannerBuffer* m_Buffer;
        INetworkInterface* m_pNetworkInterface;
        int m_ScanResultCountMax;
    };

    /**
     * @brief         アドバータイズを監視するクラスです。
     */
    struct AdvertiseMonitor
    {
    public:

        /**
         * @brief       スコープ内限定でアドバータイズの監視を有効化します。
         */
        class ScopedMonitor
        {
        public:

            ScopedMonitor(
                AdvertiseMonitor* pMonitor, int* pOutAid, LdnNetworkInfo* pOutInfo,
                const NetworkId& networkId, SecurityMode security) NN_NOEXCEPT
                : m_pMonitor(pMonitor)
            {
                m_Result = m_pMonitor->StartMonitoring(pOutAid, pOutInfo, networkId, security);
            };

            ~ScopedMonitor() NN_NOEXCEPT
            {
                if (m_Result.IsSuccess() && m_pMonitor != nullptr)
                {
                    m_pMonitor->StopMonitoring();
                }
            }

            Result GetResult() const NN_NOEXCEPT
            {
                return m_Result;
            }

            void Detach() NN_NOEXCEPT
            {
                m_pMonitor = nullptr;
            }

        private:

            AdvertiseMonitor* m_pMonitor;
            Result m_Result;
        };

        /**
         * @brief           コンストラクタです。
         */
        AdvertiseMonitor() NN_NOEXCEPT;

        /**
         * @brief           デストラクタです。
         */
        ~AdvertiseMonitor() NN_NOEXCEPT;

        /**
         * @brief           初期化に必要なバッファサイズを取得します。
         * @return          初期化に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        /**
         * @brief           AdvertiseMonitor を初期化します。
         * @param[in]       buffer          アドバータイズの配信に使用するバッファです。
         * @param[in]       bufferSize      buffer のバッファサイズです。
         * @param[in]       pInterface      スキャンに使用するネットワーク・インタフェースです。
         */
        void Initialize(
            void* buffer, size_t bufferSize, INetworkInterface* pInterface) NN_NOEXCEPT;

        /**
         * @brief           AdvertiseScanner を解放します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief           アドバータイズの監視を開始します。
         * @param[out]      pOutAid         アクセスポイントから割り当てられた AID です。
         * @param[out]      pOutInfo        最初に受信したアドバータイズ・データの出力先です。
         * @param[in]       networkId       対象のネットワークの識別子です。
         * @param[in]       security        セキュリティ・モードです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultConnectionTimeout}
         * @endretresult
         */
        Result StartMonitoring(
            int* pOutAid, LdnNetworkInfo* pOutInfo,
            const NetworkId& networkId, SecurityMode security) NN_NOEXCEPT;

        /**
         * @brief           アドバータイズの監視を終了します。
         */
        void StopMonitoring() NN_NOEXCEPT;

        /**
         * @brief           アドバータイズの受信時にシグナルされるイベントを取得します。
         * @return          アドバータイズの受信時にシグナルされるイベントです。
         */
        nn::os::SystemEvent& GetReceivedEvent() NN_NOEXCEPT;

        /**
         * @brief           最後に受信したアドバータイズ・データを取得します。
         * @param[out]      pOutInfo        アドバータイズ・データの出力先です。
         * @return          前回取得したアドバータイズから差分が有れば true です。
         */
        bool GetAdvertise(LdnNetworkInfo* pOutInfo) NN_NOEXCEPT;

    private:

        impl::AdvertiseMonitorBuffer* m_Buffer;
        INetworkInterface* m_pNetworkInterface;
        IAdvertiseParser* m_pParser;
        MacAddress m_MacAddress;
        NetworkId m_NetworkId;
    };

}}} // end of namespace nn::ldn::detail
