﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief IR LED プロセッサに関する API の宣言。
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_IrLedProcessorTypes.h>

namespace nn { namespace irsensor {

/**
 * @name IR LED プロセッサ
 * @{
 */

/**
 * @brief      IR LED プロセッサのデフォルト設定を取得します。
 * @param[out] pOutValue IR LED プロセッサのデフォルト設定
 * @pre
 *             - pOutValue != nullptr
 */
void GetIrLedProcessorDefaultConfig(IrLedProcessorConfig* pOutValue) NN_NOEXCEPT;

/**
 * @brief      IR カメラと IR LED プロセッサの処理を開始します。
 *             すでにプロセッサが起動中の場合は、実行中のプロセッサ処理を停止し、IR LED プロセッサの処理を再実行します。
 * @param[in]  handle IR カメラのハンドル
 * @param[in]  config IR LED プロセッサの設定
 * @pre
 *             - 指定のハンドルで IR カメラが初期化済み
 * @post
 *             - 指定のハンドルに対応する IR カメラが IR LED プロセッサを利用して処理実行状態
 * @details    指定のハンドルに対応する IR カメラが停止状態の場合は、IR カメラと IR LED プロセッサに設定を反映し
 *             モーションIRカメラを処理実行状態にします。
 *
 *             すでに指定のハンドルに対応する IR カメラが処理実行状態の場合は、既存のプロセッサを停止して、
 *             IR カメラと IR LED プロセッサに設定を反映しモーションIRカメラを再度処理実行状態にします。
 *             既存の処理実行状態のプロセッサの種類は問いません。
 *             処理実行状態の間は IR カメラの撮影は行わず、IrLedProcessorConfig で指定された内容に応じた
 *             IR LED の状態が維持されます。IR LED の状態を変更したい場合には、上記 config の内容を変更した上で
 *             再度 RunIrLedProcessor を実行する必要があります。
 */
void RunIrLedProcessor(const IrCameraHandle& handle, const IrLedProcessorConfig& config) NN_NOEXCEPT;

/**
 * @brief      IR LED プロセッサの最新の処理結果を取得します。
 * @param[in]  handle    IR カメラのハンドル
 * @retresult
 *   @handleresult{nn::irsensor::ResultIrsensorNotReady}
 *   @handleresult{nn::irsensor::ResultIrsensorUnconnected}
 *   @handleresult{nn::irsensor::ResultIrsensorUnsupported}
 *   @handleresult{nn::irsensor::ResultIrsensorDeviceError}
 * @endretresult
 * @pre
 *             - 指定のハンドルで IR カメラが初期化済み
 *             - 指定のハンドルに対応する IR カメラが IR LEDプロセッサを利用して処理実行状態
 *             - pOutValue != nullptr
 *
 * @details    この関数は IR LED プロセッサが正しく動作している間は、常に ::nn::ResultSuccess を返します。
 *             処理を開始した直後で処理結果がまだない場合や IR カメラが利用可能でない場合には、この関数は ::nn::ResultSuccess 以外を返します。
 *
 *             モーションIRカメラの状態が異常である場合は、システムがそのエラー内容を画面に表示します。
 *             エラー表示を行っている間は、呼び出し元をブロックします。
 *             ユーザが操作した後に返り値として nn::irsensor::ResultIrsensorDeviceError を返し、呼び出し元に戻ります。
 *
 *             コントローラのファームウェアに更新が必要な場合は、コントローラサポートアプレットが起動し、アップデートが行われます。
 *             アップデートを行っている間は、呼び出し元をブロックします。
 *             アップデートに成功していた場合は nn::irsensor::ResultIrsensorNotReady を、失敗していた場合は
 *             nn::irsensor::ResultIrsensorDeviceError を返し、呼び出し元に戻ります。
 *
 *             上記のとおり、本関数は呼び出し元をブロックする可能性があるため、メインループとは別のスレッドから呼び出すことを推奨します。
 *
 *             少なくとも 1 台以上のコントローラで NFP 機能を使用している場合は IR センサーは利用できません。
 *             また、接続されているコントローラが 5 台以上の場合は IR センサーの複数台同時利用はできません。
 *             これらの使用条件を満たさない場合にこの関数を呼び出すと nn::irsensor::ResultIrsensorNotReady を返します。
 */
::nn::Result GetIrLedProcessorState(const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 * @}
 */

}} // namespace nn::irsensor
