﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API declaration for HeartRate Processor
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_HeartRateProcessorTypes.h>

namespace nn { namespace irsensor {

/**
 * @name HeartRate Processor
 * @{
 */

/**
 *  @brief                Start HeartRateProcessor mode of IR sensor
 *  @param[in] handle     Handle of IR camera
 *  @param[in] config     Configuration of HeartRateProcessor mode
 *  @param[in] workBuffer Workbuffer used in HeartRateProcessor
 *  @return               Return processed result
 *  @pre
 *                      - Camera handle is already initialized via Initialize API.
 *                      - For given handle, IR camera is not running yet.
 *                      - workBuffer pointer != nullptr
 *                      - specific alignment is not necessary for workBuffer.
 *  @details            User sets a configuration to HeartRateProcessor and IR sensor starts processing.
 *                      User keeps getting data via Get**() API while HeartRateProcessor is running.
 *                      When IR sensor is already running another mode, it stops running mode and starts HeartRate mode with given configuration.
 *
 *                      User workbuffer is unlocked and could be used for different usage
 *                      after "calling StopImageProcessor(), StopImageProcessorAsync() APIs" or
 *                      "calling Run*** ( other than RunHeartRateProcessor() ) APIs" or
 *                      "calling RunHeartRateProcessor() with same camera handle and different workbuffer".
 */
void RunHeartRateProcessor(const IrCameraHandle& handle, const HeartRateProcessorConfig& config, HeartRateProcessorWorkBuffer& workBuffer) NN_NOEXCEPT;

/**
 *  @brief                           Retrieve latest processed data by HeartRateProcessor
 *
 *  @param[out] pOutValue            User allocated area that will receive the processed data
 *  @param[in]  handle               Handle of IR sensor
 *
 * @details                          This function retrieves HeartRateProcessor states calcuted by IR sensor data.
 *                                   The function will return available  state.
 *
 *                                   Compared with getting other irsensor state like in Moment or Clustering mode,
 *                                   getting HeartRate state will not only copying value coming from irsensor
 *                                   but also calculating it which is not negligible.
 *
 *                                   When IR sensor is in trouble and unable to use, internal system displays the error.
 *                                   While displaying, this API blocks the thread which calls it.
 *                                   After user handles the error display, it returns nn::irsensor::ResultIrsensorDeviceError .
 *
 *                                   When the controller firmware is needed to update,
 *                                   internal system calls controller support applet and updates it.
 *                                   While updating, this API blocks the thread which calls it.
 *                                   If it succeeds it returns nn::irsensor::ResultIrsensorNotReady ,
 *                                   and if it fails it returns nn::irsensor::ResultIrsensorDeviceError .
 *
 *                                   IR sensor is not available when application uses NFP functions on at least one controller.
 *                                   The second IR sensor is not available only when the total number of connected controller is larger than 4.
 *                                   If these constraints are not satisfied, this API returns nn::irsensor::ResultIrsensorNotReady.
 *
 *  @retresult
 *   @handleresult{nn::irsensor::ResultIrsensorNotReady}
 *   @handleresult{nn::irsensor::ResultIrsensorUnavailable}
 *   @handleresult{nn::irsensor::ResultIrsensorDeviceError}
 *  @endretresult
 *  @pre
 *                         - Handle is already initialized via Initialize API.
 *                         - For given handle, IR camera is running as HeartRateProcessor.
 *                         - pOutValue != nullptr
 */
::nn::Result GetHeartRateProcessorState(HeartRateProcessorState* pOutValue, const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 *  @brief                           Reset internal buffers used for heartrate calculations
 *
 *  @param[in]  handle               Handle of IR sensor
 *
 * @details                          This function reset internal buffers used for heartrate calculations.
 *                                   When user wants to restart heartrate measurement, please call this api.
 *  @pre
 *                                   - HeartRateProcessor must be running before calling this reset API.
*/
void ResetHeartRateProcessorState(const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 * @}
 */

} } // namespace nn::irsensor
