﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       物理的な対応関係が保証されたコントローラー(UniquePad)を制御する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       物理的な対応関係が保証されたコントローラー(UniquePad)の ID を示す構造体です。
 */
struct UniquePadId
{
    int64_t _storage;
};

const int UniquePadIdCountMax = 16;  //!< ListUniquePadIds() が一度に返す UniquePadId の最大数

/**
 * @brief       コントローラーの種類を表す列挙体です。
 */
enum UniquePadType
{
    UniquePadType_Unknown,                 //!< 未知のコントローラーもしくは未接続
    UniquePadType_FullKeyController,       //!< Switch Proコントローラー
    UniquePadType_RightController,         //!< Joy-Con (R)
    UniquePadType_LeftController,          //!< Joy-Con (L)
    UniquePadType_Embedded,                //!< 本体に付属の(搭載された)コントローラー
};

/**
 * @brief       コントローラーの接続インタフェースを表す列挙体です。
 */
enum UniquePadInterface
{
    UniquePadInterface_Unknown,            //!< 未知のインタフェースもしくは未接続
    UniquePadInterface_Rail,               //!< 本体のレール部にジョイントされて接続されています。
    UniquePadInterface_Bluetooth,          //!< Bluetooth 接続されています。
    UniquePadInterface_Usb,                //!< USB 接続されています。
    UniquePadInterface_Embedded,           //!< 本体に搭載されています。
};

//!< コントローラーのシリアルナンバー長
const size_t UniquePadSerialNumberLength = 16;

/**
 * @brief       コントローラーのシリアルナンバーを表す構造体です
 */
struct UniquePadSerialNumber
{
    char _storage[UniquePadSerialNumberLength];
};

//! @name UniquePad 関連 API
//! @{

/**
 * @brief       コントローラーの接続状態変化の通知を指定のイベントオブジェクトにバインドします。
 *
 * @details     コントローラーの接続状態変化の通知を指定のイベントオブジェクトにバインドします。
 *              同時に複数のイベントオブジェクトに対してバインドすることはできません。
 *              使用済みになったシステムイベントは、 nn::os::DestroySystemEvent() で破棄して下さい。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindUniquePadConnectionEvent(::nn::os::SystemEventType* pEvent,
                                  ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       コントローラーを列挙します。
 *
 * @details     接続されているコントローラーを列挙します。
 *              Xpad では、単一の Xpad と物理的なコントローラー(デバイス)が1対1で対応づけられていることを保障しません。
 *              本体設定などで物理的なコントローラーをユニークに明示して状態を取得したり制御することが必要な場合に、本インタフェースを利用します。
 *
 *              接続されているコントローラーには、コントローラーの種類によらず1つの UniquePadId が割り当てられます。
 *              コントローラーの種類を取得するためには、GetUniquePadType() を使用してください。
 *
 *              取得できる UniquePadId の上限は UniquePadIdCountMax 個です。
 *
 * @param[out]  pOutValues                   UniquePad を読み出すバッファ（配列）
 * @param[in]   count                       読み出す UniquePad の数
 *
 * @return      読み出した UniquePadId の数です。
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count > 0
 *              - count <= UniquePadIdCountMax
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 戻り値 n について、n <= UniquePadIdCountMax
 */
int ListUniquePads(UniquePadId* pOutValues, int count) NN_NOEXCEPT;

/**
 * @brief       コントローラーの種類を取得します。
 *
 * @details     任意の UniquePad についてコントローラーの種類を取得します。
 *
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @return      コントローラーの種類
 *
 */
UniquePadType GetUniquePadType(UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーの接続インタフェースを取得します。
 *
 * @details     コントローラーがどのようにして接続されているかを取得します。
 *
 * @param[out]  pOutValue                   コントローラーの接続インタフェース
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,  コントローラーが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetUniquePadInterface(UniquePadInterface* pOutValue, UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーのコントローラー番号を取得します。
 *
 * @details     コントローラーに対して割り当てられているコントローラー番号を取得します。
 *
 *              本体コントローラー(携帯機持ち)の場合など、コントローラー番号が割り当てられていないケースも存在します。
 *              その場合は返り値にResultUniquePadNotAvailableが返ります。
 *              本体との接続インタフェースは常に1つです。
 *
 * @param[out]  pOutValue                   コントローラー番号
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,  コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultUniquePadNotAvailable,  コントローラー番号が割り当てられていません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetUniquePadControllerNumber(int* pOutValue, UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーのシリアル番号を取得します。
 *
 * @details     コントローラーに対して割り当てられているシリアル番号を取得します。
 *
 * @param[out]  pOutValue                   シリアル番号
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,  コントローラーが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetUniquePadSerialNumber(UniquePadSerialNumber*pOutValue, UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーの Bluetooth Address を取得します。
 *
 * @details     コントローラーに対して割り当てられている Bluetooth Address を取得します。
 *              Bluetooth 接続型でないデバイスなど Bluetooth Address が割り当てられていない場合は、ResultUniquePadNoBluetoothAddress が返ります。
 *
 * @param[out]  pOutValue                   Bluetooth Address
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,  コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultUniquePadNoBluetoothAddress,  コントローラー Bluetooth Address がありません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetUniquePadBluetoothAddress(nn::bluetooth::Address *pOutValue, UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーを切断します。
 *
 * @details     無線接続状態にあるコントローラーを切断します。
 *              Bluetooth 接続型でない場合は、ResultUniquePadNoWirelessConnection が返り切断に失敗します。
 *
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,  コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultUniquePadNoWirelessConnection,  コントローラーが無線接続されていないため、切断できません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result DisconnectUniquePad(UniquePadId id) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
