﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6軸センサーのキャリブレーション処理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       物理的な対応関係が保証された 6軸センサーのハンドルを表す構造体です。
 */
struct UniqueSixAxisSensorHandle
{
    int32_t _storage;
};

const int UniqueSixAxisSensorHandleCountMax = 16;  //!< ListSixAxisSensorHandles() が一度に返す UniqueSixAxisSensorHandle の最大数

/**
 * @brief       6軸センサーのユーザーキャリブレーションシーケンスの経過を示す定義です。
 */
enum SixAxisSensorUserCalibrationStage
{
    SixAxisSensorUserCalibrationStage_Measuring,             //!< キャリブレーション値の測定中
    SixAxisSensorUserCalibrationStage_Update,                //!< キャリブレーション値書き込み中
    SixAxisSensorUserCalibrationStage_Completed,             //!< キャリブレーションシーケンス完了
};

//! @name 6軸センサーのキャリブレーション関連 API
//! @{

/**
 * @brief       UniqueSixAxisSensorHandle を列挙します。
 *
 * @details     UniquePadId が割り当てられているコントローラーに搭載された6軸センサーの UniqueSixAxisSensorHandle を列挙します。
 *              物理的な6軸センサーをユニークに明示して状態を取得したり制御することが必要な場合に、本インタフェースを利用します。
 *
 *              取得できる UniqueSixAxisSensorHandle の上限は UniqueSixAxisSensorHandleCountMax 個です。
 *
 * @param[out]  pOutValues                  UniqueSixAxisSensorHandle を読み出すバッファ（配列）
 * @param[in]   id                          読み出すコントローラーの UniquePadId
 * @param[in]   count                       読み出す UniqueSixAxisSensorHandle の数
 *
 * @return      読み出した UniquePadId の数です。
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count > 0
 *              - count <= UniqueSixAxisSensorHandleCountMax
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 戻り値 n について、n <= UniqueSixAxisSensorHandleCountMax
 */
int ListSixAxisSensorHandles(UniqueSixAxisSensorHandle* pOutValues, const UniquePadId& id, int count) NN_NOEXCEPT;

/**
 * @brief       6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
 *
 * @details     指定した UniqueSixAxisSensorHandle に該当する 6軸センサーに対して、
 *              ユーザーキャリブレーション機能がサポートされているかどうかを返します。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @return      ユーザーキャリブレーション機能がサポートされている場合は true, そうでない場合は false
 *
 */
bool IsSixAxisSensorUserCalibrationSupported(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       工程出荷時の 6 軸センサーのキャリブレーション値を書き込みます。
 *
 * @details     UniqueSixAxisSensorHandle を指定して工程出荷時に書き込まれたキャリブレーション値を、
 *              ユーザーキャリブレーション領域に書き込みます。
 *              失敗した場合はユーザーキャリブレーション領域は呼出し前の状態のまま書き換わりません。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorDisconnected,  指定の 6 軸センサーを搭載したコントローラーが接続状態にありません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotSupported,  指定の 6 軸センサーは工程出荷状態への書き戻し処理に対応していません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorWriteFailure,  書き込み処理に失敗しました。}
 * @endretresult
 *
 */
Result ResetSixAxisSensorCalibrationValues(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       6 軸センサーのユーザーキャリブレーション処理を開始します。
 *
 * @details     指定した UniqueSixAxisSensorHandle に該当する 6 軸センサーに対してユーザーキャリブレーション処理を開始します。
 *              6 軸センサーのユーザーキャリブレーション処理は複数同時に行うことはできません。
 *              進行中のユーザーキャリブレーション処理がある場合は、 CancelSixAxisSensorUserCalibration() で終了する必要があります。
 *
 *              成功した場合はユーザーキャリブレーション領域に結果が書き込まれます。
 *              失敗した場合はユーザーキャリブレーション領域は呼出し前の状態のまま書き換わりません。
 *              対象の 6 軸センサーの状態に依らず、ユーザーキャリブレーション処理を行うことが可能です。
 *
 * @param[in]   handle                  6軸センサーのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorDisconnected,  指定の 6 軸センサーを搭載したコントローラーが接続状態にありません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotSupported,  指定の 6 軸センサーはユーザーキャリブレーション処理に対応していません。}
 * @endretresult
 *
 */
Result StartSixAxisSensorUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       6 軸センサーのユーザーキャリブレーション処理の経過状況を取得します。
 *
 * @details     6 軸センサーのユーザーキャリブレーション処理は以下のステージに分かれて順に実行されます。
 *
 *              1. SixAxisSensorUserCalibrationStage_Measuring
 *                 ユーザーキャリブレーション処理のための測定を行っています。
 *              2. SixAxisSensorUserCalibrationStage_Update
 *                 測定した値を元に計算したキャリブレーション値をコントローラーに書き込んでいます。
 *              3. SixAxisSensorUserCalibrationStage_Completed
 *                 ユーザーキャリブレーション処理が完了しました。
 *
 *              6 軸センサーのユーザーキャリブレーション処理が問題なく進行している間は、nn::ResultSuccess が返されます。
 *              キャリブレーション中に何か問題があった場合は、失敗の理由が返されます。
 *              キャリブレーション処理が完了すると SixAxisSensorUserCalibrationStage_Completed が取得されます。
 *
 * @param[out]  pOutValue               6 軸センサーのユーザーキャリブレーション処理の経過状況
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                 処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorDisconnected,  指定の 6 軸センサーを搭載したコントローラーが接続状態にありません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotSupported,  指定の 6 軸センサーはユーザーキャリブレーション処理に対応していません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotStable,     6 軸センサーの静止判定に失敗しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotHorizontal, 6 軸センサーの水平判定に失敗しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorWriteFailure,  書き込み処理に失敗しました。}
 * @endretresult
 *
 */
nn::Result GetSixAxisSensorUserCalibrationStage(SixAxisSensorUserCalibrationStage* pOutValue,
                                                const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       進行中の 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
 *
 * @details     指定した UniqueSixAxisSensorHandle に該当する 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
 *              キャリブレーション処理が　SixAxisSensorUserCalibrationStage_Update　のときは、
 *              コントローラーに保存されているキャリブレーション値を更新中のためキャンセルができません。
 *
 *              StartSixAxisSensorUserCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は本関数は何も作用しません。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 */
void CancelSixAxisSensorUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
