﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   撮影ボタンに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

const int CaptureButtonStateCountMax = 16;  //!< 内部的に保持される撮影ボタンの入力状態の最大数

/**
 * @brief       撮影ボタンの定義です。
 */
struct CaptureButton
{
    typedef ::nn::util::BitFlagSet<64, CaptureButton>::Flag<0> Active;  //!< アクティブな撮影ボタン
};

/**
 * @brief       撮影ボタンの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<64, CaptureButton> CaptureButtonSet;

/**
 * @brief       撮影ボタンの入力状態を表す構造体です。
 */
struct CaptureButtonState
{
    int64_t samplingNumber;     //!< 撮影ボタンの入力状態が更新される度に増加する値です。
    CaptureButtonSet buttons;   //!< 撮影ボタンの状態です。
};

//! @name 撮影ボタン 関連 API
//! @{

/**
 * @brief       撮影ボタンの入力通知をバインドします。
 *
 * @details     撮影ボタンの入力通知を指定のイベントオブジェクトにバインドします。
 *              入力通知は撮影ボタンの押下と開放の両方についてシグナルされます。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindCaptureButtonEvent(::nn::os::SystemEventType* pEvent,
                            ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       撮影ボタンを初期化します。
 */
void InitializeCaptureButton() NN_NOEXCEPT;

/**
 * @brief       撮影ボタンの入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は CaptureButtonStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は CaptureButtonState::samplingNumber を参照してください。
 *              撮影ボタンの入力状態は 15 ms 毎にその時点における最新の標本値に基づいて更新されます。
 *              "アクティブな撮影ボタン"が存在しない状態において新規に押下された撮影ボタンは、
 *              開放されるまでの間、 "アクティブな撮影ボタン"と見做され、 その状態の取得が可能です。
 *              アクティブな撮影ボタンが存在しない場合、 アクティブな撮影ボタンの状態は開放となります。
 *
 * @param[out]  outStates                   入力状態を読み出すバッファ（配列）
 * @param[in]   count                       outStates の数
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - InitializeCaptureButton() の呼び出しが完了している
 *              - outStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetCaptureButtonStates(CaptureButtonState* outStates, int count) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
