﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   アプレット管理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/system/hid_Common.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       アプレットリソースユーザ ID を設定します。
 *
 * @details     hid サービスの提供先をアプレットリソースユーザ ID で指定します。
 *              有効な AppletResourceUserId が指定されていなければ、
 *              hid サービスの提供先の制限は無効となります。
 *
 * @param[in]   value                       アプレットリソースユーザ ID
 */
void SetAppletResourceUserId(const ::nn::applet::AppletResourceUserId& value
                             ) NN_NOEXCEPT;

/**
 * @brief       アプレットリソースユーザ ID を登録します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象を hid サービスの
 *              提供先として登録します。
 *
 * @param[in]   aruid                       アプレットリソースユーザ ID
 * @param[in]   enablesInput                HID 入力の提供を行うか否か
 *
 * @pre
 *              - aruid != nn::applet::AppletResourceUserId::GetInvalidId()
 */
void RegisterAppletResourceUserId(
    const ::nn::applet::AppletResourceUserId& aruid,
    bool enablesInput) NN_NOEXCEPT;

/**
 * @brief       アプレットリソースユーザ ID の登録を解除します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象を hid サービスの
 *              提供先から削除します。
 *
 * @param[in]   aruid                       アプレットリソースユーザ ID
 *
 * @pre
 *              - aruid != nn::applet::AppletResourceUserId::GetInvalidId()
 */
void UnregisterAppletResourceUserId(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

/**
 * @brief       HID 入力の提供を行うか否かを設定します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象に HID 入力の提供を
 *              行うか否かを設定します。
 *
 * @param[in]   aruid                       アプレットリソースユーザ ID
 * @param[in]   enablesInput                HID 入力の提供を行うか否か
 *
 * @pre
 *              - aruid != nn::applet::AppletResourceUserId::GetInvalidId()
 */
void EnableAppletToGetInput(const ::nn::applet::AppletResourceUserId& aruid,
                            bool enablesInput) NN_NOEXCEPT;

/**
 * @brief       6 軸センサー入力の提供を行うか否かを設定します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象に 6 軸センサー入力の提供を
 *              行うか否かを設定します。
 *
 * @param[in]   aruid                       アプレットリソースユーザ ID
 * @param[in]   enablesInput                6 軸センサー入力の提供を行うか否か
 *
 * @pre
 *              - aruid != nn::applet::AppletResourceUserId::GetInvalidId()
 */
void EnableAppletToGetSixAxisSensor(const ::nn::applet::AppletResourceUserId& aruid,
                                    bool enablesInput) NN_NOEXCEPT;

/**
* @brief       振動子の制御権の提供を行うか否かを設定します。
*
* @details     アプレットリソースユーザ ID で指定された対象に 振動子の制御権の提供を
*              行うか否かを設定します。
*
* @param[in]   aruid                       アプレットリソースユーザ ID
* @param[in]   enable                      振動子の制御権の提供を行うか否か
*
* @pre
*              - aruid != nn::applet::AppletResourceUserId::GetInvalidId()
*/
void SetAruidValidForVibration(
    const ::nn::applet::AppletResourceUserId& aruid, bool enable) NN_NOEXCEPT;


}}} // namespace nn::hid::system
