﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Xpad のエミュレーション設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/settings_Xpad.h>

namespace nn { namespace hid {

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てを取得します。
 *
 * @details     ::nn::settings::GetXpadGenericPadMap() の内部実装です。
 *
 * @param[out]  outValue                    Xpad に対する汎用ゲームパッドの割り当てを読み出すバッファ
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 * @retval      ResultBasicXpadNullGenericPadMapBuffer  BasicXpad に対する汎用ゲームパッドの割り当ての読み出し先が NULL 値です。
 *
 * @pre
 *              - outValue != nullptr
 */
::nn::Result GetXpadGenericPadMap(
    ::nn::settings::BasicXpadGenericPadMap* outValue,
    int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てを設定します。
 *
 * @details     ::nn::settings::SetXpadGenericPadMap() の内部実装です。
 *
 * @param[in]   value                       Xpad に対する汎用ゲームパッドの割り当て
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 */
::nn::Result SetXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& value,
    int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てをリセットします。
 *
 * @details     ::nn::settings::ResetXpadGenericPadMap() の内部実装です。
 *
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 */
::nn::Result ResetXpadGenericPadMap(int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てを取得します。
 *
 * @details     ::nn::settings::GetXpadKeyboardMap() の内部実装です。
 *
 * @param[out]  outValue                    Xpad に対するキーボードの割り当てを読み出すバッファ
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 * @retval      ResultBasicXpadNullKeyboardMapBuffer    BasicXpad に対するキーボードの割り当ての読み出し先が NULL 値です。
 *
 * @pre
 *              - outValue != nullptr
 */
::nn::Result GetXpadKeyboardMap(::nn::settings::BasicXpadKeyboardMap* outValue,
                                int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てを設定します。
 *
 * @details     ::nn::settings::SetXpadKeyboardMap() の内部実装です。
 *
 * @param[in]   value                       Xpad に対するキーボードの割り当て
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 */
::nn::Result SetXpadKeyboardMap(
    const ::nn::settings::BasicXpadKeyboardMap& value,
    int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てをリセットします。
 *
 * @details     ::nn::settings::ResetXpadKeyboardMap() の内部実装です。
 *
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                           成功しました。
 */
::nn::Result ResetXpadKeyboardMap(int playerNumber) NN_NOEXCEPT;

}} // namespace nn::hid
