﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Palma の接続にかかわる機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>

namespace nn { namespace hid {

/**
 * @brief       ペアリング済み Palma の接続待ち受けを行います。
 *
 * @return      ペアリング済みの Palma のみの接続を待ち受けます。
 *              未ペアリングのものも含めた接続の待ち受けを行う場合は、 EnableAnyPalmaConnection() を使用してください。
 *              接続待ち受けを終了する場合は、DisablePairedPalmaConnection() を使用します。
 *              アプリケーションがインフォーカス状態でなくなった場合は、接続待ち受けが一時的に停止し、フォーカス状態になった際に再開します。
 */
void EnablePairedPalmaConnection() NN_NOEXCEPT;

/**
 * @brief       ペアリング済み Palma の接続待ち受け処理を停止します。
 *
 * @return      EnablePairedPalmaConnection() で開始した接続待ち受け処理を終了します。
 */
void DisablePairedPalmaConnection() NN_NOEXCEPT;

/**
 * @brief       全ての Palma の接続待ち受けを行います。
 *
 * @return      ペアリングの有無にかかわらず全ての Palma の接続を待ち受けます。
 *              接続待ち受けを終了する場合は、DisableAnyPalmaConnection() を使用します。
 *              アプリケーションがインフォーカス状態でなくなった場合は、接続待ち受けが一時的に停止し、フォーカス状態になった際に再開します。
 *              接続された Palma とペアリングをする場合は、PairPalma() を接続してください。
 */
void EnableAnyPalmaConnection() NN_NOEXCEPT;

/**
 * @brief       EnableAnyPalmaConnection() で開始した接続待ち受け処理を停止します。
 *
 * @return      EnablePairedPalmaConnection() でペアリング済みの Palma の接続待ち受け処理が有効化されている場合は、未ペアリングのもののみ接続できなくなります。
 */
void DisableAnyPalmaConnection() NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
