﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       GamePad の状態設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_BatteryLevel.h>
#include <nn/hid/hid_GamePad.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace hid {

/**
 * @brief       GamePad に設定可能な状態の定義です。
 */
struct GamePadAutoPilotState
{
    int32_t powerState;             //!< GamePad の電源状態です。
    int32_t batteryLevel;           //!< GamePad のバッテリー残量です。
    GamePadButtonSet buttons;       //!< GamePad のデジタルボタンの状態です。
    AnalogStickState analogStickL;  //!< GamePad の左アナログスティックの状態です。
    AnalogStickState analogStickR;  //!< GamePad の右アナログスティックの状態です。
};

/**
 * @brief       GamePad のサンプリング通知をバインドします。
 *
 * @details     指定の GamePad のサンプリング通知を指定のイベントオブジェクトにバインドします。
 *              同時に複数のイベントオブジェクトに対してバインドすることはできません。
 *
 * @param[in]   index                       GamePad のインデックス
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - 0 <= index && index < GamePadCountMax
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 */
void BindGamePadSamplingEvent(int index,
                              ::nn::os::SystemEventType* pEvent,
                              ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       GamePad のサンプリング通知をアンバインドします。
 *
 * @details     GamePad のサンプリング通知を指定のイベントオブジェクトからアンバインドします。
 *              通知がバインドされていないイベントオブジェクトが指定された場合は何も行いません。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は初期化状態
 *              - *pEvent を待機しているスレッドはいない
 * @post
 *              - *pEvent は未初期化状態
 */
void UnbindGamePadSamplingEvent(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       GamePad の状態を設定します。
 *
 * @details     指定の GamePad に対して指定の状態を設定します。
 *              設定された状態は以降のサンプリング結果に対して反映されます。
 *              一度行われた設定は、再度設定が行われるか UnsetGamePadAutoPilotState() で解除されるまで反映され続けます。
 *
 * @param[in]   index                       GamePad のインデックス
 * @param[in]   state                       GamePad の状態
 *
 * @pre
 *              - 0 <= index && index < GamePadCountMax
 */
void SetGamePadAutoPilotState(int index,
                              const GamePadAutoPilotState& state) NN_NOEXCEPT;

/**
 * @brief       GamePad の状態設定を解除します。
 *
 * @details     指定の GamePad の状態設定を解除します。
 *              設定の解除は以降のサンプリング結果に対して反映されます。
 *              設定が行われていない状態で呼び出された場合は何も起こりません。
 *
 * @param[in]   index                       GamePad のインデックス
 *
 * @pre
 *              - 0 <= index && index < GamePadCountMax
 */
void UnsetGamePadAutoPilotState(int index) NN_NOEXCEPT;

}} // namespace nn::hid
