﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ens {

/**
 * @brief   シンプルな MessagePack ライター
 *
 * @details
 *  表現できるデータ構造を少なくする代わりに、インターフェイスがシンプルな MessagePack のライターです。
 *
 *  最初に呼び出す関数は以下のいずれかです。
 *
 *  - WriteValue
 *  - WriteArray
 *  - WriteMap
 *
 *  キーや文字列値は UTF-8 でエンコードする必要があります。@n
 *  また、最大長は INT_MAX に制限されます。
 *
 *  以下に MessagePack の記述例を示します。
 *
 *  記述例１：文字列単体を MessagePack 化する
 *
 *  - WriteValue("string")
 *
 *  記述例２：数値の配列を MessagePack 化する
 *
 *  - WriteArray(3)
 *      - WriteValue(1)
 *      - WriteValue(2)
 *      - WriteValue(3)
 *
 *  記述例３：構造体 "struct {int a; int b;}" を MessagePack 化する
 *
 *  - WriteMap(2)
 *      - WriteKeyValue("a", 100)
 *      - WriteKeyValue("b", 200)
 *
 *  記述例４：配列を含む複雑な構造体 "struct {struct {int x; int y;} points[3]; int color;}" を MessagePack 化する
 *
 *  - WriteMap(2)
 *      - WriteKeyArray("points", 3)
 *          - WriteMap(2)
 *              - WriteKeyValue("x", 1)
 *              - WriteKeyValue("y", 1)
 *          - WriteMap(2)
 *              - WriteKeyValue("x", 2)
 *              - WriteKeyValue("y", 2)
 *          - WriteMap(2)
 *              - WriteKeyValue("x", 3)
 *              - WriteKeyValue("y", 3)
 *      - WriteKeyValue("color", 0)
 */
class SimpleMessagePackWriter
{
private:
    NN_DISALLOW_COPY(SimpleMessagePackWriter);
    NN_DISALLOW_MOVE(SimpleMessagePackWriter);

public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pBuffer   バッファ
     * @param[in]   size      バッファサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    SimpleMessagePackWriter(void* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   連想配列要素を書き込みます。
     *
     * @param[in]   num 要素数
     *
     * @return  書き込めたかどうか
     *
     * @pre
     *  - num <= UINT32_MAX
     *
     * @details
     *  本関数はルート要素を記述するために使用します。@n
     *  本関数を呼び出した後、要素の数だけキーと値を書き込む必要があります。（以下の関数のいずれかを使用）
     *
     *  - WriteKeyValue
     *  - WriteKeyArray
     *  - WriteKeyMap
     */
    bool WriteMap(size_t num) NN_NOEXCEPT;

    /**
     * @brief   配列要素を書き込みます。
     *
     * @param[in]   num 要素数
     *
     * @return  書き込めたかどうか
     *
     * @pre
     *  - num <= UINT32_MAX
     *
     * @details
     *  本関数はルート要素を記述するために使用します。@n
     *  本関数を呼び出した後、要素の数だけ値を書き込む必要があります。（以下の関数のいずれかを使用）
     *
     *  - WriteValue
     *  - WriteArray
     *  - WriteMap
     */
    bool WriteArray(size_t num) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（論理値）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(bool value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(int8_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(int16_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(int32_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(int64_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(uint8_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(uint16_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(uint32_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(uint64_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（浮動小数点数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(double value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（文字列）
     *
     * @param[in]   pValue  値
     *
     * @pre
     *  - pValue != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(const char* pValue) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（バイナリ）
     *
     * @param[in]   pValue  値
     * @param[in]   size    pValue のサイズ
     *
     * @pre
     *  - pValue != nullptr
     *  - size <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     *  本関数は、配列の値を書き込む時に使用します。
     */
    bool WriteValue(const void* pValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（連想配列要素）
     *
     * @param[in]   pKey    キー
     * @param[in]   num     要素数
     *
     * @return  書き込めたかどうか
     *
     * @pre
     *  - pKey != nullptr
     *  - num <= UINT32_MAX
     *
     * @details
     *  本関数を呼び出した後、要素の数だけキーと値を書き込む必要があります。（以下の関数のいずれかを使用）
     *
     *  - WriteKeyValue
     *  - WriteKeyArray
     *  - WriteKeyMap
     */
    bool WriteKeyMap(const char* pKey, size_t num) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（配列要素）
     *
     * @param[in]   pKey    キー
     * @param[in]   num     要素数
     *
     * @return  書き込めたかどうか
     *
     * @pre
     *  - pKey != nullptr
     *  - num <= UINT32_MAX
     *
     * @details
     *  本関数を呼び出した後、要素の数だけ値を書き込む必要があります。（以下の関数のいずれかを使用）
     *
     *  - WriteValue
     *  - WriteArray
     *  - WriteMap
     */
    bool WriteKeyArray(const char* pKey, size_t num) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（論理値）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @pre
     *  - pKey != nullptr
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, bool value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, int8_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, int16_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, int32_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, int64_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, uint8_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, uint16_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, uint32_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（整数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, uint64_t value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（浮動小数点数）
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, double value) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（文字列）
     *
     * @param[in]   pKey    キー
     * @param[in]   pValue  値
     *
     * @pre
     *  - pKey != nullptr
     *  - pValue != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, const char* pValue) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。（バイナリ）
     *
     * @param[in]   pKey    キー
     * @param[in]   pValue  値
     * @param[in]   size    pValue のサイズ
     *
     * @pre
     *  - pKey != nullptr
     *  - pValue != nullptr
     *  - size <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteKeyValue(const char* pKey, const void* pValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   書き込んだサイズを取得します。
     *
     * @return  書き込んだサイズ
     *
     * @details
     */
    size_t GetWrittenSize() const NN_NOEXCEPT;

    /**
     * @brief   妥当な MessagePack が生成されているかどうかを判定します。
     *
     * @return  妥当な MessagePack が生成されているかどうか
     *
     * @details
     */
    bool IsValid() const NN_NOEXCEPT;

private:
    //
    nn::Bit8* m_pBuffer;
    //
    size_t m_Size;
    size_t m_Written;
};

}}
