﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/os.h>
#include <nn/ens/ens_AsyncContext.h>
#include <nn/ens/ens_ResultPublic.h>
#include <nn/ens/ens_Types.h>
#include <nn/ens/ens_TypesForAcbaa.h>

namespace nn { namespace ens {

//! @name メッセージ
//! @{

/**
 * @brief   メッセージを送信します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[in]   targetUserId    送信先のユーザー ID
 * @param[in]   metadata        メタデータ
 * @param[in]   body            ボディ
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - targetUserId != nn::ens::InvalidUserId
 *  - metadata.IsValid() && !metadata.IsNull()
 *  - body.IsValid() && !body.IsNull()
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  指定した相手にメッセージを送信します。
 *
 *  メタデータはボディに対する付加情報であり、 nn::ens::ReceiveMessageHeaderList() で一覧取得できます。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @note
 *  メタデータとボディの上限サイズについては、サーバー側と相談の上決定してください。
 */
void SendMessage(AsyncContext* pOutContext,
    const UserId& targetUserId,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   任天堂宛てにメッセージを送信します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[in]   metadata        メタデータ
 * @param[in]   body            ボディ
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - metadata.IsValid() && !metadata.IsNull()
 *  - body.IsValid() && !body.IsNull()
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  任天堂宛てにメッセージを送信します。
 *
 *  メタデータはボディに対する付加情報です。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @note
 *  メタデータとボディの上限サイズについては、サーバー側と相談の上決定してください。
 */
void SendMessageToNintendo(AsyncContext* pOutContext,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   自分宛てのメッセージのヘッダーリストを取得します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[out]  pOutCount       pOutHeaderList に格納した要素数（非同期処理成功後に取得可能）
 * @param[out]  pOutTotalCount  サーバーで保持しているメッセージの総数（非同期処理成功後に取得可能）
 * @param[out]  pOutHeaderList  ヘッダーリストを格納するバッファ（非同期処理成功後に取得可能）
 * @param[in]   count           pOutHeaderList の要素数
 * @param[in]   offset          取得オフセット
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutCount != nullptr
 *  - pOutTotalCount != nullptr
 *  - pOutHeaderList != nullptr
 *  - count > 0
 *  - offset >= 0
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  自分宛てのメッセージのヘッダーリストを取得します。
 *
 *  ヘッダーリストはメッセージの送信時刻の昇順でソートされます。
 *
 *  pOutHeaderList の各要素のメタデータには、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void ReceiveMessageHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MessageHeader pOutHeaderList[],
    int count,
    int offset,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   自分宛てのメッセージのボディを取得します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[out]  pOutBody    ボディ（非同期処理成功後に取得可能）
 * @param[in]   messageId   メッセージ ID
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutBody != nullptr
 *  - pOutBody->IsValid() && !pOutBody->IsNull()
 *  - messageId != InvalidMessageId
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  自分宛てのメッセージのボディを取得します。
 *
 *  メッセージ ID は nn::ens::ReceiveMessageHeaderList() で取得したものを指定します。@n
 *  他人宛てのメッセージのメッセージ ID を指定することはできません。
 *
 *  メッセージのボディを受信した後もメッセージはサーバーに残り続けます。@n
 *  メッセージを削除するには、 nn::ens::DeleteMessage() を使用してください。
 *
 *  pOutBody には、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void ReceiveMessageBody(AsyncContext* pOutContext,
    ReceiveBuffer* pOutBody,
    const MessageId& messageId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   自分宛てのメッセージを削除します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[in]   pMessageIdList  削除するメッセージの ID のリスト
 * @param[in]   count           pMessageIdList の要素数
 * @param[in]   pReason         削除理由
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスIDトークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pMessageIdList != nullptr
 *  - pMessageIdList の各要素が InvalidMessageId ではない
 *  - count > 0
 *  - pReason != nullptr
 *  - pReason が UTF-8 でエンコードされている
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  自分宛てのメッセージを削除します。
 *
 *  複数のメッセージに対して、メッセージごとに異なる削除理由を指定することはできません。@n
 *  削除理由が異なるメッセージがある場合、本関数を複数回実行してください。
 *
 *  自分宛てでないメッセージや存在しないメッセージの ID を指定しても本関数は成功します。@n
 *  これにより他人宛てのメッセージが削除されることはありません。
 *
 * @note
 *  指定可能な削除理由については、サーバー側と相談の上決定してください。（以下、指定例）
 *  - "RECEIVED" （正常受信）
 *  - "REJECTED" （アプリケーション独自のフィルターによる受信拒否）
 */
void DeleteMessage(AsyncContext* pOutContext,
    const MessageId pMessageIdList[],
    int count,
    const char* pReason,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

//! @}

//! @name 通知
//! @{

/**
 * @brief   通知サービスを有効化します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[in]   user        通知の宛先となるユーザーアカウント
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  通知サービスを有効化します。
 *
 *  通知サービスを有効化することで自分宛ての通知が指定したユーザーアカウントに届きます。@n
 *  同一ユーザーアカウントに対して異なる複数の認証情報を指定した場合、それら宛ての通知はすべて同一ユーザーアカウントに届きます。
 *
 *  通知サービスの有効化は本関数を呼び出したデバイス上でのみ有効です。@n
 *  「ユーザーとセーブデータの引っ越し」等でデバイスが変更されると、通知が届かなくなります。
 *
 *  デバイス変更に対応するためには、アプリケーション内でユーザーを選択する度に本関数を呼び出すことを推奨します。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @note
 *  システムの通知サービスは ENS ライブラリのユーザー (nn::ens::UserId) ではなく、ユーザーアカウント (nn::account::Uid) 単位で提供されます。@n
 *  そのため、引数にユーザーアカウントを指定する必要があります。
 *
 * @see nn::ens::GetNotificationEvent(), nn::ens::PopNotificationData()
 */
void ActivateNotificationService(AsyncContext* pOutContext,
    const nn::account::Uid& user,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   通知イベントを取得します。
 *
 * @return  通知イベント
 *
 * @details
 *  通知を受信するとシグナルするイベントを取得します。
 *
 *  イベントのクリアモードは nn::os::EventClearMode_AutoClear です。@n
 *  イベントはアプリケーションのネットワーク利用宣言に関わらずシグナルされる可能性があります。
 */
nn::os::SystemEvent& GetNotificationEvent() NN_NOEXCEPT;

/**
 * @brief   通知を 1 件取得します。
 *
 * @param[out]  pOut    通知データ
 *
 * @return  通知を取得できたかどうか
 *
 * @pre
 *  - pOut != nullptr
 *
 * @details
 *  通知を 1 件取得します。
 *
 *  通知は複数届いている場合があります。@n
 *  nn::ens::GetNotificationEvent() で受信待ちを行う場合、通知の取りこぼしを防ぐため本関数が false を返すまで繰り返し呼び出してください。
 */
bool PopNotificationData(NotificationData* pOut) NN_NOEXCEPT;

//! @}

//! @name マイデザイン
//! @{

/**
 * @brief   マイデザインの作者プロフィールを登録します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[out]  pOutId      マイデザイン作者 ID （非同期処理成功後に取得可能）
 * @param[in]   name        マイデザイン作者名
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutId != nullptr
 *  - name が UTF-8 でエンコードされている
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  マイデザインの作者プロフィールを登録します。
 *
 *  作者プロフィールの登録は初回のみ有効です。@n
 *  同一ユーザーに対して本関数を何回呼び出しても、登録内容が変更されることはありません。
 *
 *  2 回目以降の呼び出しは失敗せず、初回に作成したマイデザイン作者 ID が返ります。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @attention
 *  現状、登録された自分のマイデザイン作者プロフィールをアプリケーションが知る手段はありません。@n
 *  アプリケーション内で自分のマイデザイン作者プロフィールを表示したい場合、本関数を呼び出す前にセーブデータに記録する必要があります。
 */
void RegisterMyDesignAuthorProfile(AsyncContext* pOutContext,
    MyDesignAuthorId* pOutId,
    const MyDesignAuthorName& name,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   マイデザインを投稿します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[out]  pOutId      マイデザイン ID （非同期処理成功後に取得可能）
 * @param[in]   metadata    メタデータ
 * @param[in]   body        ボディ
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutId != nullptr
 *  - metadata.IsValid() && !metadata.IsNull()
 *  - body.IsValid() && !body.IsNull()
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  マイデザインを投稿します。
 *
 *  マイデザインを投稿する前に、 nn::ens::RegisterMyDesignAuthorProfile() で事前に作者プロフィールを登録する必要があります。
 *
 *  事前に作者プロフィールを登録していない場合、非同期処理の結果は nn::ens::ResultApplicationSpecificError になります。@n
 *  この時のエラーの詳細値は 1001 です。
 *
 *  メタデータはボディに対する付加情報であり、 nn::ens::GetMyDesignHeaderList() で一覧取得できます。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @note
 *  メタデータとボディの上限サイズについては、サーバー側と相談の上決定してください。
 */
void PostMyDesign(AsyncContext* pOutContext,
    MyDesignId* pOutId,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   投稿したマイデザインを削除します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[in]   myDesignId  マイデザイン ID
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - myDesignId != InvalidMyDesignId
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  投稿したマイデザインを削除します。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void DeleteMyDesign(AsyncContext* pOutContext,
    const MyDesignId& myDesignId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   指定した作者が投稿したマイデザインのヘッダーリストを取得します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[out]  pOutCount       pOutHeaderList に格納した要素数（非同期処理成功後に取得可能）
 * @param[out]  pOutTotalCount  サーバーで保持している投稿したマイデザインの総数（非同期処理成功後に取得可能）
 * @param[out]  pOutHeaderList  ヘッダーリストを格納するバッファ（非同期処理成功後に取得可能）
 * @param[in]   count           pOutHeaderList の要素数
 * @param[in]   offset          取得オフセット
 * @param[in]   authorId        作者 ID
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutCount != nullptr
 *  - pOutTotalCount != nullptr
 *  - pOutHeaderList != nullptr
 *  - count > 0
 *  - offset >= 0
 *  - authorId != InvalidMyDesignAuthorId
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  指定した作者が投稿したマイデザインのヘッダーリストを取得します。
 *
 *  ヘッダーリストはマイデザインの更新時刻の降順でソートされます。
 *
 *  pOutHeaderList の各要素のメタデータには、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const MyDesignAuthorId& authorId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   検索条件を指定してマイデザインのヘッダーリストを取得します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[out]  pOutCount       pOutHeaderList に格納した要素数（非同期処理成功後に取得可能）
 * @param[out]  pOutTotalCount  検索条件にマッチしたマイデザインの総数（非同期処理成功後に取得可能）
 * @param[out]  pOutHeaderList  ヘッダーリストを格納するバッファ（非同期処理成功後に取得可能）
 * @param[in]   count           pOutHeaderList の要素数
 * @param[in]   offset          取得オフセット
 * @param[in]   pSearchQuery    検索クエリ
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutCount != nullptr
 *  - pOutTotalCount != nullptr
 *  - pOutHeaderList != nullptr
 *  - count > 0
 *  - offset >= 0
 *  - pSearchQuery != nullptr
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  検索条件を指定してマイデザインのヘッダーリストを取得します。
 *
 *  ヘッダーリストはマイデザインの更新時刻の降順でソートされます。
 *
 *  pOutHeaderList の各要素のメタデータには、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 *
 * @note
 *  検索クエリは nn::ens::SearchQueryBuilder を使用して生成できます。
 */
void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const char* pSearchQuery,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   投稿されたすべてのマイデザインのヘッダーリストを取得します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[out]  pOutCount       pOutHeaderList に格納した要素数（非同期処理成功後に取得可能）
 * @param[out]  pOutTotalCount  サーバーで保持している投稿されたマイデザインの総数（非同期処理成功後に取得可能）
 * @param[out]  pOutHeaderList  ヘッダーリストを格納するバッファ（非同期処理成功後に取得可能）
 * @param[in]   count           pOutHeaderList の要素数
 * @param[in]   offset          取得オフセット
 * @param[in]   credential      認証情報
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutCount != nullptr
 *  - pOutTotalCount != nullptr
 *  - pOutHeaderList != nullptr
 *  - count > 0
 *  - offset >= 0
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  サーバーに投稿されたすべてのマイデザインのヘッダーリストを取得します。
 *
 *  ヘッダーリストはマイデザインの更新時刻の降順でソートされます。
 *
 *  pOutHeaderList の各要素のメタデータには、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

/**
 * @brief   マイデザインのボディを取得します。
 *
 * @param[out]  pOutContext 非同期コンテキスト
 * @param[out]  pOutBody    ボディの格納先（非同期処理成功後に取得可能）
 * @param[in]   myDesignId  マイデザイン ID
 * @param[in]   credential  認証情報
 * @param[in]   pToken      ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutBody != nullptr
 *  - pOutBody->IsValid() && !pOutBody->IsNull()
 *  - myDesignId != InvalidMyDesignId
 *  - credential.IsValid()
 *  - pToken != nullptr
 *
 * @details
 *  マイデザインのボディを取得します。
 *
 *  マイデザイン ID は nn::ens::GetMyDesignHeaderList() で取得したものを指定します。
 *
 *  pOutBody には、受信し得るデータを格納するのに十分なサイズのバッファをあらかじめ指定してください。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void GetMyDesignBody(AsyncContext* pOutContext,
    ReceiveBuffer* pOutBody,
    const MyDesignId& myDesignId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT;

//! @}

}}
