﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   MessagePack 読み込み用の入力ストリーム
 *
 * @details
 */
class MessagePackInputStream
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    MessagePackInputStream() NN_NOEXCEPT = default;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~MessagePackInputStream() NN_NOEXCEPT = default;

    /**
     * @brief   データを読み込みます。
     *
     * @param[in]   pBuffer バッファ
     * @param[in]   size    バッファサイズ
     *
     * @return  成功したかどうか
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     *  本関数は、指定したサイズ分データを読み込むか、入力ストリームがエラーを返すまでブロッキングするように実装してください。
     */
    virtual bool Read(void* pBuffer, size_t size) NN_NOEXCEPT = 0;

    /**
     * @brief   メモリ上に展開されているデータのポインタを取得します。
     *
     * @param[out]  ppOutData   データのポインタ
     * @param[in]   size        読み込みたいサイズ
     *
     * @return  読み込みたいサイズのデータがメモリ上に展開されているかどうか
     *
     * @pre
     *  - ppOutData != nullptr
     *  - size > 0
     *
     * @details
     *  MessagePackReader はサイズの大きいデータ（文字列やバイナリデータ等）を読み込む場合、本関数を使用してメモリ上のデータの直接参照を試みます。
     *
     *  直接参照に成功した場合、ポインタには次に読み込まれるデータの先頭位置が設定されます。@n
     *  また、読み込みたいサイズ分、ストリームの位置が移動します。
     *
     *  本関数を使用することで、データバッファへのメモリコピーが行われなくなり、読み込み処理の高速化が期待できます。
     *
     * @note
     *  すべてのデータがメモリ上に展開されている場合、 MessagePackReader にデータバッファを設定する必要はありません。
     */
    virtual bool GetDataPointerFromMemoryCache(const void** ppOutData, size_t size) NN_NOEXCEPT = 0;
};

}}}}
