﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/core/ens_Task.h>
#include <nn/ens/detail/util/ens_EventHolder.h>

namespace nn { namespace ens { namespace detail { namespace core {

/**
 * @brief   タスクを管理するクラス
 *
 * @details
 */
class TaskManager
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    TaskManager() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    ~TaskManager() NN_NOEXCEPT;

    /**
     * @brief   タスクをリストの先頭に追加します。
     *
     * @param[in]   pTask   タスク
     *
     * @pre
     *  - pTask != nullptr
     *
     * @details
     */
    void PushHead(Task* pTask) NN_NOEXCEPT;

    /**
     * @brief   タスクをリストの末尾に追加します。
     *
     * @param[in]   pTask   タスク
     *
     * @pre
     *  - pTask != nullptr
     *
     * @details
     */
    void PushTail(Task* pTask) NN_NOEXCEPT;

    /**
     * @brief   先頭のタスクを 1 つ取り出します。
     *
     * @return  タスク
     *
     * @details
     *  本関数は、先頭のタスクを 1 つ取りだし、タスクリストから除外します。@n
     *  タスクが 1 つも存在しない場合、 nullptr を返します。
     */
    Task* Pop() NN_NOEXCEPT;

    /**
     * @brief   任意のタスクを 1 つ取り出します。
     *
     * @param[in]   id   タスク ID
     *
     * @return  タスク
     *
     * @details
     *  本関数は、指定したタスク ID のタスクを検索して取りだし、タスクリストから除外します。@n
     *  指定したタスクが存在しない場合、 nullptr を返します。
     */
    Task* Fetch(const TaskId& id) NN_NOEXCEPT;

    /**
     * @brief   タスクの追加がシグナルされるイベントを登録します。
     *
     * @param[in]   pEvent  イベント
     *
     * @pre
     *  - pEvent != nullptr
     *
     * @details
     */
    void RegisterEvent(detail::util::EventHolder* pEvent) NN_NOEXCEPT;

    /**
     * @brief   タスクの追加がシグナルされるイベントの登録を解除します。
     *
     * @param[in]   pEvent  イベント
     *
     * @pre
     *  - pEvent != nullptr
     *
     * @details
     */
    void UnregisterEvent(detail::util::EventHolder* pEvent) NN_NOEXCEPT;

private:
    //
    mutable nn::os::SdkMutex m_Mutex;
    //
    nn::util::IntrusiveList<Task,
        nn::util::IntrusiveListBaseNodeTraits<Task>> m_TaskList;
    //
    nn::util::IntrusiveList<detail::util::EventHolder,
        nn::util::IntrusiveListBaseNodeTraits<detail::util::EventHolder>> m_EventList;
};

}}}}
