﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/ens_AsyncContext.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace ens { namespace detail { namespace core {

/**
 * @brief   タスクの基底クラス
 *
 * @details
 *  本ライブラリが提供するタスクの基底クラスです。@n
 *  本クラスを継承したクラスは Main() を実装する必要があります。
 */
class Task : public nn::util::IntrusiveListBaseNode<Task>
{
public:
    NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO_ROOT(Task);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    Task() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~Task() NN_NOEXCEPT;

    /**
     * @brief   タスク ID を取得します。
     *
     * @return  タスク ID
     *
     * @details
     */
    TaskId GetId() const NN_NOEXCEPT;

    /**
     * @brief   ワークバッファを設定します。
     *
     * @param[in]   pBuffer     ワークバッファ
     * @param[in]   bufferSize  ワークバッファのサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - bufferSize > 0
     *
     * @details
     */
    void SetWorkBuffer(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief   非同期コンテキストを登録します。
     *
     * @param[in]   pContext    非同期コンテキスト
     *
     * @pre
     *  - pContext != nullptr
     *
     * @details
     */
    void RegisterAsyncContext(AsyncContext* pContext) NN_NOEXCEPT;

    /**
     * @brief   非同期コンテキストを登録解除します。
     *
     * @details
     */
    virtual void UnregisterAsyncContext() NN_NOEXCEPT;

    /**
     * @brief   タスクを実行します。
     *
     * @details
     *  本関数はタスクの実行が完了するまでブロックします。
     */
    void Run() NN_NOEXCEPT;

    /**
     * @brief   タスクをキャンセルします。
     *
     * @details
     *  別スレッド上で動いているタスクにキャンセル要求を通知します。
     */
    virtual void Cancel() NN_NOEXCEPT;

protected:
    /**
     * @brief   タスクのエントリーポイントです。
     *
     * @return  処理結果
     *
     * @details
     *  本関数は必ず実装する必要があります。
     */
    virtual nn::Result Main() NN_NOEXCEPT = 0;

    /**
     * @brief   パラメータアクセス用のロックを取得します。
     *
     * @return  パラメータアクセス用のロック
     *
     * @details
     *  ライブラリ利用者から渡されたパラメータにアクセスする時、本関数で取得したロックを使用してメモリを保護する必要があります。
     */
    nn::os::SdkMutex& GetParameterLock() NN_NOEXCEPT;

    /**
     * @brief   ワークバッファを取得します。
     *
     * @return  ワークバッファ
     *
     * @details
     */
    void* GetWorkBuffer() NN_NOEXCEPT;

    /**
     * @brief   ワークバッファサイズを取得します。
     *
     * @return  ワークバッファサイズ
     *
     * @details
     */
    size_t GetWorkBufferSize() const NN_NOEXCEPT;

    /**
     * @brief   キャンセル時にシグナルされるイベントを取得します。
     *
     * @return  キャンセル通知イベント
     *
     * @details
     */
    nn::os::Event& GetCancelEvent() NN_NOEXCEPT;

    /**
     * @brief   非同期コンテキストが登録されているかどうかを取得します。
     *
     * @return  非同期コンテキストが登録されているかどうか
     *
     * @pre
     *  - パラメータアクセス用のロックを取っている
     *
     * @details
     *  非同期コンテキストが登録されていない場合、各タスクに渡されたパラメータにアクセスしてはいけません。@n
     *  本関数は、 GetParameterLock() をロックしている間に使用してください。
     */
    bool IsAsyncContextRegistered() const NN_NOEXCEPT;

    /**
     * @brief   エラーの詳細を設定します。
     *
     * @param[in]   errorDetail エラーの詳細
     *
     * @details
     */
    void SetErrorDetail(int32_t errorDetail) NN_NOEXCEPT;

private:
    //
    nn::os::SdkMutex m_Mutex;
    //
    nn::os::Event m_CancelEvent;
    //
    TaskId m_Id;
    //
    void* m_pWorkBuffer;
    size_t m_WorkBufferSize;
    //
    AsyncContext* m_pContext;
    int32_t m_ErrorDetail;
};

}}}}
