﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>

namespace nn { namespace ens { namespace detail { namespace core {

/**
 * @brief   認証トークンのキャッシュを管理するクラス
 *
 * @details
 */
class AuthTokenCacheManagerImpl
{
public:
    // internal use only
    struct Item
    {
        UserId userId;
        AuthToken token;
    };

public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pList   アイテムリスト
     * @param[in]   num     アイテムリストの要素数
     *
     * @pre
     *  - pList != nullptr
     *  - num > 0
     *
     * @details
     */
    AuthTokenCacheManagerImpl(Item* pList, size_t num) NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    ~AuthTokenCacheManagerImpl() NN_NOEXCEPT;

    /**
     * @brief   クリアします。
     *
     * @details
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   認証トークンを取得します。
     *
     * @param[out]  pOutToken   認証トークン
     * @param[in]   userId      ユーザー ID
     *
     * @return  指定したユーザーの認証トークンが存在するかどうか
     *
     * @pre
     *  - pOutToken != nullptr
     *  - userId != InvalidUserId
     *
     * @details
     */
    bool Get(AuthToken* pOutToken, const UserId& userId) const NN_NOEXCEPT;

    /**
     * @brief   認証トークンを設定します。
     *
     * @param[in]   userId  ユーザー ID
     * @param[in]   token   認証トークン
     *
     * @pre
     *  - userId != InvalidUserId
     *
     * @details
     *  指定したユーザー ID の認証トークンがすでに存在する場合、指定した認証トークンで置き換えます。@n
     *  件数が上限に達していた場合、設定タイミングが一番古い認証トークンが破棄されます。
     */
    void Set(const UserId& userId, const AuthToken& token) NN_NOEXCEPT;

private:
    //
    Item* m_pList;
    const size_t m_Num;
    //
    int m_Count;

private:
    //
    int Find(const UserId& userId) const NN_NOEXCEPT;
};

/**
 * @brief   認証トークンのキャッシュを管理するクラス
 *
 * @tparam      N   要素数
 *
 * @details
 */
template <size_t N>
class AuthTokenCacheManager : public AuthTokenCacheManagerImpl
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    AuthTokenCacheManager() NN_NOEXCEPT
        : AuthTokenCacheManagerImpl(m_List, N)
    {
    }

private:
    //
    Item m_List[N];
};

}}}}
