﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/util/util_Optional.h>

#include "capsrv_MovieConfig.h"
#include "capsrv_MovieFilePath.h"
#include "capsrv_CachedMovieStream.h"
#include "capsrv_AlbumMovieWriteStreamDataSectionAccessor.h"
#include "capsrv_MovieWriterCacheStrategy.h"
#include "../capsrv_AlbumFileId.h"

namespace nn{ namespace capsrv{ namespace movie{

    namespace detail{

        class MovieWriterFileImpl
        {
            NN_DISALLOW_COPY(MovieWriterFileImpl);
            NN_DISALLOW_MOVE(MovieWriterFileImpl);
        public:
            typedef MovieFilePath FilePath;

        public:
            static size_t GetRequiredMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
            static size_t GetRequiredMemoryAlignment() NN_NOEXCEPT;

            MovieWriterFileImpl() NN_NOEXCEPT;

            bool IsInitialized() const NN_NOEXCEPT;

            void Initialize(
                const FilePath& filename,
                AlbumMovieWriteStreamHandle handle,
                int64_t cacheChunkSize,
                int64_t cacheChunkCount,
                void* memory,
                size_t memorySize,
                util::optional<fs::PriorityRaw> fsPriorityRaw
            ) NN_NOEXCEPT;
            void Finalize() NN_NOEXCEPT;

            bool IsMatchFilename(const FilePath& filename) const NN_NOEXCEPT;
            AlbumMovieWriteStreamHandle GetHandle() const NN_NOEXCEPT;

        private:
            nn::Result DoReadImpl(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;
            nn::Result DoWriteImpl(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT;
            nn::Result DoFlushImpl() NN_NOEXCEPT;
            nn::Result DoSetSizeImpl(int64_t size) NN_NOEXCEPT;
            nn::Result DoGetSizeImpl(int64_t* outValue) NN_NOEXCEPT;
            nn::Result DoOperateRangeImpl(
                void* outBuffer,
                size_t outBufferSize,
                fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT;

        private:
            friend class MovieWriterFileProxy;
            CachedMovieStream                        m_Stream;
            AlbumMovieWriteStreamDataSectionAccessor m_Accessor;
            MovieWriterCacheStrategy                 m_Strategy;

            FilePath m_Filename;
            int64_t m_FileSize;
            util::optional<fs::PriorityRaw> m_pFsPriorityRaw;

            class ScopedFsPriorityRawChanger;
        };

        class MovieWriterFileProxy
            : public nn::fs::fsa::IFile
        {
        public:
            explicit MovieWriterFileProxy(MovieWriterFileImpl* pImpl) NN_NOEXCEPT;
            virtual ~MovieWriterFileProxy() NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE;

        private:
            MovieWriterFileImpl* m_pImpl;
        };

        class MovieWriterFileSystemImpl
        {
            NN_DISALLOW_COPY(MovieWriterFileSystemImpl);
            NN_DISALLOW_MOVE(MovieWriterFileSystemImpl);
        public:
            static const int FileCountMax = MovieWriterFileSystemFileCountMax;
            typedef MovieFilePath FilePath;

        public:
            // @pre fileCount > 0
            // @pre fileCount >= FileCountMax
            static size_t GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
            static size_t GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT;

            MovieWriterFileSystemImpl() NN_NOEXCEPT;

            void Initialize() NN_NOEXCEPT;
            void Finalize() NN_NOEXCEPT;

            void AttachWriteStream(
                int index,
                const FilePath& filename,
                AlbumMovieWriteStreamHandle handle,
                int64_t cacheChunkSize,
                int64_t cacheChunkCount,
                void* memory,
                size_t memorySize,
                util::optional<fs::PriorityRaw> pFsPriorityRaw
            ) NN_NOEXCEPT;
            void DetachWriteStream(int index) NN_NOEXCEPT;

        private:
            MovieWriterFileImpl* FindFile(const char* filename) NN_NOEXCEPT;

            nn::Result DoGetEntryTypeImpl(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT;
            nn::Result DoOpenFileImpl(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT;

        private:
            friend class MovieWriterFileSystemProxy;
            MovieWriterFileImpl m_File[FileCountMax];
        };

        class MovieWriterFileSystemProxy
            : public nn::fs::fsa::IFileSystem
        {
        public:
            explicit MovieWriterFileSystemProxy(MovieWriterFileSystemImpl* pImpl) NN_NOEXCEPT;
            virtual ~MovieWriterFileSystemProxy() NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimestamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        private:
            MovieWriterFileSystemImpl* m_pImpl;
        };

    }

    class MovieWriterFileSystem
    {
        NN_DISALLOW_COPY(MovieWriterFileSystem);
        NN_DISALLOW_MOVE(MovieWriterFileSystem);
    public:
        static const int FileCountMax = MovieWriterFileSystemFileCountMax;
        typedef MovieFilePath FilePath;

    private:
        enum FileState
        {
            FileState_Empty = 0,
            FileState_Opened,
        };

    public:
        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkSize % MovieDataChunkUnitSize == 0
        //! @pre cacheChunkCount >= 1
        static size_t GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
        static size_t GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT;

        MovieWriterFileSystem() NN_NOEXCEPT;

        //! @brief ファイルシステムを初期化して指定したマウント名で nn::fs に登録する
        //! @param mountName fs に登録するマウント名
        void Initialize(const char* mountName) NN_NOEXCEPT;

        //! @brief nn::fs から登録解除してファイルシステムの終了処理を行う
        //! @pre Attach 中の AlbumMovieWriteStream が存在しない
        void Finalize() NN_NOEXCEPT;

        //! @brief AlbumMovieWriteStream をファイルシステムにアタッチしてファイルハンドルを取得する。
        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkSize % MovieDataChunkUnitSize == 0
        //! @pre cacheChunkCount >= 1
        //! @pre memory != nullptr
        //! @pre memory % GetRequiredMemoryAlignmentForFile(cacheChunkSize, cacheChunkCount) == 0
        //! @pre memorySize >= GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount)
        //! @retval nn::fs::ResultFileEntryFull
        //! @details
        //!   AlbumMovieWriteStream のデータ部分を読み書きするためのファイルハンドルを返します。
        //!   アタッチ中の Stream の状態に制限はありませんが、
        //!   nn::fs::WriteFile や nn::fs::SetSize を行うためには Stream が WritingData 状態である必要があります。
        //!
        //!   memory で渡されたメモリ領域は DetachMovieWriteStream() を呼び出すまで使用されます。
        nn::Result AttachMovieWriteStreamDataSection(
            nn::fs::FileHandle* pOutFileHandle,
            AlbumMovieWriteStreamHandle streamHandle,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize,
            util::optional<fs::PriorityRaw> pFsPriorityRaw = util::nullopt
        ) NN_NOEXCEPT;

        //! @brief AlbumMovieWriteStream をファイルシステムからデタッチします。
        //! @details
        //!   この関数の呼出後、アタッチ時に渡したメモリを再利用できます。
        void DetachMovieWriteStreamDataSection(nn::fs::FileHandle fileHandle) NN_NOEXCEPT;

    private:
        FilePath GetFilename(int index) const NN_NOEXCEPT;
        FilePath GetFilePath(int index) const NN_NOEXCEPT;

        int FindEmptyFileSlot() const NN_NOEXCEPT;
        int FindFileSlot(nn::fs::FileHandle h) const NN_NOEXCEPT;

    private:
        detail::MovieWriterFileSystemImpl m_Impl;
        FilePath m_MountName;

        FileState m_FileStateList[FileCountMax];
        nn::fs::FileHandle m_FileHandleList[FileCountMax];

    };

}}}
